package org.lsst.ccs.drivers.opc;

import java.util.Optional;
import java.util.stream.Stream;

/**
 * Gives the non-array data types returnable by an OPC service.
 * See Appendix C in the manual that comes with the Matrikon OPC Server for Simulation.
 * In the standard Windows installation that's in {@code C:\Program Files (x86)\Matrikon\OPC\Simulation\}.
 * @author tether
 */
public enum ScalarType {

    /** Nada. */
    VT_EMPTY(0),

    /** 2-byte signed integer. */
    VT_I2(2),

    /** 4-byte signed integer. */
    VT_I4(3),

    /** IEEE single. */
    VT_R4(4),

    /** IEEE double. */
    VT_R8(5),

    /**  Currency. Eight-byte singed integer scaled to allow 4 decimal places, for example $12.34
     *   is represented as 123400.  */
    VT_CY(6),

    /** A A double counting the number of days, including a fraction, since 0 hours on Dec 30 1899. */
    VT_DATE(7),

    /** Unicode string, encoded on the server as UTF-16LE. */
    VT_BSTR(8),

    /** 32-bit error code. */
    VT_ERROR(10),

    /** Boolean. 0 = false, -1 = true */
    VT_BOOL(11),

    /** Signed byte. Treated as a Java char value by the j-interop library. */
    VT_I1(16),

    /** Unsigned byte. */
    VT_UI1(17),

    /** Unsigned short. */
    VT_UI2(18),

    /** Unsigned int. */
    VT_UI4(19),

    /** Not a recognized type code. */
    ILLEGAL_TYPE(-1);

    private final int code;

    private ScalarType(final int code) {this.code = code;}

    /**
     * Gets the numerical code designating the OPC type.
     * @return The code.
     */
    public int getCode() {return code;}

    /**
     * Converts a an OPC type code to an ScalarType.
     * @param code The OPC type code (five bits).
     * @return The type enumerator.
     */
    static public ScalarType fromCode(final int code) {
        final Optional<ScalarType> type = Stream.of(ScalarType.values()).filter(x->x.getCode()==code).findAny();
        if (!type.isPresent()) {return ILLEGAL_TYPE;}
        else {return type.get();}
    }
}
