package org.lsst.ccs.drivers.opc;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.net.UnknownHostException;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.jinterop.dcom.common.JIErrorCodes;
import org.jinterop.dcom.common.JIException;
import org.jinterop.dcom.core.JICurrency;
import org.jinterop.dcom.core.JIVariant;
import org.openscada.opc.lib.common.AlreadyConnectedException;
import org.openscada.opc.lib.common.ConnectionInformation;
import org.openscada.opc.lib.common.ConnectionProtocol;
import org.openscada.opc.lib.common.NotConnectedException;
import org.openscada.opc.lib.da.AddFailedException;
import org.openscada.opc.lib.da.DuplicateGroupException;
import org.openscada.opc.lib.da.Group;
import org.openscada.opc.lib.da.Item;
import org.openscada.opc.lib.da.ItemState;
import org.openscada.opc.lib.da.Server;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.exc.BundledException;


/**
 * Manages a connection to an OPC DA service and fetches values from a connected server.
 * Immutable.
 * <p>
 * The OPC DA service is assumed to be running under Windows 7 or 10, which must have
 * its DCOM permissions and firewall configured to allow remote access to the service.
 * The j-interop library used to communicate with the service doesn't support Kerberos,
 * and so can't pass along existing Kerberos credentials. This
 * requires you to provide a username and password valid on the machine running the service.
 * SLAC policy requires that you create a local account on that machine solely
 * for this purpose so that the password is not valid except on that machine.
 * @author tether
 */
public class OPCClient {

    private final Server service;
    private final Group group;
    private final Item[] items;
    private final Throwable exception;
    private final List<OPCItem> data;

    /**
     * Establishes a connection to the OPC DA service.
     * @param serverName The hostname or IPv4 address of the server machine.
     * @param domain The Windows domain to which the user name belongs. Must be local to the
     * machine running the server.
     * @param classID The class ID (CLSID) of the OPC DA service.
     * @param username The remote username used to log in to the server.
     * @param password The password used to log in to the server.
     * @exception DriverException if the connection can't be made.
     */
    public OPCClient
        (final String serverName,
         final String domain,
         final String classID,
         final String username,
         final String password
        )
                throws DriverException
    {
        // Create a server proxy object with no item group yet created..
        final ConnectionInformation ci = new ConnectionInformation();
        ci.setHost(serverName);
        ci.setDomain(domain);
        ci.setUser(username);
        ci.setPassword(password);
        ci.setClsid(classID);
        ci.setProtocol(ConnectionProtocol.NTLMV2_SESSION_SECURITY); // Required by SLAC policy.
        try {
            this.service = new Server(ci, null);
            service.connect();
        }
        catch (UnknownHostException | JIException | AlreadyConnectedException exc) {
            throw new DriverException(exc);
        }
        this.group = null;
        this.items = null;
        this.exception = null;
        this.data = Collections.emptyList();
    }

    // Constructs a new instance from values already on hand.
    private OPCClient(
            final Server newService,
            final Group newGroup,
            final Item[] newItems,
            final Throwable newExc,
            List<OPCItem> newData)
    {
        this.service = newService;
        this.group = newGroup;
        this.items = newItems;
        this.exception = newExc;
        this.data = newData;
    }

    /**
     * Creates the group of data items to be read. Each addition is checked for validity with the server.
     * @param itemNames A list of the item names.
     * @return A new object whose item group contains the items specified.
     * @exception DriverException if the operation failed.
     */
    public OPCClient setGroup(final List<String> itemNames) throws DriverException {
        try {
            final Group newGroup = this.service.addGroup();
            final String[] ins = new String[itemNames.size()];
            final Map<String, Item> itemmap = newGroup.addItems( itemNames.toArray(ins) );
            final Item[] newItems = new Item[ins.length];
            itemmap.values().toArray(newItems);
            newGroup.setActive(true);
            return new OPCClient(service, newGroup, newItems, null, Collections.emptyList());
        }
        catch (JIException
               | AddFailedException
               | UnknownHostException
               | NotConnectedException
               | DuplicateGroupException exc
                )
        {
            throw new DriverException(exc);
        }
   }

    /**
     * Reads all the values in the group.
     * @param forceUpdate If true causes the service to re-read each value from its device rather
     * than returning a cached value.
     * @return A new object that contains the data read (if any) and any exception that
     * halted reading (if any).
     */
    public OPCClient readGroup(final boolean forceUpdate) {
        // Execute the read to get the jinterop Items, or an exception.
        BundledException excList = null;
        Map<Item, ItemState> jiItemMap = Collections.emptyMap();
        try {
            jiItemMap = group.read(forceUpdate, items);
        }
        catch (JIException jiexc) {
            excList = new BundledException(jiexc, excList);
        }
        // Produce an OPCItem or an exception for each jinterop Item.
        final List<ItemResult> results = jiItemMap
           .entrySet()
           .stream()
           .map(OPCClient::makeOPCItem)
           .collect(Collectors.toList());
        // Separate the exceptions from the OPCItems.
        final List<OPCItem> newData = results
            .stream()
            .filter(x -> !x.exc.isPresent())
            .map(x -> x.item.get())
            .collect(Collectors.toList());
        for (final ItemResult x: results) {
            if (x.exc.isPresent()) {excList = new BundledException(x.exc.get(), excList);}
        }
        return new OPCClient(service, group, items,
                             (excList == null) ? null : new DriverException(excList), newData);
    }

    private static class ItemResult {
        public final Optional<JIException> exc;
        public final Optional<OPCItem> item;
        ItemResult(final JIException exc, final OPCItem item) {
            this.exc = Optional.ofNullable(exc);
            this.item = Optional.ofNullable(item);
        }
    }

    private static ItemResult makeOPCItem(final Entry<Item, ItemState> entry) {
        try {
            final Item it = entry.getKey();
            final ItemState state = entry.getValue();
            final ScalarType type = ScalarType.fromCode(state.getValue().getType() & 0x1f);
            final boolean arrayFlag = 0 != (state.getValue().getType() & 0x2000);
            final int qualCode = state.getQuality() & 0xff;
            String modifier = MODIFIER_STRINGS.get(qualCode);
            if (modifier == null) {
                modifier = String.format("** Unknown quality code %d (0x%02x).", qualCode, qualCode);
            }
            Instant timestamp = state.getTimestamp().toInstant();
            return new ItemResult(null, new OPCItem(
                                  it.getId(),
                                  convertToStandard(type, state.getValue(), it.getId()),
                                  type,
                                  arrayFlag,
                                  Quality.fromQualityCode(qualCode),
                                  modifier,
                                  timestamp)
            );
        }
        catch (JIException exc) {
            return new ItemResult(exc, null);
        }
    }

    /**
     * Converts the value of an item to an instance of, or an array of instances of, one
     * of the standard Java types Double, Long, BigDecimal, String, Boolean or Instant.
     * If all else fails a conversion to String will be done by calling toString().
     */
    private static Object convertToStandard(ScalarType type, JIVariant value, String name) throws JIException
    {
        // Underlying classes used by the jinterop library:
        // Arrays: JIArray
        // VT_R4: java.lang.Float
        // VT_R8: java.lang.Double
        // VT_CY: JICurrency
        // VT_DATE: java.util.Date
        // VT_I1: java.lang.Character
        // VT_I2: java.lang.Short
        // VT_I4: java.lang.Integer
        // VT_UI1/2/4: JIUnsigned(Byte/Short/Integer) which all implement IJIUnsigned.
        // VT_BSTR: JIString
        // VT_ERROR: JIVariant.SCODE (a class private to JIVariant).
        // VT_BOOL: java.lang.Boolean
        Object result;
        if (value.isArray()) {
            // Alas, JIArray objects don't contain arrays of JIVariant, so to avoid
            // trying to test for all the possibilities we just try to make
            // a JIVariant from each value, then invoke the code that converts JIVariants.
            if (value.getObjectAsArray().getDimensions() > 1) {
                throw new JIException(JIErrorCodes.E_FAIL, "No driver support for multi-dimensional arrays");
            }
            final Object[] valarray = (Object[])value.getObjectAsArray().getArrayInstance();
            final Object[] newarray = new Object[valarray.length];
            for (int i = 0; i < valarray.length; ++i) {
                    final JIVariant element = JIVariant.makeVariant(valarray[i]);
                    newarray[i] = convertToStandard(type, element, name);
            }
            result = newarray;
        }
        else {
            // We have an object of the underlying base type, not an array.
            switch (type) {
                case VT_R8:
                    result = value.getObjectAsDouble();
                    break;
                case VT_R4:
                    result = Double.valueOf(value.getObjectAsFloat());
                    break;
                case VT_CY:
                    {   final JICurrency cy = (JICurrency)value.getObject();
                        result = BigDecimal.valueOf(cy.getUnits()*10000L + cy.getFractionalUnits(), 4);
                    }
                    break;
                case VT_DATE:
                    result = value.getObjectAsDate().toInstant();
                    break;
                case VT_I1:
                    result = Long.valueOf(value.getObjectAsChar());
                    break;
                case VT_I2:
                    result = Long.valueOf(value.getObjectAsShort());
                    break;
                case VT_I4:
                    result = Long.valueOf(value.getObjectAsInt());
                    break;
                case VT_ERROR:
                    throw new JIException(JIErrorCodes.E_FAIL, 
                    String.format("Server returned error code %d for %s", value.getObjectAsSCODE(), name));
                case VT_UI1:
                case VT_UI2:
                case VT_UI4:
                    result = value.getObjectAsUnsigned().getValue().longValue();
                    break;
                case VT_BSTR:
                    result = value.getObjectAsString2();
                    break;
                case VT_BOOL:
                    result = value.getObjectAsBoolean();
                    break;
                default:
                    result = value.toString();
            }
        }
        return result;
    }

    /**
     * Gets the data read by the last call to readGroup().
     * @return A possibly empty list of {@code OPCItem}.
     */
    public List<OPCItem> getData() {return data;}

    /**
     * Gets the exception, if any, that stopped the last group read.
     * @return An Optional which if nonempty holds the exception.
     */
    public Optional<Throwable> getException() {return Optional.ofNullable(exception);}

    /**
     * Tells whether the connection to the service is still open. Some reading errors may result
     * in a closed connection.
     * @return
     */
    public boolean isOpen() {return service != null;}

    /**
     * Closes the connection to the service.
     * @return null.
     */
    public OPCClient close() {
        service.disconnect();
        return null;
    }

    /** These are essentially status codes derived from the lower eight bits of the
     *  raw 16-bit quality code. The upper eight bits are not standardized.
     *  <p>
     *  See https://www.opcsupport.com/link/portal/4164/4590/Article/5/What-are-the-OPC-Quality-Codes
     */
    private final static Map<Integer, String> MODIFIER_STRINGS;

    static {
        final Map<Integer, String> result = new HashMap<Integer, String>();
        result.put(0, "Bad [Non-Specific]");
        result.put(4, "Bad [Configuration Error]");
        result.put(8, "Bad [Not Connected]");
        result.put(12, "Bad [Device Failure]");
        result.put(16, "Bad [Sensor Failure]");
        result.put(20, "Bad [Last Known Value]");
        result.put(24, "Bad [Communication Failure]");
        result.put(28, "Bad [Out of Service]");
        result.put(64, "Uncertain [Non-Specific]");
        result.put(65, "Uncertain [Non-Specific] (Low Limited)");
        result.put(66, "Uncertain [Non-Specific] (High Limited)");
        result.put(67, "Uncertain [Non-Specific] (Constant)");
        result.put(68, "Uncertain [Last Usable]");
        result.put(69, "Uncertain [Last Usable] (Low Limited)");
        result.put(70, "Uncertain [Last Usable] (High Limited)");
        result.put(71, "Uncertain [Last Usable] (Constant)");
        result.put(80, "Uncertain [Sensor Not Accurate]");
        result.put(81, "Uncertain [Sensor Not Accurate] (Low Limited)");
        result.put(82, "Uncertain [Sensor Not Accurate] (High Limited)");
        result.put(83, "Uncertain [Sensor Not Accurate] (Constant)");
        result.put(84, "Uncertain [EU Exceeded]");
        result.put(85, "Uncertain [EU Exceeded] (Low Limited)");
        result.put(86, "Uncertain [EU Exceeded] (High Limited)");
        result.put(87, "Uncertain [EU Exceeded] (Constant)");
        result.put(88, "Uncertain [Sub-Normal]");
        result.put(89, "Uncertain [Sub-Normal] (Low Limited)");
        result.put(90, "Uncertain [Sub-Normal] (High Limited)");
        result.put(91, "Uncertain [Sub-Normal] (Constant)");
        result.put(192, "Good [Non-Specific]");
        result.put(193, "Good [Non-Specific] (Low Limited)");
        result.put(194, "Good [Non-Specific] (High Limited)");
        result.put(195, "Good [Non-Specific] (Constant)");
        result.put(216, "Good [Local Override]");
        result.put(217, "Good [Local Override] (Low Limited)");
        result.put(218, "Good [Local Override] (High Limited)");
        result.put(219, "Good [Local Override] (Constant)");
        MODIFIER_STRINGS = Collections.unmodifiableMap(result);
    }
}

