package org.lsst.ccs.drivers.opc;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import jline.console.ConsoleReader;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 **************************************************************************
 **
 ** Defines an interactive test harness for the OPCClient class.
 ** @author tether
 **
 ***************************************************************************
 */


public class OPCDriver {

    // For password eading.
    private final ConsoleReader console;

    // Contains a usable client or nothing.
    private Optional<OPCClient> client;

    public OPCDriver() throws IOException {
        this.console = new ConsoleReader();
        client = Optional.empty();
    }

    /**
     * Creates an OPCClient object for a Windows OPC DA server, closing any existing connection.
     * @param host The name, or dotted IPv4 address, of the machine running the OPC DA server.
     * @param domain The Windows domain for the user account used to access the server. Dur to
     * SLAC policy this has to be be the local domain for the machine in question, not "SLAC".
     * @param classID The full class ID (CLSID) of the server.
     * @param username The name of the user account to be used to access the server.
     * @return The command response, for example "Done."
     * @throws Exception
     */
    @Command(description="Creates an OPCClient object for a Windows OPC DA server, closing any existing connection.")
    public String open(
        @Argument(name="host", description="A host name or a dotted IPv4 address.")
        final String host,
        @Argument(name="domain", description="The local Windows domain of the host.")
        final String domain,
        @Argument(name="classID", description="The CLSID string of the OPC service.")
        final String classID,
        @Argument(name="username", description="A user name known on the host. You'll be prompted for the password which won't echo.")
        final String username)
    throws Exception
    {
        String response = "Done.";
        client.ifPresent(OPCClient::close);
        String password;
        try {
            password = console.readLine("Password? ", new Character((char)0) /* No echo. */);
            client = Optional.of(new OPCClient(host, domain, classID, username, password));
        }
        catch (Exception exc) {
            client = Optional.empty();
            throw exc;
        }
        return response;
    }

    /**
     * Closes the client connection to the server, if a client has been created.
     * @return "Done." or a message stating that you need to create a client.
     */
    @Command(description="Call client.close() if a client has been created using open().")
    public String close() {
        client.ifPresent(OPCClient::close);
        client = Optional.empty();
        return "Done.";
    }

    /**
     * Sets the group of items to read from the server, if the client exists. Note that the addition
     * of each item results in a query to the server to check whether the item exists.
     * @param itemNames The names, or tags,  of the items.
     * @return "Done." or a message about having to create a client.
     * @throws DriverException
     */
    @Command(description="Calls client.setGroup() if the client exists.")
    public String setGroup(
        @Argument(name="itemName", description="A set of item names separated by spaces.")
        String... itemNames)
    throws DriverException
    {   String response = "Done.";
        if (client.isPresent()) {
            final List<String> names = Arrays.asList(itemNames);
            client = Optional.of(client.get().setGroup(names));
        }
        else {response = "Use open() to create a client.";}
        return response;
    }

    /**
     * Reads the group of items defined via {@code setGroup}.
     * @return "Done." or a message about having to create a client.
     * @throws Throwable
     */
    @Command(description="Calls client.readGroup(true) if a client exists.")
    public String readGroup() throws Throwable
    {   String response = "Done.";
        if (client.isPresent()) {
            client = Optional.of(client.get().readGroup(true));
            final Optional<Throwable> exc = client.get().getException();
            if (exc.isPresent()) {
                throw exc.get();
            }
        }
        else {response = "Use open() to create a client.";}
        return response;
    }

    /**
     * Prints the last set of items read from the server.
     * @return "Done." or a message about missing clients or {@code setGroup/readGroup} commands.
     */
    @Command(description="Calls client.getData() if a client exists.")
    public String getData() {
        String response = "Use open() to create a client.";
        if (client.isPresent()) {
            final List<OPCItem> items = client.get().getData();
            if (items.isEmpty()) {
                response = "No data. Have you called setGroup() and readGroup()?";
            }
            else {
                final StringBuilder buf = new StringBuilder();
                items.stream().forEachOrdered(
                    itm -> {
                        buf.append("\ntag:       "); buf.append(itm.getTag());
                        buf.append("\ntime:      "); buf.append(itm.getTimestamp());
                        buf.append("\nbase type: "); buf.append(itm.getType());
                        buf.append("\narray?:    "); buf.append(itm.isArray());
                        buf.append("\nvalue:     "); buf.append(itm.getValue().toString());
                        buf.append("\nquality:   "); buf.append(itm.getQuality());
                        buf.append("\nmodifier:  "); buf.append(itm.getModifier());
                        buf.append("\n");
                    }
                );
                response = buf.toString();
            }
        }
        return response;
    }

    /**
     * Checks if the current client, if any, still has a connection to the server.
     * @return "true", "false" or a messages about having to create a client.
     */
    @Command(description="Calls client.isOpen() if the client exists.")
    public String isOpen() {
        String response = "Use open() to create a client.";
        if (client.isPresent()) {
            response = "" + client.get().isOpen();
        }
        return response;
    }
}
