package org.lsst.ccs.drivers.newportlamp;

import java.text.DecimalFormat;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * General access routines for the NewportLamp device
 *
 * @author Homer Neal and Heather Kelly
 * BNL uses 69911 and Harvard has 69931
 *
 * 69931: 9600 baud, 1 stop bit, no parity
 */
public class NewportLamp extends Ascii {

    private final int numberOfReadAttempts = 20;
    
    /**
     * Public constants
     */
    public enum onOff {

        OFF, ON;
    }

    /**
     * As specified in the manual pg 13, Current limits *
     */
    private final double minCurrent = 3.00;
    private final double maxCurrent = 24.00;

    /**
     * As specified in the manual pge 13, Power limits
     */
    private final int minPower = 40;
    private final int maxPower = 300;

    /**
     * Constructor.
     */
    public NewportLamp() {
        setOptions(Option.NO_NET);
    }

    /**
     * Opens a connection.
     *
     * @param connType  identifies type of connection FTDI, SERIAL
     * @param devcName  The device name of the serial port
     * @param baudRate  The baud rate
     * @param dataChar  The data characteristics (not used)
     * @throws DriverException
     */
    @Override
    public void open(ConnType connType, String devcName, int baudRate, int dataChar) throws DriverException {
        super.open(connType, devcName, baudRate, 0);
        init();
    }

    /**
     * Initializes device.
     */
    public void init() {
        setCommandTerm(Terminator.CR);
    }

    /**
     * required by teststand interface.
     */
    public boolean getState() throws DriverException {
        return statusLampOn();
    }

    /**
     * Turns lamp on or off.
     */
    public void setState(boolean on) throws DriverException {
        readNewportLamp(on ? "START" : "STOP");
    }

    /**
     * reset to factory defaults.
     */
    public void reset() throws DriverException {
        readNewportLamp("RST");
    }

    /**
     * Return the full Status byte.
     */
    public String getStatus() throws DriverException {
        return readNewportLamp("STB?");
    }

    /**
     * Strips off "STB" if present and converts hex code to int.
     */
    public int stripStatus(String status) throws DriverException {
        int stripped_val = 0;
        try {
            if (status.contains("STB")) {
                stripped_val = Integer.parseInt(status.substring(3), 16);
            } else {
                stripped_val = Integer.parseInt(status, 16);
            }
        } catch (NumberFormatException ex) {
            System.out.println("Failed to make an int out of status stripped string(" + status + ") : " + ex);
        }
        return(stripped_val);
    }

    public boolean statusLampOn() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 7)) != 0);
    }

    public boolean statusExt() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 6)) != 0);
    }

    public boolean statusPowerMode() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 5)) != 0);
    }

    public boolean statusCurrentMode() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 5)) != 1);
    }

    public boolean statusCalMode() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 4)) != 0);
    }

    public boolean statusFault() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 3)) != 0);
    }

    public boolean statusComm() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 2)) != 0);
    }

    public boolean statusLimit() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 1)) != 0);
    }

    public boolean statusInterlock() throws DriverException {
        return ((stripStatus(getStatus()) & 1) != 0);
    }

    /**
     * Return the error register.
     */
    public int stripError(String err) throws DriverException {
        if (err.contains("ESR")) {
            return (Integer.parseInt(err.substring(3), 16));
        } else {
            return (Integer.parseInt(err, 16));
        }
    }

    public String getError() throws DriverException {
        return (readNewportLamp("ESR?"));
    }

    public boolean errorCommand() throws DriverException {
        return (parseErrorCommand(getError()));
    }

    public boolean errorOpComplete() throws DriverException {
        return (parseErrorOpComplete(getError()));
    }

    /**
     * check the error register to see if command error bit is on.
     */
    public boolean parseErrorCommand(String err) throws DriverException {
        //System.out.println("ErrorCmd " + err);
        return ((stripError(err) & (1 << 5)) != 0);
    }

    /**
     * Check the error register to see if Operation Complete bit on.
     */
    public boolean parseErrorOpComplete(String err) throws DriverException {
        return ((stripError(err) & 1) != 0);
    }

    /**
     * Gets the NewportLamp identity.
     */
    public String getIdent() throws DriverException {
        return readNewportLamp("IDN?");
    }

    /**
     * Gets the NewportLamp amps reading.
     */
    public double getAmps() throws DriverException {
        return Double.valueOf(readNewportLamp("AMPS?"));
    }

    /**
     * sets the mode, 1 is power mode, and 0 is current mode.
     */
    public void setMode(int mode) throws DriverException {
        if ((mode != 0) && (mode != 1)) {
            throw new IllegalArgumentException("mode is not 0 or 1");
        }
        String command = "MODE=" + Integer.toString(mode);
        String status = readNewportLamp(command);
    }

    public double getVolts() throws DriverException {
        return Double.valueOf(readNewportLamp("VOLTS?"));
    }

    public double getWatts() throws DriverException {
        return Double.valueOf(readNewportLamp("WATTS?"));
    }

    public double getCurrentPreset() throws DriverException {
        return Double.valueOf(readNewportLamp("A-PRESET?"));
    }

    public void setCurrentPreset(double a) throws DriverException {
        if ((a < minCurrent) || (a > maxCurrent)) {
            throw new IllegalArgumentException("requested value is above current Limit");
        }
        if ((a > getCurrentLim()) || (a < 0)) {
            throw new IllegalArgumentException("requested value is above current Limit");
        }
        DecimalFormat df = new DecimalFormat("##.##");
        String command = "A-PRESET=" + df.format(a);
        String err = readNewportLamp(command);
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    public double getCurrentLim() throws DriverException {
        return Double.valueOf(readNewportLamp("A-LIM?"));
    }

    public void setCurrentLim(double a) throws DriverException {
        if ((a < minCurrent) || (a > maxCurrent)) {
            throw new IllegalArgumentException("requested value is above current Limit");
        }
        DecimalFormat df = new DecimalFormat("##.##");
        String command = "A-LIM=" + df.format(a);
        String err = readNewportLamp(command);
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    public int getPowerPreset() throws DriverException {
        return Integer.valueOf(readNewportLamp("P-PRESET?"));
    }

    public void setPowerPreset(int p) throws DriverException {
        if ((p < minPower) || (p > maxPower)) {
            throw new IllegalArgumentException("power setting out of manufacturer's bounds");
        }
        if ((p < 0) || (p > getPowerLim())) {
            throw new IllegalArgumentException("power setting invalid");
        }
        String command = "P-PRESET=" + Integer.toString(p);
        String err = readNewportLamp(command);
        System.out.println("Setting power preset: " + err);
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    public int getPowerLim() throws DriverException {
        return Integer.valueOf(readNewportLamp("P-LIM?"));
    }

    public void setPowerLim(int p) throws DriverException {
        if ((p < minPower) || (p > maxPower)) {
            throw new IllegalArgumentException("power setting out of manufacturer's bounds");
        }
        if ((p < 0) || (p > 9999)) {
            throw new IllegalArgumentException("power setting invalid must be [0,9999]");
        }
        String command = "P-LIM=" + Integer.toString(p);
        String err = readNewportLamp(command);
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    /**
     * Reads a response after writing a command.
     *
     * @param command The command to write, excluding terminator
     * @return The command response string
     * @throws DriverException
     * @throws DriverTimeoutException
     */
    public synchronized String readNewportLamp(String command) throws DriverException {
        write(command);
        for ( int i = 0; i < numberOfReadAttempts; i++) {
            String result = "";
            try {
                result = read();
            } catch (DriverTimeoutException e) {

            }
            if ( ! "".equals(result) ) {                
                return result;
            }

        }
        throw new DriverException("Could not get a valid response for command: "+command);
    }

}
