package org.lsst.ccs.drivers.newportlamp;

import java.text.DecimalFormat;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 **************************************************************************
 **
 ** General access routines for the NewportLamp device
 *
 ** @author Homer Neal and Heather Kelly * BNL uses 69911 and Harvard has 69931
 * * 69931: 9600 baud, 1 stop bit, no parity
 * **************************************************************************
 */
public class NewportLamp extends Ascii {

    /**
     ** Public constants *
     *
     */
    public enum onOff {

        OFF, ON;
    }

    /**
     ** Private constants *
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private String terminator = "\r";
    private int timeout = 1000;

    /**
     * As specified in the manual pg 13, Current limits *
     */
    private double minCurrent = 3.00;
    private double maxCurrent = 24.00;
    /**
     * As specified in the manual pge 13, Power limits *
     */
    private int minPower = 40;
    private int maxPower = 300;

    public NewportLamp() throws DriverException {
    }

    /**
     **
     ** Opens a connection.
     *
     * *
     ** @param itype identifies type of connection NETWORK (0), FTDI(1), *
     * SERIAL (2)
     ** @param serialname The device name of the serial port
     ** @param baud The baud rate
     *
     */
    public void open(int itype, String serialname, int baud) throws DriverException {
        init();
        super.open(itype, serialname, baud);
    }

    /**
     ** Opens an FTDI connection.
     */
    public void openftdi(String serialname, int baud) throws DriverException {
        init();
        System.out.println("opening connection to the Newport Lamp");
        open(Ascii.CONN_TYPE_FTDI, serialname, baud);

    }

    /**
     ** Initializes device.
     *
     */
    public void init() {
        setTerminator(terminator);
    }

    /**
     ** Closes the connection. * * @throws DriverException *
     *
     */
    public void close() throws DriverException {
        close();
    }

    /**
     * required by teststand interface
     */
    public boolean getState() throws DriverException {
        return statusLampOn();
    }

    /**
     * Turns lamp on or off. *
     */
    public void setState(boolean on) throws DriverException {
        if (on) {
            writeNewportLamp("START");
        } else {
            writeNewportLamp("STOP");
        }
    }

    /**
     * reset to factory defaults *
     */
    public void reset() throws DriverException {
        writeNewportLamp("RST");
    }

    /**
     * Return the full Status byte *
     */
    public String getStatus() throws DriverException {
        return readNewportLamp("STB?");
    }

    /**
     * Strips off "STB" if present and converts hex code to int *
     */
    public int stripStatus(String status) throws DriverException {
        int stripped_val = 0;
        try {
            if (status.contains("STB")) {
                stripped_val = Integer.parseInt(status.substring(3), 16);
            } else {
                stripped_val = Integer.parseInt(status, 16);
            }
        } catch (NumberFormatException ex) {
            System.out.println("Failed to make an int out of status stripped string(" + status + ") : " + ex);
        }
        return(stripped_val);
    }

    public boolean statusLampOn() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 7)) != 0);
    }

    public boolean statusExt() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 6)) != 0);
    }

    public boolean statusPowerMode() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 5)) != 0);
    }

    public boolean statusCurrentMode() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 5)) != 1);
    }

    public boolean statusCalMode() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 4)) != 0);
    }

    public boolean statusFault() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 3)) != 0);
    }

    public boolean statusComm() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 2)) != 0);
    }

    public boolean statusLimit() throws DriverException {
        return ((stripStatus(getStatus()) & (1 << 1)) != 0);
    }

    public boolean statusInterlock() throws DriverException {
        return ((stripStatus(getStatus()) & 1) != 0);
    }

    /**
     * Return the error register *
     */
    public int stripError(String err) throws DriverException {
        if (err.contains("ESR")) {
            return (Integer.parseInt(err.substring(3), 16));
        } else {
            return (Integer.parseInt(err, 16));
        }
    }

    public String getError() throws DriverException {
        return (readNewportLamp("ESR?"));
    }

    public boolean errorCommand() throws DriverException {
        return (parseErrorCommand(getError()));
    }

    public boolean errorOpComplete() throws DriverException {
        return (parseErrorOpComplete(getError()));
    }

    /**
     * check the error register to see if command error bit is on *
     */
    public boolean parseErrorCommand(String err) throws DriverException {
        //System.out.println("ErrorCmd " + err);
        return ((stripError(err) & (1 << 5)) != 0);
    }

    /**
     * Check the error register to see if Operation Complete bit on *
     */
    public boolean parseErrorOpComplete(String err) throws DriverException {
        return ((stripError(err) & 1) != 0);
    }

    /**
     * Gets the NewportLamp identity. *
     */
    public String getIdent() throws DriverException {
        return readNewportLamp("IDN?");
    }

    /**
     * Gets the NewportLamp amps reading *
     */
    public double getAmps() throws DriverException {
        return Double.valueOf(readNewportLamp("AMPS?"));
    }

    /**
     * sets the mode, 1 is power mode, and 0 is current mode *
     */
    public void setMode(int mode) throws DriverException {
        if ((mode != 0) && (mode != 1)) {
            throw new IllegalArgumentException("mode is not 0 or 1");
        }
        String command = "MODE=" + Integer.toString(mode);
        writeNewportLamp(command);
        String status = readNewportLamp();
    }

    public double getVolts() throws DriverException {
        return Double.valueOf(readNewportLamp("VOLTS?"));
    }

    public double getWatts() throws DriverException {
        return Double.valueOf(readNewportLamp("WATTS?"));
    }

    public double getCurrentPreset() throws DriverException {
        return Double.valueOf(readNewportLamp("A-PRESET?"));
    }

    public void setCurrentPreset(double a) throws DriverException {
        if ((a < minCurrent) || (a > maxCurrent)) {
            throw new IllegalArgumentException("requested value is above current Limit");
        }
        if ((a > getCurrentLim()) || (a < 0)) {
            throw new IllegalArgumentException("requested value is above current Limit");
        }
        DecimalFormat df = new DecimalFormat("##.##");
        String command = "A-PRESET=" + df.format(a);
        writeNewportLamp(command);
        String err = readNewportLamp();
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    public double getCurrentLim() throws DriverException {
        return Double.valueOf(readNewportLamp("A-LIM?"));
    }

    public void setCurrentLim(double a) throws DriverException {
        if ((a < minCurrent) || (a > maxCurrent)) {
            throw new IllegalArgumentException("requested value is above current Limit");
        }
        DecimalFormat df = new DecimalFormat("##.##");
        String command = "A-LIM=" + df.format(a);
        writeNewportLamp(command);
        String err = readNewportLamp();
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    public int getPowerPreset() throws DriverException {
        return Integer.valueOf(readNewportLamp("P-PRESET?"));
    }

    public void setPowerPreset(int p) throws DriverException {
        if ((p < minPower) || (p > maxPower)) {
            throw new IllegalArgumentException("power setting out of manufacturer's bounds");
        }
        if ((p < 0) || (p > getPowerLim())) {
            throw new IllegalArgumentException("power setting invalid");
        }
        String command = "P-PRESET=" + Integer.toString(p);
        writeNewportLamp(command);
        String err = readNewportLamp();
        System.out.println("Setting power preset: " + err);
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    public int getPowerLim() throws DriverException {
        return Integer.valueOf(readNewportLamp("P-LIM?"));
    }

    public void setPowerLim(int p) throws DriverException {
        if ((p < minPower) || (p > maxPower)) {
            throw new IllegalArgumentException("power setting out of manufacturer's bounds");
        }
        if ((p < 0) || (p > 9999)) {
            throw new IllegalArgumentException("power setting invalid must be [0,9999]");
        }
        String command = "P-LIM=" + Integer.toString(p);
        writeNewportLamp(command);
        String err = readNewportLamp();
        if (parseErrorCommand(err)) {
            throw new DriverException("Command Error");
        }
    }

    /**
     ** Sets the receive timeout. * * @param timeout The receive timeout (ms).
     * 0 means no timeout. * * @throws DriverException *
     */
    public void setTimeout(int timeout) throws DriverException {
        this.timeout = timeout;
    }

    /**
     ** Writes a command. * * @param command The command to write, excluding
     * terminator * * @throws DriverException *
     *
     */
    public synchronized void writeNewportLamp(String command) throws DriverException {
        // write(command + terminator);
        write(command);
    }

    /**
     ** Reads a response. * * @return The command response string * * @throws
     * DriverException
     *
     ** @throws DriverTimeoutException *
     */
    public synchronized String readNewportLamp() throws DriverException {
        return (read());
    }

    /**
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     *
     ** @throws DriverTimeoutException *
     */
    public synchronized String readNewportLamp(String command) throws DriverException {
        writeNewportLamp(command);
        return readNewportLamp();
    }

}
