package org.lsst.ccs.drivers.newportlamp;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 **************************************************************************
 **
 ** General access routines for the NewportLamp device 
 ** @author Homer Neal *
 * **************************************************************************
 */
public class NewportLamp extends Ascii {

    /**
     **************************************************************************
     **
     ** Public constants *
     * **************************************************************************
     */

    public enum onOff {

        OFF, ON;
    }

    /**
     **************************************************************************
     **
     ** Private constants *
     * **************************************************************************
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private String terminator = "\r\n";
    private int timeout = 1000;
    private String saddr;

    /**
     **************************************************************************
     **
     ** Constructor * *
     * **************************************************************************
     */
    public NewportLamp() throws DriverException {
    }

    /**
     **************************************************************************
     **
     ** Opens a connection. * * @param device The device name of the serial
     * port
     ** @param baud The baud rate
     ** @param dummy Dummy argument * * @throws DriverException *
     * **************************************************************************
     */
    public void open(String serialname, int port, int addr) throws DriverException
    {
        open(Ascii.CONN_TYPE_SERIAL, serialname, port);
        Integer iaddr = addr;
        saddr = iaddr.toString();
        init();
//        setTimeout(timeout); // set SCPI timeout    
    }

      /**
    ***************************************************************************
    **
    **  Opens an FTDI connection.
    **
    ***************************************************************************
    */
    public void openftdi(String serialname, int port, int addr) throws DriverException
    {
        System.out.println("opening connection to the Newport Lamp");
        open(Ascii.CONN_TYPE_FTDI, serialname, port);
        Integer iaddr = addr;
        saddr = iaddr.toString();
        init();
//        setTimeout(timeout); // set SCPI timeout
    }

      /**
    ***************************************************************************
    **
    **  Initiolizes device.
    **
    ***************************************************************************
    */
    public void init()
    {
    }    
    
    /**
     **************************************************************************
     **
     ** Closes the connection. * * @throws DriverException *
     * *************************************************************************
     */
    public void close() throws DriverException {
//        port.closePort();
    }

    /**
     ***************************************************************************
     **
     ** Turns lamp on or off. *
     * **************************************************************************
     */
    public void setState(boolean on) throws DriverException {
        if (on) {
            writeNewportLamp("START");
        } else {
            writeNewportLamp("STOP");
        }
    }

    /**
     ***************************************************************************
     **
     ** Gets the NewportLamp power state. *
     * **************************************************************************
     */
    public boolean getState() throws DriverException {
        return (Integer.decode(readNewportLamp("STB?")) & (2>>7)) != 0 ;
    }

    /**
     ***************************************************************************
     **
     ** Gets the NewportLamp identity. *
     * **************************************************************************
     */
    public String getIdent() throws DriverException {
        return readNewportLamp("IDN?") ;
    }

    /**
     ***************************************************************************
     **
     ** Gets the NewportLamp amps reading *
     * **************************************************************************
     */
    public double getAmps() throws DriverException {
        return Double.valueOf(readNewportLamp("AMPS?"));
    }
    /**
     **************************************************************************
     **
     ** Sets the receive timeout. * * @param timeout The receive timeout (ms).
     * 0 means no timeout. * * @throws DriverException *
     * *************************************************************************
     */
    public void setTimeout(int timeout) throws DriverException {
        this.timeout = timeout;
    }

        /**
     **************************************************************************
     **
     ** Writes a command. * * @param command The command to write, excluding
     * terminator * * @throws DriverException *
     * *************************************************************************
     */
    public synchronized void writeNewportLamp(String command) throws DriverException {
//        checkOpen();
        write(command + terminator);
    }

    /**
     **************************************************************************
     **
     ** Reads a response. * * @return The command response string * * @throws
     * DriverException
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public synchronized String readNewportLamp() throws DriverException {
//        checkOpen();
        return(read());
    }

    /**
     **************************************************************************
     **
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public synchronized String readNewportLamp(String command) throws DriverException {
//        flush();

        writeNewportLamp(command);
        return readNewportLamp();
    }

}
