package org.lsst.ccs.drivers.nanotec;

import java.io.OutputStream;
import java.lang.Math;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  Driver for Nanotec PD4-N stepper motor.
 *
 *  20-May-2019:  remove assignment of digital output line to error
 *  (doesn't work as expected).  Instead, save value of latest location
 *  in error memory before a move; a change in this indicates an error.
 *
 *  @author Al Eisner
 */

public class NanotecPD4N extends Ascii {

    private final int ADDR_DEFAULT = 1;  // RS485 device address
    private final int BAUD_DEFAULT = 115000;
    private final int TIMEOUT = 1000;    // milliseconds

    private int driveAddr = ADDR_DEFAULT;
    private boolean useCRC = false;      // needs some value to start
    private boolean debug = false;       // true to enable debug printout
    private int lastErrorLocation;      // location in error memory

    /* Label identifying source for messages */
    private final static String Lbl = "NanotecPD4N:  ";

    /**
     *  Hardware commands for setting or reading general parameters.
     *
     *  Commands not relevant to expected operation are not implemented.
     *  Several commands which require special treatment are not enumerated.
     */

    public enum CmndGeneral {

        /**
         *  "Short-format" commands are single-character. (The hardware has 
         *  a few which are several characters long, but none is implemented
         *  here.)  If writing a particular quantity is allowed, it is read
         *  back by prefacing the command with a "Z".  If writing is not
         *  allowed, the basic command reads the value.
         *
         *  "Long-format" commands all begin with a ":".  They differ from
         *  short-format commands in write and read syntax.
         *
         *  Commands are listed here in order of presentation in the
         *  device Programming Manual, except that long-format commands
         *  are separated for readability.
         */

        PHASE_CURRENT       ("i", true,  0, 150, "percent"),
        PHASE_CURRENT_STILL ("r", true,  0, 150, "percent"),
	STEP_MODE           ("g", true,  1, 255, ""),
	SWITCH_BEHAVIOR     ("l", true,  2565, 17442, "(bitcode)"),
	// Limits correspond to bit codes 0XA05, 0X4422 
	ERRCORR_MODE        ("U", true,  0, 1, "(0/1 for off/on)"),
	ERRCORR_RECNUM      ("F", true,  0, 32, ""),
	ENCODER_DIR         ("q", true,  0, 1, ""), 
	ENCODER_SETTLE_TIME ("O", true,  0, 250, "10 ms"),
	ENCODER_MAXDEV      ("X", true,  0, 250, "steps"),
	READ_ERROR_MEMORY   ("E", false, 0, 32, ""),  // special behavior
	// (bare command reads address of most recent error)
	READ_ENCODER        ("I", false, 0, 0, ""),
	READ_STATUS         ("$", false, 0, 0, "(bitcode)"),
	FIRMWARE_VERSION    ("v", false, 0, 0, ""),
	REVERSE_POLARITIES  ("h", true,  0, 524287, "(bitcode)"), // bits 0-18
	INPUT_DEBOUNCE_TIME ("K", true,  0, 250, "ms"),
	REVERSE_CLEARANCE   ("z", true,  0, 9999, "steps"),

	OPERATING_TIME (":optime",     false, 0, 0, ""),
	RAMP_TYPE      (":ramp_mode",  true,  0, 2, ""),
	BAUD_RATE      (":baud",       true,  1, 12, ""),  // 110-115200 baud
	TEMPERATURE    (":temp_adc",   false, 0, 0, ""),   // needs decoding
	QUICKSTOP_RAMP (":decelquick", true,  0, 3000000, "Hz/s");

        private String symbol;
        private boolean writeAllowed;
        private int minValue;
        private int maxValue;
        private String units;
        private boolean longFormat;
        
        CmndGeneral(String symbol, boolean writeAllowed,
                    int minValue, int maxValue, String units) {
            this.symbol = symbol;
            this.writeAllowed = writeAllowed;
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.units = units;
            this.longFormat = symbol.startsWith(":");
        }
        public String getUnits() {return units;}
    }

    /**
     *  Commands to be used internally, not exposed outside class.
     *  The first set is for initialization purposes.  The second set
     *  is for use by individual public methods.
     */

    private enum CmndPrivate {

	MOTOR_TYPE      (":CL_motor_type", true,  0, 2),   //  Set to 0
	DRIVE_ADDRESS   ("m",              true,  1, 254), //  Set to 1
	DIGITAL_INPUT1  (":port_in_a",     true,  0, 13),  //  Set to 7
	DIGITAL_INPUT2  (":port_in_b",     true,  0, 13),  //  Set to 7
	AUTOMATE_STATUS ("J", true,  0, 1),                //  Set to 0
	CLOSED_LOOP     (":CL_enable",     true,  0, 3),   //  Set to 0

	USE_CHECKSUM    (":crc",           true,  0, 1),
	RESET_ERROR     ("D", true,  -100000000, 100000000),
	OUTPUT_LINES    ("Y", false, 0, 0),   //  Bitcode, read bits 0,1,16
        START_MOTOR     ("A", false, 0, 0),
	STOP_MOTOR      ("S", true,  0, 1),   // 0/1 for quickstop/brake ramp
	LOAD_RECORD     ("y", true,  1, 32),  // from EEPROM
	SAVE_RECORD     (">", true,  1, 32);  // to EEPROM

        private String symbol;
        private boolean writeAllowed;
        private int minValue;
        private int maxValue;
        private boolean longFormat;
        
        CmndPrivate(String symbol, boolean writeAllowed, 
                    int minValue, int maxValue) {
            this.symbol = symbol;
            this.writeAllowed = writeAllowed;
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.longFormat = symbol.startsWith(":");
        }
    }

    /**
     *  Hardware commands to set/read parameters of a motion (run) record.
     */

    public enum CmndMotion {
        MODE                ("p",      1, 4, ""),    //Allow only four modes
        DISTANCE            ("s",      -100000000, 100000000, "steps"),
        FREQUENCY_MIN       ("u",      1, 160000, "steps/s"),
        FREQUENCY_MAX       ("o",      1, 1000000, "steps/s"),
        FREQ2_MAX           ("n",      1, 1000000, "steps/s"),  // Not used
        RAMP_ACCEL          (":accel", 1, 3000000, "Hz/s"),
        RAMP_BRAKE          (":decel", 0, 3000000, "Hz/s"),
        ROTATION_DIRECTION  ("d",      0, 1, ""),
        DIR_CHANGE_FOR_REP  ("t",      0, 1, ""),               // Not used
        REPETITIONS         ("W",      0, 254, ""),             // Set to 1
        PAUSE_BETW_RECORDS  ("P",      0, 65535, "ms"),         // Not used
        CONTINUATION_RECNUM ("N",      0, 32,""),
        JERK_MAX_ACCEL      (":b",     1, 100000000, "100/s^3"),
	JERK_MAX_BRAKE      (":B",     0, 100000000, "100/s^3");

        private String symbol;
        private int minValue;
        private int maxValue;
        private String units;
        private boolean longFormat;
        
        CmndMotion(String symbol, int minValue, int maxValue, String units) {
            this.symbol = symbol;
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.units = units;
            this.longFormat = symbol.startsWith(":");
        }
        public String getUnits() {return units;}
    }

    /**
     *  Meaning of error bits (in error memory)
     */

    public enum MotorErrors {

        LOWVOLTAGE  (0x01, "under voltage"),
        TEMPERATURE (0x02, "controller temperature out of range"),
        OVERCURRENT (0x04, "overcurrent triggered switch-off"),
	EEPROM      (0x08, "incorrect data in EEPROM"),
        POSITION    (0x10, "position error"),
        INTERNAL    (0x20, "internal error"),
        UNASSIGNED  (0x40, "unassigned bit"),
        COMPONENT   (0x80, "component returned one error");

        private int mask;
        private String descr;

        MotorErrors(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public static String decode(int errorCode) {
            String meaning = "";
            MotorErrors[] motorErr = MotorErrors.values();
            int nErrors = motorErr.length;
            for (int i = 0; i < nErrors; i++) {
                if ((errorCode & motorErr[i].mask) != 0) {
                    meaning += (motorErr[i].toString() + " ");
                }
            }
            return meaning;
        }
    }

    /**
     *  Enumeration of choices for date rate
     */

    enum DataRate {

        BAUD_110(1),
        BAUD_300(2),
        BAUD_600(3),
        BAUD_1200(4),
        BAUD_2400(5),
        BAUD_4800(6),
        BAUD_9600(7),
        BAUD_14400(8),
        BAUD_19200(9),
        BAUD_38400(10),
        BAUD_5760000(11),
        BAUD_115200(12);

        private int rateCode;    // Code for Nanotec command

        DataRate(int rateCode) {
            this.rateCode = rateCode;
        }

        int getCode() {
            return rateCode;
        }
    }

    /**
     *  Enumerated choices for external limit-switch behavior (normal run)
     *  These determine the motor's behavior when a limit switch registers.
     *  "TRAVEL" means to move motor until limit switch is unset.
     *  These correspond to bits 11 to 14 of CmndGeneral.SWITCH_BEHAVIOR.
     */

    enum LimitSwitchBehavior {
        TRAVEL_FORWARD(0X800),
	TRAVEL_BACKWARD(0X1000),
	STOP(0X2000),
	IGNORE(0X4000);

        // other bits in SWITCH_BEHAVIOR
        private final static int otherBits = 0X7FF;   

        private int bitCode;

        LimitSwitchBehavior(int bitCode) {
            this.bitCode = bitCode;
	}

        public static String decode(int bitPattern) throws DriverException {
            int bits = bitPattern & 0X7800;
            LimitSwitchBehavior[] behavior = LimitSwitchBehavior.values();
            int found = -1;
            for (int i = 0; i < behavior.length; i++) {
                if (bits == behavior[i].bitCode) {
                    found = i;
                    break;
                }
            }
            if (found==-1) {
                throw new DriverException(Lbl + "invalid code for limit-switch behavior");
            }
            return behavior[found].toString();
        }
    }
            

    /**

     *  Constructor.
     */
    public NanotecPD4N()
    {
        super(Option.NO_NET);   // Disallow network connections
    }

    /**
     *  Override underlying open method of Ascii class.
     *
     *  Note that default communication parameters (specified with 0
     *  for commParm) match what is used by the motor controller.
     *
     *  @param  connType  The enumerated connection type: must be SERIAL
     *  @param  ident     The port name (SERIAL)
     *  @param  baudRate  The baud rate, or 0 for the default (115200)
     *  @param  commParm  The communications parameters (ignored, 0 used)
     *
     *  @throws  DriverException
     */
    @Override
    public synchronized void open(ConnType connType, String ident, 
        int baudRate, int commParm) throws DriverException
    {
        super.open(connType, ident, baudRate == 0 ? BAUD_DEFAULT : baudRate, 0);
        setTimeout(TIMEOUT);
        setTerminator(Terminator.CR);

        /*   Initialize some controller settings */

        setChecksumUsage(true);
        try {
            // Fixed parameters, not subject to outside change
	    writePrivate(CmndPrivate.MOTOR_TYPE,      0);
	    writePrivate(CmndPrivate.DRIVE_ADDRESS,   1);
	    writePrivate(CmndPrivate.DIGITAL_INPUT1,  7);
	    writePrivate(CmndPrivate.DIGITAL_INPUT2,  7);
	    writePrivate(CmndPrivate.AUTOMATE_STATUS, 0);
	    writePrivate(CmndPrivate.CLOSED_LOOP,     0);
            // Get initial value for last-used error-memory location
            lastErrorLocation = getErrorMemoryLocation();
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }

    /**
     *  Open serial connection, default data characteristics and baud rate
     *
     *  @param  serialName   Serial device name
     *  @throws DriverException
     */
    public void open(String serialName) throws DriverException {
        openSerial(serialName, BAUD_DEFAULT);
    }

    /**
     *  Open serial connection with selected baud rate
     *
     *  @param  serialName   Serial device name
     *  @param  baudRate     Baud rate
     *  @throws DriverException
     */
    public void open(String serialName, int baudRate) throws DriverException {

    /* Check validity of baudRate (throws IllegalArgumentException) */
         DataRate dataRate = DataRate.valueOf("BAUD_"+ Integer.toString(baudRate));        
         openSerial(serialName, baudRate);
    }


    /** Write general datum to motor controller
     *
     *  @param   cmnd   enumerated command identifier
     *  @param   ndata  integer data to write
     *  @throws  DriverException
     */
    public void writeGeneral(CmndGeneral cmnd, int value) throws DriverException {
        if (!cmnd.writeAllowed) {
            throw new DriverException(Lbl + "Write not allowed for " + cmnd.toString());
        }
        if (value < cmnd.minValue || value > cmnd.maxValue) {
            throw new DriverException(Lbl + "Value outside of allowed range "
             + String.valueOf(cmnd.minValue) + " to " 
             + String.valueOf(cmnd.maxValue) + " for " + cmnd.toString());
        }
        String cmndText = "Write " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr) + cmnd.symbol;
        if (!cmnd.longFormat) {                // Short-format command
            cmndBody += String.valueOf(value);
        } else {                               // Long-format command
            cmndBody += ("=" + String.valueOf(value));
        }

        responseToCommand(cmndBody, cmndText);
        return;
    }


    /** Write private datum to motor controller
     *
     *  @param   cmnd   enumerated command identifier
     *  @param   ndata  integer data to write
     *  @throws  DriverException
     */
    private void writePrivate(CmndPrivate cmnd, int value) throws DriverException {
        if (!cmnd.writeAllowed) {
            throw new DriverException(Lbl + "Write not allowed for " + cmnd.toString());
        }
        if (value < cmnd.minValue || value > cmnd.maxValue) {
            throw new DriverException(Lbl + "Value outside of allowed range "
             + String.valueOf(cmnd.minValue) + " to " 
             + String.valueOf(cmnd.maxValue) + " for " + cmnd.toString());
        }
        String cmndText = "Write " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr) + cmnd.symbol;
        if (!cmnd.longFormat) {                // Short-format command
            cmndBody += String.valueOf(value);
        } else {                               // Long-format command
            cmndBody += ("=" + String.valueOf(value));
        }

        responseToCommand(cmndBody, cmndText);
        return;
    }

    /** Private write-command with no data to motor controller
     *
     *  @param   cmnd   enumerated command identifier
     *  @throws  DriverException
     */
    private void writePrivate(CmndPrivate cmnd) throws DriverException {
        if (!cmnd.writeAllowed) {
            throw new DriverException(Lbl + "Write not allowed for " + cmnd.toString());
        }
        String cmndText = "Write " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr) + cmnd.symbol;

        responseToCommand(cmndBody, cmndText);
        return;
    }


    /** Write motion-record datum to motor controller
     *
     *  @param   cmnd   enumerated command identifier
     *  @param   ndata  integer data to write
     *  @throws  DriverException
     */

    public void writeMotion(CmndMotion cmnd, int value) throws DriverException {

        if (!isMotorReady()) {
            throw new DriverException(Lbl + "motion parameters cannot be " +
                                      "changed while motor not ready");
        }
        if (value < cmnd.minValue || value > cmnd.maxValue) {
            throw new DriverException(Lbl + "Value outside of allowed range "
             + String.valueOf(cmnd.minValue) + " to " 
             + String.valueOf(cmnd.maxValue) + " for " + cmnd.toString());
        }
        String cmndText = "Write " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr) + cmnd.symbol;
        if (!cmnd.longFormat) {                // Short-format command
            cmndBody += String.valueOf(value);
        } else {                               // Long-format command
            cmndBody += ("=" + String.valueOf(value));
        }

        responseToCommand(cmndBody, cmndText);
        return;
    }


    /** Read general datum from motor controller
     *
     *  @param   cmnd   enumerated command identifier
     *  @return  String data
     *  @throws  DriverException
     */

    public String readGeneral(CmndGeneral cmnd) throws DriverException {
        String cmndText = "Read " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr);
        if (!cmnd.longFormat) {                // Short-format command
            if (cmnd.writeAllowed) {
                cmndBody += "Z";
            }
            cmndBody += cmnd.symbol;
        } else {                               // Long-format command
            cmndBody += cmnd.symbol;
        }

        String response = responseToCommand(cmndBody, cmndText);
        return response.substring(cmndBody.length());       
    }


    /** Read private datum from motor controller
     *
     *  @param   cmnd   enumerated command identifier
     *  @return  integer data
     *  @throws  DriverException
     */
    private String readPrivate(CmndPrivate cmnd) throws DriverException {
        String cmndText = "Read " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr);
        if (!cmnd.longFormat) {                // Short-format command
            if (cmnd.writeAllowed) {
                cmndBody += "Z";
            }
            cmndBody += cmnd.symbol;
        } else {                               // Long-format command
            cmndBody += cmnd.symbol;
        }

        String response = responseToCommand(cmndBody, cmndText);
        return response.substring(cmndBody.length());       
    }


    /** Read motion-record datum from motor controller 
     *
     *  @param   cmnd   enumerated command identifier
     *  @return  integer data
     *  @throws  DriverException
     */
    public String readMotion(CmndMotion cmnd) throws DriverException {
        String cmndText = "Read " +  cmnd.toString();
        String cmndBody = String.valueOf(driveAddr);
        if (!cmnd.longFormat) {                // Short-format command
            cmndBody += "Z";
            cmndBody += cmnd.symbol;
        } else {                               // Long-format command
            cmndBody += cmnd.symbol;
        }

        String response = responseToCommand(cmndBody, cmndText);
        return response.substring(cmndBody.length());       
    }


    /**
     *  Wrapper for Ascii.read(String command). 
     *  It additionally checks for a motor response indicating an invalid
     *  command.  If the CRC checksum is enabled, this method provides the 
     *  checksum to send, and verifies the checksum in the response.
     *
     *  First version:  omit checksum usage.
     *
     *  @param  Command body without checksum
     *  @return Command response, if valid, without checksum
     *  @throws DriverException
     */
    private String responseToCommand(String cmndBody, String cmndText) 
     throws DriverException {
        String cmndSend = "#"+cmndBody;
        if (useCRC) {
	    cmndSend += String.format("\t%02X",computeCRC(cmndSend));
        }
        String response = read(cmndSend);
        if (debug) {
            System.out.println("Command sent = " + cmndSend 
                               + ", response = " + response);
        }
        if (response.indexOf("?crc") > -1) {
            throw new DriverException(Lbl +"controller reports checksum error");
        }
        if (response.indexOf("?") > -1) {
            String cmndSym = cmndBody.substring(1);  // strip drive address
            throw new DriverException(Lbl + "invalid command " + cmndSym
            + " (" + cmndText + ")");
        }
        int indexTab = response.indexOf("\t");
        boolean foundCrc = (indexTab > -1);
        if (useCRC != foundCrc) {
            throw new DriverException(Lbl + String.format("checksum expected = %b, found = %b", useCRC, foundCrc));
        }
        if (foundCrc) {
            String crcStr = response.substring(indexTab + 1);
	    String expect = String.format("%02X",computeCRC(response.substring(0,indexTab)));
            if (!crcStr.equals(expect)) {
	        throw new DriverException(Lbl + "checksum = " + crcStr + 
					  ", expected = " + expect);
            }
            return response.substring(0,indexTab);
        } else {
            return response;
        }
    }


    /**
     *  Compute CRC (checksum)
     *
     *  @param  String for which CRC is computed
     *  @return byte    computed CRC
     */
    private byte computeCRC(String input) {
        final byte polyCRC = 0x07;
        byte [] inArray = input.getBytes();
        int nbyte = inArray.length;
        byte crc = 0;
        for (int ibyte = 0; ibyte < nbyte; ibyte++) {
            for (int i = 0; i < 8; i++) {
                if ((crc & 0x80) != (inArray[ibyte] & 0x80)) {
                    crc = (byte)((crc << 1)^polyCRC);
                } else {
                    crc <<= 1;
                }
                inArray[ibyte] <<= 1;
            }
        }
        return crc;
    }


    /**
     *  Special hardware commands (requiring decoding or multiple steps)
     */


    /**
     *  Read status.  (Necessary because readGeneral(READ_STATUS) returns
     *  two extra leading 0's, which the company hasn't explained.
     *
     *  @return int 
     *  @throws Drier Exception
     */
    public int readStatus() throws DriverException {
        String resp = readGeneral(CmndGeneral.READ_STATUS);
        int idx = resp.indexOf(CmndGeneral.READ_STATUS.symbol);
        String statusStr = resp.substring(idx+1);
        return Integer.parseInt(statusStr);
    }

    /**
     *  Check if motor is in a ready state (for a new move command)
     *
     *  @return boolean (true if ready)
     *  @throws DriverException
     */
    public boolean isMotorReady() throws DriverException { 
        int status = readStatus();
        return ((status & 1) != 0);
    }

    /**
     *  Check if motor status shows a position error
     *
     *  @return boolean (true if position error)
     *  @throws DriverException
     */
    public boolean isPositionError() throws DriverException { 
        int status = readStatus();
        return ((status & 4) != 0);
    }

    /**
     *  Read limit switches (assigned to digital inputs 1 and 2)
     *
     *  @return int limitSwithces (two-bit pattern)
     *  @throws DriverException
     */
    public int getLimitSwitches() throws DriverException {
        int outputs = Integer.parseInt(readPrivate(CmndPrivate.OUTPUT_LINES));
        return outputs & 0X3;
    }

    /**
     *  Get last location used in error memory
     *
     *  @return  int  last error-memory location used
     *  @throws  DriverException
     */
    private int getErrorMemoryLocation() throws DriverException {
        String errMem = readGeneral(CmndGeneral.READ_ERROR_MEMORY);
        int errorRecord = Integer.parseInt(errMem);
        if (errorRecord < 1 || errorRecord > 32) {
            throw new DriverException(
             Lbl + "Error record number "+ errMem +" is out of range 1 to 32");
        }
        return errorRecord;
    }

    /**
     *  Return equivalent of hardware error flag (which did not work)
     *
     *  @return boolean errorFlag (true if error)
     *  @throws DriverException
     */
    public boolean getErrorFlag() throws DriverException {
        return !getErrorType().equals("NONE");
    }

    /**
     *  Read error type(s).  If current location in error memory != the
     *  last saved location, read last entry and list error type(s) set.
     *
     *  @return String  space-separated list of error types 
     *  @throws DriverException
     */
    public String getErrorType() throws DriverException {
        String errorType = "";
        int currentErrorLocation = getErrorMemoryLocation();
        if ((currentErrorLocation != lastErrorLocation) || debug) {
            if (debug && (currentErrorLocation == lastErrorLocation)) {
                System.out.println("Debug mode: read although not new");
            }
            lastErrorLocation = currentErrorLocation;
            String cmndBody = String.valueOf(driveAddr) + "Z" + String.valueOf(currentErrorLocation) + "E";
            String cmndText = "Read error-memory record " + String.valueOf(currentErrorLocation);
            String response = responseToCommand(cmndBody, cmndText);
            int errorCode = Integer.parseInt(response.substring(cmndBody.length()));
            errorType = MotorErrors.decode(errorCode);
	}
        return !errorType.equals("") ? errorType : "NONE";
    }

    /**
     *  Reset error flag (and position-error flag)
     *  Only the default (no-parameter) version is implemented here:
     *  if error is in position-following, the motor's position value
     *  is set to that of the encoder.
     *
     *  @throws DriverException
     */
    public void resetError() throws DriverException {
        writePrivate(CmndPrivate.RESET_ERROR);
    }

    /**
     *  Read controller-temperature ADC and decode to degrees C
     *
     *  @return double temperature in degrees C
     *  @throws DriverException
    */

    public double readTemperature() throws DriverException {
        String resp =  readGeneral(CmndGeneral.TEMPERATURE);
        double adc = (double) Integer.parseInt(resp);
        double frac = adc/1023.;
        double temperature = (1266500./(4250.+Math.log10(0.33*frac/(1.-frac))
					* 298.)) - 273.;
        return temperature;
    }

    /**
     *  Set limit-switch behavior.
     *
     *  @param  boolean <true|false> for <free backward motion|stop>
     *  @throws DriverException
     */

    public void setLimitSwitchBehavior(boolean value) throws DriverException {
        int bitcode = (value ? LimitSwitchBehavior.TRAVEL_BACKWARD : LimitSwitchBehavior.STOP).bitCode;
        int setting = Integer.parseInt(readGeneral(CmndGeneral.SWITCH_BEHAVIOR));
        int newSetting = (setting & LimitSwitchBehavior.otherBits) | bitcode;
        writeGeneral(NanotecPD4N.CmndGeneral.SWITCH_BEHAVIOR, newSetting);
    }

    /**
     *  Motion and record commands
     */

    /**
     *  Start motor (move according to parameters in current record)
     *
     *  @throws DriverException
     */
    public void startMotor() throws DriverException{
        if (!isMotorReady()) {
            throw new DriverException(Lbl + "startMotor invoked while motor not ready");
        }
        lastErrorLocation = getErrorMemoryLocation();
        readPrivate(CmndPrivate.START_MOTOR);
    }

    /**
     *  Stop motor (if currently moving)
     *
     *  @param int ramp   <0|1> for <quickstop | current brake ramp>
     *  @throws DriverException
     */
    public void stopMotor(int ramp) throws DriverException{
        writePrivate(CmndPrivate.STOP_MOTOR, ramp);
    }

    /**
     *  Load movement record from EEPROM
     *
     *  @param int record   record number (1 to 32)
     *  @throws DriverException
     */
    public void loadRecord(int record) throws DriverException {
        writePrivate(CmndPrivate.LOAD_RECORD, record);
    } 

    /**
     *  Save current movement record to EEPROM
     *
     *  @param int record   record number (1 to 32)
     *  @throws DriverException
     */
    public void saveRecord(int record) throws DriverException {
        writePrivate(CmndPrivate.SAVE_RECORD, record);
    } 

    /**
     *  Read whether checksum is in use
     *
     *  @return boolean <true|false> if checksum <in use | not in use>
     */
    public boolean readChecksumUsage() {
        return useCRC;
    }

    /**
     *  Turn usage of checksum on or off.  Available only within package.
     *  Value must be set both in this software and in hardware.
     *
     *  @param  boolean  value to be set
     *  @throws Driverexception
     */
    void setChecksumUsage(boolean value) throws DriverException {
        try {
            writePrivate(CmndPrivate.USE_CHECKSUM, value ? 1 : 0);
            useCRC = value;  // do this only after setting controller value
        }

        //  If this fails, it might be because current flag useCRC does
        //  not match parameter set in the motor controller.  If so,
        //  reverse useCRC and try again.  If that still generates an
        //  exception, restore useCRC to its initial value.
				       
        catch (DriverException e) {
            useCRC = !useCRC;
            try {
                writePrivate(CmndPrivate.USE_CHECKSUM, value ? 1 : 0);
                useCRC = value;
            }
            catch (DriverException ex) {
                useCRC = !useCRC;
                throw ex;
	    }
        }
    }

    /**
     *  Turn debug flag on or off.  Available only within this package.
     * 
     *  @param  boolean value to set
     */
    void setDebug(boolean value) {
        debug = value;
    }
}
