package org.lsst.ccs.drivers.nanotec;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.TestAscii;
import java.util.Date;

/**
 *  Program to test the Nanotex PD4N stepper motor driver
 *
 *  @author Al Eisner
 */

public class TestPD4N {

    private final NanotecPD4N pd4n;

    /**
     *  Constructor
     */
    public TestPD4N()
    {
        pd4n = new NanotecPD4N();
    }

    /**
     * Determines current time
     *
     * @return current time as a String
     */
    @Command(name="timestamp", description="Prints current time")
    public String timestamp() {
	Date now = new Date();
        return now.toString();
    }

     /**
      *  Opens connection to a device.
      * 
      *  @param  String ident     Identification (serial port name)
      *  @throws DriverException
      */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        pd4n.open(ident);
        pd4n.writeGeneral(NanotecPD4N.CmndGeneral.ENCODER_DIR, 1);
    }

     /**
      *  Open connection to a device with specified baud rate.
      * 
      *  @param  String ident     Identification (serial port name)
      *  @param  int    baudRate  Must match NanotecPD4N.DataRate enumeration
      *  @throws DriverException
      */
    @Command(name="open", description="Open connection with specified baud rate")
    public void open(@Argument(name="ident", description="Identification")
                     String ident, @Argument(name="baudRate",
                     description="must match NanotecPD4N.DataRate enumeration")
                     int baudRate) throws DriverException
    {
        pd4n.open(ident,baudRate);
        pd4n.writeGeneral(NanotecPD4N.CmndGeneral.ENCODER_DIR, 1);
    }


    /**
     * Read all settings and data from motor controller.
     *
     * Loops over all read commands and returns them in a table format.
     * All DriverExceptions are caught; if one occurs, the data field
     * is replaced by the text (String) associated with the exception.
     *  
     * @return  String reporting all data read and exceptions.
     */
    @Command(name="readAll", description="Read all controller settings and data")
    public String readAll()
    {
        String table = "Read all stepper motor settings and data\n" + timestamp() +"\n";

        /* General-parameter commands */

        NanotecPD4N.CmndGeneral cmndN[] = NanotecPD4N.CmndGeneral.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                String respN = pd4n.readGeneral(cmndN[i]);
                table += (respN + "   " + cmndN[i].getUnits());
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n\n";

        /* Checksum usage */

        table += String.format("Checksum in use %14b\n\n", isChecksumUsed());
                               
        /* Motion-record commands */

        table += readMotionRecord();

        return table;
    }

    /**
     * Is checksum being used?
     *
     * @return boolean  <true|false> if checksum is <in use | not in use>
     */
    @Command(name="isChecksumUsed",
             description="<true|false> if checksum is <in use | not in use>")
    public boolean isChecksumUsed() 
    {
        return pd4n.readChecksumUsage();
    }

    /**
     * Read all motion-record settings from motor controller
     *
     * Loops over all read commands and returns them in a table format.
     * All DriverExceptions are caught; if one occurs, the data field
     * is replaced by the text (String) associated with the exception.
     *  
     * @return  String reporting all data read and exceptions.
     */
    @Command(name="readMotionRecord", description="Read all motion-record settings")
    public String readMotionRecord() 
    {
        String table = "Stepper motor motion-record settings\n" + timestamp() +"\n";
        NanotecPD4N.CmndMotion cmndN[] = NanotecPD4N.CmndMotion.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                String respN = pd4n.readMotion(cmndN[i]);
                table += (String.format("%-12s",respN) + cmndN[i].getUnits());
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";
        return table;
    }

    /** 
     *  Generic read command for general quantities
     *
     *  @param   CmndGeneral  Enumerated command name
     *  @return  String       response
     *  @throws  DriverException
     */
    @Command(name="readGeneral", description="Read specified general quantity")
    public String readGeneral(@Argument(name="enum quantity", 
     description="name of general quantity") NanotecPD4N.CmndGeneral quantity) 
     throws DriverException
    {
        return pd4n.readGeneral(quantity);
    }

    /**
     *  Generic set command for a general parameter
     *
     *  @param  CmndGeneral Enumerated command name
     *  @param  int         value to be set
     *  @throws DriverException
     */
    @Command(name = "setGeneral", description = "set a general parameter")
    public void setGeneral(@Argument(name="enum quantity",
     description="CmndGeneral instance") NanotecPD4N.CmndGeneral param, 
     @Argument(name="value",description="value to be set") int value)
     throws DriverException
    {
        pd4n.writeGeneral(param, value);
    }

    /**
     *  Set motor behavior upon reaching an external limit switch
     *
     *  @param  boolean <true|false> for <free backward motion|stop>
     *  @throws DriverException
     */
    @Command(name="setLimitSwitchBehavior", 
             description="select behavior when motor reaches limit switch")
    public void setLimitSwitchBehavior(@Argument(name="boolean choice",
     description="<true|false> for <free backward motion|stop>") boolean choice)
     throws DriverException {
        pd4n.setLimitSwitchBehavior(choice);
    }


    /* A few by-item read commands */

    /**
     *  Show behavior of motor upon reaching a limit switch
     *
     *  @return String corresponds to enumerated behavior
     *  @throws DriverException
     */
    @Command(name="showLimitSwitchBehavior", 
             description="show motor behavior upon reaching a limit switch")
    public String showLimitSwitchBehavior() throws DriverException{
        int behavior = Integer.parseInt(pd4n.readGeneral(NanotecPD4N.CmndGeneral.SWITCH_BEHAVIOR));
        return NanotecPD4N.LimitSwitchBehavior.decode(behavior);
    }

    /**
     *  Read motor encoder position
     *
     *  @return  int position
     *  @throws DriverException
     */
    @Command(name="readEncoder", description="Read motor encoder position")
    public int readEncoder() throws DriverException
    {
        String resp = pd4n.readGeneral(NanotecPD4N.CmndGeneral.READ_ENCODER);
        return Integer.parseInt(resp);
    }

    /**
     *  Read motor controller temperature, converting raw reading to degrees C
     *
     *  @return  String showing pump temperature in degrees C
     *  @throws  Driver Exception
     */
    @Command(name="readTemperature", description="Read motor temperature in degrees C")
    public String readTemperature()  throws DriverException
    {
        double temp = pd4n.readTemperature();
        return String.format("% .2f", temp);
    }

    /**
     *  Repeat read of motor controller temperature n times
     *
     *  @param   Number of times to repeat reading
     *  @return  Formatted String containing all read results
     *  @throws  DriverException
     */
    @Command(name="readTempRepeat", description="Read motor T multiple times")
    public String readTempRepeat(@Argument(name="repeats",
       description="number of times") int nRepeat) throws DriverException
    {
        String output = "";
        for (int i = 0; i < nRepeat; i++) {
            output += (" " + readTemperature());
        }
        return output;
    }

    /**
     *  Read status (ready and position -error), error flag, limit switches
     *
     *  @return  String showing flags
     *  @throws  Driver Exception
     */
    @Command(name="readFlags", description="Read useful status flags and error flag")
    public String readFlags()  throws DriverException
    {
        boolean ready = pd4n.isMotorReady();
        boolean posErr = pd4n.isPositionError();
        boolean error = pd4n.getErrorFlag();
        int switches = pd4n.getLimitSwitches();
        return "Ready = " + Boolean.toString(ready) + ", Position error = " +
	 Boolean.toString(posErr) + ", Error = " + Boolean.toString(error) +
	    ", Limit switches = " + Integer.toString(switches);
    }

    /**
     *  Read decoded error type(s)
     *
     *  @return  String showing all error types set (or none)
     *  @throws  Driver Exception
     */
    @Command(name="decodeLastError", description="Decode last error types")
    public String decodeLastError()  throws DriverException
    {
        return pd4n.getErrorType();
    }

    /**
     *  Reset error flags
     *
     *  @throws DriverException
     */
    @Command(name="resetError", description="Reset controller error flags")
    public void resetError() throws DriverException 
    {
        pd4n.resetError();
    }


    /*  Motion-record commands  */

    /** 
     *  Generic read command for motion-record quantities
     *
     *  @param   CmndMotion   Enumerated command name
     *  @return  String       response
     *  @throws  DriverException
     */
    @Command(name="readMotion", description="Read specified motion quantity")
    public String readMotion(@Argument(name="enum quantity", 
     description="name of motion quantity") NanotecPD4N.CmndMotion quantity) 
     throws DriverException
    {
        return pd4n.readMotion(quantity);
    }

    /**
     *  Generic set command for a parameter in motion-record
     *
     *  @param  CmndMotion  Enumerated command name
     *  @param  int         value to be set
     *  @throws DriverException
     */
    @Command(name = "setMotion", description = "set a motion-record parameter")
    public void setMotion(@Argument(name="enum quantity",
     description="CmndMotion instance") NanotecPD4N.CmndMotion param, 
     @Argument(name="value",description="value to be set") int value)
     throws DriverException
    {
        pd4n.writeMotion(param, value);
    }

    /**
     *  Save current motion record to controller's EEPROM
     *
     *  @param  int recordNumber (memory location, 1 to 32)
     *  @throws DriverException
     */
    @Command(name="saveMotionRecord", description="save current motion record")
    public void saveMotionRecord(@Argument(name="recordNumber",
     description="location in EEPROM (1 to 32)") int recordNumber) 
     throws DriverException
    {
        pd4n.saveRecord(recordNumber);
    }

    /**
     *  Load current record from controller's EEPROM
     *
     *  @param  int recordNumber (memory location, 1 to 32)
     *  @throws DriverException
     */
    @Command(name="loadMotionRecord", description="load motion record from EEPROM")
    public void loadMotionRecord(@Argument(name="recordNumber",
     description="location in EEPROM (1 to 32)") int recordNumber) 
     throws DriverException
    {
        pd4n.loadRecord(recordNumber);
    }

    /**
     *  Start motor and move (according to current record)
     *
     *  @throws DriverException
     */
    @Command(name="move", description="Start motor and move")
    public void move() throws DriverException
    {
        pd4n.startMotor();
    }

    /**
     *  Stop motor (if currently moving), using current brake ramp value
     *
     *  @throws DriverException
     */
    @Command(name="stop",description="stop motor, using current brake ramp")
    public void stop() throws DriverException
    {
        pd4n.stopMotor(1);
    }

    /**
     *  Stop motor (if currently moving), using quickstop ramp
     *
     *  @throws DriverException
     */
    @Command(name="stopQuick",description="stop motor, using quickstop ramp")
    public void stopQ() throws DriverException
    {
        pd4n.stopMotor(0);
    }

    /*  Communication settings   */

    /**
     *  Select baud rate, 110 to 1115200 (default)
     *  For emergency use:  "open" must be called with value set in 
     *  controller, then this command can change rate to desired value.
     * 
     *  @param  DataRate   Enumerated value
     *  @throws DriverException
     */
    @Command(name="setBaudRate", description="Select baud rate")
    public void setBaudRate(@Argument(name="<DataRate>",
     description="enumerated value, BAUD_110 to BAUD_115200 (default)")
     NanotecPD4N.DataRate value) throws DriverException
    {
        pd4n.writeGeneral(NanotecPD4N.CmndGeneral.BAUD_RATE, value.getCode());
    }

    /**
     *  Turn on/of usage of CRC checksum
     *
     *  @param  boolean <true|false> for <use | do not use> checksum
     *  @throws DriverException
     */
    @Command(name="useChecksum", description="set usage of CRC checksum")
	public void useChecksum(@Argument(name="boolean setting",
	 description="<true|false> to <use | not use>") boolean setting)
         throws DriverException
    {
        pd4n.setChecksumUsage(setting);
    }

    /**
     *  Enable/disable debug printout from driver
     *
     *  @param boolean value to set
     */
    @Command(name="setDebug",description="enable/disable debug printout")
    public void setDebug(@Argument(name="enable",
			           description="setting") boolean enable) {
        pd4n.setDebug(enable);
    }
}
