package org.lsst.ccs.drivers.motrona;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.ascii.TestAscii;
import org.lsst.ccs.drivers.commons.DriverException;
import java.util.Date;

/**
 *  Program to test Motrona IV251 SSI to serial converter
 */

public class TestIV251 {

    private final MotronaIV251 iv251;

   /**
    *  Constructor
    */
    public TestIV251()
    {
        iv251 = new MotronaIV251();
    }


    /**
     * Determines current time
     *
     * @return current time as a String
     */
    @Command(name="timestamp", description="Prints current time")
    public String timestamp() {
	Date now = new Date();
        return now.toString();
    }


   /**
    *  Opens serial connection to MotronaIV251 device with default baud 
    *  rate and data characteristics.
    * 
    *  @param   ident     The identification (serial port name)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open default connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        iv251.open(ident);
    }

   /**
    *  Opens serial connection to MotronaIV251 device with specified baud 
    *  rate and data characteristics.
    * 
    *  @param   ident     The identification (serial port name)
    *  @param   baud rate (Hz)
    *  @param   data characteristics (enumerated format)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
	public void open(@Argument(name="ident", description="Identification") String ident, 
			 @Argument(name="baud", description="baud rate in Hz") int baudRate, 
                         @Argument(name="format", description="enumerated data characteristics") MotronaIV251.SerialFormat format)
                     throws DriverException
    {
        iv251.openSerial(ident, baudRate, format.getOpenFormat());
    }

   /**
    *  Closes connection
    * 
    *  @throws  DriverException
    */
    @Command(name="close", description="Closes connection to device")
    public void close() throws DriverException {
        iv251.close();
    }


   /**
    * Read all settings and data from Motrona IV251
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(name="readAll", description="Read all IV251 settings and data")
    public String readAll()
    {

        String table = "Read all IV251 settings and data\n" + timestamp() +"\n";

        MotronaIV251.CmndParam cmndN[] = MotronaIV251.CmndParam.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                int respN = iv251.readParam(cmndN[i]);
                table += String.format("%d",respN);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";
        table += String.format("\n   %-22s", "Encoder position");   
        try {
	    int pos = iv251.readEncoder();
            table += String.format("%d", pos);
        } catch (DriverException ex) {
            table += ex.getMessage();
        }
        table += String.format("\n   %-22s", "Encoder error");   
        try {
            int err = iv251.readErrorBit();
            table += String.format("%d", err);
        } catch (DriverException ex) {
            table += ex.getMessage();
        }
        table += "\n";
        return table;
    }

   /**
    *  Read and decode serial settings of device
    *
    *  @eturn  String decoding serial baud rate and format settings
    *  @throws DriverException
    */
    @Command(name="readSerialSettings", description="Read and decode settings")
    public String readSerialSettings() throws DriverException
    {
       MotronaIV251.SerialRate rate = MotronaIV251.SerialRate.decode(iv251.readParam(MotronaIV251.CmndParam.SERIAL_BAUD_RATE));
       MotronaIV251.SerialFormat format = MotronaIV251.SerialFormat.decode(iv251.readParam(MotronaIV251.CmndParam.SERIAL_FORMAT));
       return ("IV251 serial baud rate = " + rate.toString() +
             ",  format = " + format.toString());
    }
       
   /**
    *  Read encoder position.
    *
    *  @return  encoder position value
    *  @throws  Driver Exception
    */
    @Command(name="readPosition", description="Read encoder position")
    public int readPosition()  throws DriverException
    {
        return iv251.readEncoder();
    }

   /**
    *  Read encoder error bit
    *
    *  @return  <1|0> if <encoder error|no encoder error>
    *  @throws  Driver Exception
    */

    @Command(name="readEncoderError", 
    description="<1|0> if <encoder error|no encoder error>")
    public int readEncoderError()  throws DriverException
    {
        return iv251.readErrorBit();
    }


   /**
    *  Set parameter value (pending activate command)
    * 
    *  @param   Enumerated parameter name
    *  @param   Value to set
    *  @throws  DriverException
    */
    @Command(name="setParameter", 
             description="Set parameter value (pending activate command)")
    public void setParameter(@Argument(name=" enumerated CmndParam")
     MotronaIV251.CmndParam param, @Argument(name="int value") int value)
     throws DriverException 
    {
        iv251.setParam(param, value);
    }

   /**
    *  Set a parameter multiple times in close succession
    *  (PUrpose: to help diagnose intermittent timeouts)
    *
    *  @param   Enumerated parameter name
    *  @param   Value to set
    @  @param   Number of repetitions
    *  @throws  DriverException
    */
    @Command(name="repeatSetParameter", 
             description="Set parameter value multiple times in succession")
    public void repeatSetParameter(@Argument(name=" enumerated CmndParam")
     MotronaIV251.CmndParam param, @Argument(name="int value") int value,
     @Argument(name="repetitions") int reps) throws DriverException 
    {
        if (reps > 0) {
            for (int i = 0; i < reps; i++) {
                iv251.setParam(param, value);
            }
        }
    }

   /**
    *  Activate new settings
     *
    *  @throws DriverException
    */
    @Command(name="activateNewSettings",description="copy settings to registers")
    public void acivateNewSettings() throws DriverException {
        iv251.activateData();
    }

   /**
    *  Set and activate parameter value
    * 
    *  @param   Enumerated parameter name
    *  @param   Value to set
    *  @throws  DriverException
    */
    @Command(name="setAndActivateParameter", 
             description="Set and activate parameter value")
    public void setAndActivateParameter(@Argument(name=" enumerated CmndParam")
     MotronaIV251.CmndParam param, @Argument(name="int value") int value)
     throws DriverException 
    {
        iv251.setAndActivateParam(param, value);
    }

   /**
    *  Set serial baud rate (pending activate command)
    * 
    *  @param   Serial baud rate of device only (600 to 38400, default 9600)
    *  @throws  DriverException
    */
    @Command(name="setDataRate", 
             description="Set baud rate.  Danger: last-resort reset only!")
    public void setDataRate(@Argument(name=" 600 to 38400", 
     description="Motrona IV251 baud rate setting") int value) 
     throws DriverException 
    {
        iv251.setBaudRate(value);
    }


   /**
    *  Set serial format (pending activate command)
    * 
    *  @param   IV251 serial format (enum); use listSerialFormats to see enum's
    *  @throws  DriverException
    */
    @Command(name="setSerialFormat", 
             description="Set setiral format.  Danger: last-resort reset only!")
    public void setSerialFormat(@Argument(name=" <enumerated format>", 
     description="Motrona IV251 serial format setting") 
     MotronaIV251.SerialFormat format) throws DriverException 
    {
        iv251.setFormat(format);
    }

   /** 
    *  List available serial formats
    *
    *  @return  List of formats
    */
    @Command(name="listSerialFormats",description="list of enum names")
    public String listSerialFormats() {
        MotronaIV251.SerialFormat[] form = MotronaIV251.SerialFormat.values();
        String listf = "MotronIV251.SerialFormat enum instances";
        for (int i = 0; i < form.length; i++) {
            listf += ("\n   " + form[i].toString());
        }
        return listf;
    }

   /**
    *  Save settings for next power-up
    * 
    *  @throws  DriverException
    */
    @Command(name="saveSettings", 
     description="Save settings to EEProm for next power-up")
    public void saveSettings() throws DriverException 
    {
        iv251.saveToEEProm();
    }

    /**
     *  Set debug mode on or off
     *
     *  <true|false>
     */
    @Command(name="setDebug",description="set debug mode <true|false>")
    public void setDebug(@Argument(name="<true|false>",
			 description="debug mode") boolean mode)
    {
        iv251.debug = mode;
    }

    /**
     *  Set delay after completing a command to hardware
     *
     *  @param  long  delay in ms
     */
    @Command(name="setDelay",description="set delay after hardware command")
	public void setDelay(@Argument(name="delay",description="in ms")
                             long delay) 
    {
        iv251.delay = delay;
    }

}
