package org.lsst.ccs.drivers.motrona;

import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  Driver for Motrona IV251 SSI signal converter
 *  
 *  Implements functions needed for conversion of signals from an
 *  SSI encoder to RS-232 serial output.
 *
 *  @author Al Eisner
 */

public class MotronaIV251 extends Ascii {

    /**
     *  Operation commands
     */

    public enum CmndOp{

        POSITION       (":9", 1),
	ERROR_BIT      (";9", 1),
	ACTIVATE_DATA  ("67", 0),
	SAVE_TO_EEPROM ("68", 0);

        private String register;     // Ascii code
        private int type;            // 0 for write-only, 1 for read-only

        CmndOp(String register, int type) {
            this.register = register;
            this.type = type;
        }

        public String getRegister() {return register;}
    }

    /**
     *  Enumeration of commands to set or read relevant control parameters.
     *  All of these have values which are positive-definite integers.
     *  Newly set values become active after CmndOp:ACTIVATE_DATA..
     *  Data members of each enum are the corresponding register code
     *  and the range of allowed values.
     */

    public enum CmndParam{

        SSI_LOW_BIT         ("08",   0,       25),
        SSI_HIGH_BIT        ("09",   1,       25),
        SSI_BAUD_RATE       ("10", 100,   100000),
        SSI_WAIT_TIME       ("11",   0,    10000),
        SSI_OFFSET          ("12",   0, 99999999),
        SSI_SET_VALUE       ("13",   0, 99999999),
        SSI_ERROR_BIT       ("14",   0,       25),
        SSI_ERROR_POLARITY  ("15",   0,        1),
        UNIT_NUMBER         ("90",   0,       99),
        SERIAL_BAUD_RATE    ("91",   0,        6),
        SERIAL_FORMAT       ("92",   0,        9),
        SERIAL_PROTOCOL     ("30",   0,        1),
	SERIAL_TIMER        ("31",   0,   999999),  // ms, 0 for normal use
	SERIAL_VALUE        ("32",   0,       19);
    
        private String register;     // Ascii code
        private int minValue;
        private int maxValue;
        
        CmndParam(String register, int minValue, int maxValue) {
            this.register = register;
            this.minValue = minValue;
            this.maxValue = maxValue;
        }
        
        public String getRegister() {return register;}
        public int getMinValue() {return minValue;}
        public int getMaxValue() {return maxValue;}
    }

    /**
     *  Enumeration of choices for serial baud rate
     */

    enum SerialRate {

        BAUD_9600(0),
	BAUD_4800(1),
	BAUD_2400(2),
	BAUD_1200(3),
	BAUD_600(4),
	BAUD_19200(5),
	BAUD_38400(6);

        private int rateCode;    // Code for converter command

        SerialRate(int rateCode) {
            this.rateCode = rateCode;
        }

        /**
         *  Method to decode the meaning of an IV251 baud rate code
         */

        public static SerialRate decode(int code) throws DriverException {
            SerialRate[] rates = SerialRate.values();
            int n = rates.length;
            int found = -1;
            for (int i = 0; i < n; i++) {
                if (rates[i].rateCode == code) {
                    found = i;
                    break;
                }
            }
            if (found == -1) throw new DriverException(Lbl + "invalid format code");
            return rates[found];
        }
    }

   /**
    *  Enumeration of choices for serial format (bits, parity, stop bits)
    */

    enum SerialFormat{

        FORMAT_7EVEN1(0, DataBits.SEVEN, Parity.EVEN, StopBits.ONE),
	FORMAT_7EVEN2(1, DataBits.SEVEN, Parity.EVEN, StopBits.TWO),
	FORMAT_7ODD1 (2, DataBits.SEVEN, Parity.ODD,  StopBits.ONE),
	FORMAT_7ODD2 (3, DataBits.SEVEN, Parity.ODD,  StopBits.TWO),
	FORMAT_7NONE1(4, DataBits.SEVEN, Parity.NONE, StopBits.ONE),
	FORMAT_7NONE2(5, DataBits.SEVEN, Parity.NONE, StopBits.TWO),
	FORMAT_8EVEN1(6, DataBits.EIGHT, Parity.EVEN, StopBits.ONE),
	FORMAT_8ODD1 (7, DataBits.EIGHT, Parity.ODD,  StopBits.ONE),
	FORMAT_8NONE1(8, DataBits.EIGHT, Parity.NONE, StopBits.ONE),
	FORMAT_8NONE2(9, DataBits.EIGHT, Parity.NONE, StopBits.ONE);

        private int code;
        private DataBits dbits;
        private Parity parity;
        private StopBits sbits;
        
        SerialFormat(int code, DataBits dbits, Parity parity, StopBits sbits) {
            this.code = code;       // Code used by IV251 for format
            this.dbits = dbits;
            this.parity = parity;
            this.sbits = sbits;
        }

        /* Method to create format code needed in open methods  */

	int getOpenFormat() {
	    return makeDataCharacteristics(dbits, sbits, parity, FlowCtrl.NONE);
	}

        /**
         *  Method to decode (to an enum) the meaning of an IV251 format code
         */

        public static SerialFormat decode(int code) throws DriverException {
            SerialFormat[] form = SerialFormat.values();
            int n = form.length;
            int found = -1;
            for (int i = 0; i < n; i++) {
                if (form[i].code == code) {
                    found = i;
                    break;
                }
            }
            if (found == -1) throw new DriverException(Lbl + "invalid format code");
            return form[found];
        }
    }

   /**
    *  Data members
    */

    private final int TIMEOUT = 3000;       // milliseconds
    private String lastCmndText;  // save Cmnd info for timeout exception
    boolean debug = false;   // Flag to print hex content of ascii buffers
    long delay = 6;          // ms delay after a command is complete

    /* Defaults for serial connection */
    private final int BAUD_DEFAULT = 9600;  
    private final int FORMAT_DEFAULT = SerialFormat.FORMAT_7EVEN1.getOpenFormat();

    /* Default unit address = 11 */
    private final String ADDR = "11"; 

    /* Expected length of controller response for Write (acknowledgement) */
    private static final int ACK_RESP_LEN = 1;  // bytes

    /* Response length in case of an invalid register in command */
    private static final int INV_RESP_LEN = 4;

    /* Minimum response length for valid Read (one digit of data) */
    private static final int MIN_READ_LEN = 6;

    /* Maximum response length for valid Read (ten digits of data) */
    private static final int MAX_READ_LEN = 15;

    /* Offset in bytes to data field in successful read responses */
    private static final int DATA_OFFSET = 3;

    /* Number of bytes other than data field in successful read resposne */
    private static final int NON_DATA_LEN = 5;

    /* Constant bytes in messages to and from controller */
    private static final byte STX = 0x02;
    private static final byte ETX = 0x03;
    private static final byte EOT = 0x04;
    private static final byte ENQ = 0x05;
    private static final byte ACK = 0x06;
    private static final byte NAK = 0x15;

    /* Label identifying source of messages */
    private static final String Lbl = "MotronaIV251:  ";

    /* Save information needed to decode error bit */
    private int errorBitLocation;
    private int errorBitPolarity;

   /**
    *  Constructor.
    */

    public MotronaIV251() {
        super(Option.NO_NET);   // Disallow network connections
    }

    /**
     *  Override underlying open method of Ascii class
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate  The baud rate, or 0 for the default (9600)
     *  @param  commParm  The communications parameters or -1 for detault
     *
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int baudRate, 
                     int commParm) throws DriverException
    {
        super.open(connType, ident, baudRate == 0 ? BAUD_DEFAULT : baudRate,
                   commParm == -1 ? FORMAT_DEFAULT : commParm);
        setTimeout(TIMEOUT);

        /* Check validity of baudRate (throwsIllegal ArgumentException) */
        SerialRate rate = SerialRate.valueOf("BAUD_" +
                                             Integer.toString(baudRate));
        if (debug) {
            System.out.println("Opening connection with " + rate.toString());
        } 

        /*
         *   Initialize some controller settings
         */

        try {
	   /** It doesn't help to set baud rate, unit address or serial
            *  format as part of initialization, since if the values are 
            *  not already correct the commands will not work.
            */
            // setParam(CmndParam.SERIAL_BAUD_RATE, rate.code);
            // setParam(CmndParam.UNIT_NUMBER, Integer.parseInt(ADDR));
	    // setParam(CmndParam.SERIAL_FORMAT, 8);  // matches CommParm = 0

	    setParam(CmndParam.SERIAL_TIMER, 0);    // no cyclic transmission
            setParam(CmndParam.SERIAL_VALUE, 9);    // register if cyclic
            setParam(CmndParam.SERIAL_PROTOCOL, 0); // irrelevant if no cyclic
            activateData();
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }

    /**
     *  Open serial connection using default data characteristics
     *
     *  @param  serialName   Serial device name
     *  @throws DriverException
     */
    public void open(String serialName) throws DriverException {
        openSerial(serialName, BAUD_DEFAULT, FORMAT_DEFAULT);
    }

   /**
    *  Methods to send commands to controller and obtain response.
    *  Successful responses will be an acknowledgement for a Write,
    *  and returned data for a Read.  Failures should throw a
    *  DriverException or (in case of timeout) a DriverTimeoutException.
    */

   /** 
    *  Read parameter value
    *
    *  @param  cmnd  enumerated command identifier
    *  @return data  returned from controller
    *  @throws DriverException
    */

    public int readParam(CmndParam cmnd) throws DriverException {
        String cmndBody = cmnd.getRegister();
        String cmndText = "Read " +  cmnd.toString();
	byte[] response = respToRead(cmndBody, cmndText);
        String strdata = new String(response, DATA_OFFSET, 
                                    response.length - NON_DATA_LEN);
        return Integer.parseInt(strdata);
    }

   /** 
    *  Set parameter value (pending activate-data command
    *
    *  @param  cmnd   enumerated command identifier
    *  @param  value  integer data to write
    *  @throws DriverException
    */

    public void setParam(CmndParam cmnd, int value)  throws DriverException {
        if (value < cmnd.getMinValue() || value > cmnd.getMaxValue()) {
            throw new IllegalArgumentException(Lbl + String.format("Value out of range %d to %d", cmnd.getMinValue(), cmnd.getMaxValue()));
        }
        String cmndBody = cmnd.getRegister() + String.format("%d", value);
        String cmndText = "Set " +  cmnd.toString() + " " + 
	                  Integer.toString(value);
	respToWrite(cmndBody, cmndText);
        return;
    }

   /** 
    *  Set and activate parameter value
    *
    *  @param  cmnd   enumerated command identifier
    *  @param  value  integer data to write
    *  @throws DriverException
    */

    public void setAndActivateParam(CmndParam cmnd, int value)
     throws DriverException {
        setParam(cmnd, value);
        activateData();
        return;
    }

   /** 
    *  Set serial baud rate in Motrona converter (for use in Test program)
    *
    *  @param  value  requested baud rate
    *  @throws DriverException
    */

    void setBaudRate(int value) throws DriverException {
        SerialRate rate = SerialRate.valueOf("BAUD_" +
                                             Integer.toString(value));
        setParam(CmndParam.SERIAL_BAUD_RATE, rate.rateCode);
        return;
    }

   /** 
    *  Set serial format in Motrona converter (for use in Test program)
    *
    *  @param  format  requested format (enum)
    *  @throws DriverException
    */

    void setFormat(SerialFormat format) throws DriverException {
        setParam(CmndParam.SERIAL_FORMAT, format.code);
        return;
    }

   /** 
    *  Send activate-data command to converter.  (This activates any new
    *  parameter values since last activate command or load from EEProm.)
    *
    *  @throws DriverException
    */

    public void activateData() throws DriverException {
        String cmndBody = CmndOp.ACTIVATE_DATA.getRegister() + "1";
        String cmndText = "Operation ACTIVATE_DATA";
        respToWrite(cmndBody, cmndText);

        // Save values needed to decode error bit
        errorBitLocation = readParam(CmndParam.SSI_ERROR_BIT);
        errorBitPolarity = readParam(CmndParam.SSI_ERROR_POLARITY);

        return;
    }

   /** 
    *  Save active values of parameters to EEProm
    *
    *  @throws DriverException
    */

    public void saveToEEProm() throws DriverException {
        String cmndBody = CmndOp.SAVE_TO_EEPROM.getRegister() + "1";
        String cmndText = "Operation SAVE_TO_EEPROM";
        respToWrite(cmndBody, cmndText);
        return;
    }

   /** 
    *  Read linear-encoder position
    * 
    *  @return Value read
    *  @throws DriverException
    */

    public int readEncoder() throws DriverException {
        String cmndBody = CmndOp.POSITION.getRegister();
        String cmndText = "Read encoder position";
	byte[] response = respToRead(cmndBody, cmndText);
        String strdata = new String(response, DATA_OFFSET, 
                                    response.length - NON_DATA_LEN);
        return Integer.parseInt(strdata);
    }        

   /** 
    *  Read linear-encoder error bit
    * 
    *  @return 0 (no error) or 1 (error)
    *  @throws DriverException
    */

    public int readErrorBit() throws DriverException {
        String cmndBody = CmndOp.ERROR_BIT.getRegister();
        String cmndText = "Read encoder error bit";
	byte[] response = respToRead(cmndBody, cmndText);
        String strdata = new String(response, DATA_OFFSET, 
                                    response.length - NON_DATA_LEN);
        int errbit = (Integer.parseInt(strdata) >>> errorBitLocation) & 1;
        return ((errorBitPolarity == 1) ? errbit : 1 - errbit);
    }

   /**
    *  Send Write command to controller and get response.  Both the
    *  command and the response are byte arrays.  
    *
    *  @param  commandBody   Command String before converting to a byte 
    *                        array and including control characters, the 
    *                        device address and the BCC checksum.
    *  @param  cmndText      describes command (for text for an Exception)
    *  @return response
    *  @throws DriverException
    */

    private synchronized byte[] respToWrite(String commandBody, String cmndText)
     throws DriverException {

        /*  Create command byte array; send it to MotronaIV251 converter.
         *  Note that BCC checksum omits the first four bytes, as well
         *  as the last bytes (where the BCC itself will be stored).
         */

        byte[] command = new byte[commandBody.length() + 6];
        command[0] = EOT;
        System.arraycopy(ADDR.getBytes(), 0, command, 1, 2);
        command[3] = STX;
        System.arraycopy(commandBody.getBytes(), 0, command, 4, commandBody.length());
        command[command.length - 2] = ETX;
        command[command.length - 1] = computeBCC(command, 4, 1);
        flush();
        writeBytes(command);
        if (debug) {
            System.out.println("Write Command  " + cmndText);
            String sent = "  byte buffer sent:  ";
            for (int isent = 0; isent < command.length; isent++) {
                sent += String.format("%02x ",command[isent]);
	    }
            System.out.println(sent);
        }

        /*  Get response from controller ("false" denotes Write command)  */

        lastCmndText = cmndText;
        return read(false);
    }

   /**
    *  Send Read command to controller and get response.  Both the
    *  command and the response are byte arrays.  
    *
    *  @param  commandBody   Command String before converting to a byte 
    *                        array and including control characters, the 
    *                        device address and the BCC checksum.
    *  @param  cmndText      describes command (for text for an Exception)
    *  @return response
    *  @throws DriverException
    */

    private synchronized byte[] respToRead(String commandBody, String cmndText)
     throws DriverException {

        /*  Create command byte array; send it to MotronaIV251 converter. */

        byte[] command = new byte[commandBody.length() + 4];
        command[0] = EOT;
        System.arraycopy(ADDR.getBytes(), 0, command, 1, 2);
        System.arraycopy(commandBody.getBytes(), 0, command, 3, commandBody.length());
        command[command.length - 1] = ENQ;
        flush();
        writeBytes(command);
        if (debug) {
            System.out.println("Read Command  " + cmndText);
            String sent = "  byte buffer sent:  ";
            for (int isent = 0; isent < command.length; isent++) {
                sent += String.format("%02x ",command[isent]);
	    }
            System.out.println(sent);
        }

        /*  Get response from controller ("false" denotes Write command)  */

        lastCmndText = cmndText;
        return read(true);
    }

   /**
    *  Read response from controller
    * 
    *  @param  readCmnd  true/false if command to hardware is read/write
    *                    (determines possible expected response formats)
    *  @return response buffer (byte array)
    *  @throws DriverException
    */

    private byte[] read(boolean readCmnd) throws DriverException {

        byte[] buff = new byte[MAX_READ_LEN + 10];  // with extra space
        int maxExpectedLen = (readCmnd ? MAX_READ_LEN : INV_RESP_LEN);
        int curr = 0;   // index to next unread byte
        while (true) {
            try {
                curr += readBytes(buff, curr);
            }
            catch (DriverTimeoutException ete) {
                String rsp = " Cmnd = " + lastCmndText + ", response buffer = ";
                for (int irsp = 0; irsp < curr; irsp++) {
                    rsp += String.format("%02x ",buff[irsp]);
		}
                throw new DriverTimeoutException(ete + rsp);
            }
            if (debug) {
                String rsp = "  Response buffer: ";
                for (int irsp = 0; irsp < curr; irsp++) {
                    rsp += String.format("%02x ",buff[irsp]);
		}
                System.out.println(rsp);
            }
            if (curr > maxExpectedLen) {
                throw new DriverException(Lbl + String.format("response of %d bytes, max. expected = %d", curr-1, maxExpectedLen));
            }

            // Check for error response
            if (getIndex(buff, EOT) == INV_RESP_LEN - 1)  { 
	        throw new DriverException(Lbl + " invalid register " + String.format("0x%02x%02x", buff[1], buff[2]));
            }
            if (getIndex(buff, NAK) == ACK_RESP_LEN - 1)  {
	        throw new DriverException(Lbl + " error in read command");
            }

            // Check for valid response
            if (readCmnd) {                  // response to Read command
                // Check for successful response to a read command
                int indexETX = getIndex(buff, ETX);
                if (indexETX > -1) {
                    if (curr >= indexETX + 2) {
                        byte cBCC = computeBCC(buff, 1, buff.length-indexETX-1);
			if (buff[indexETX+1] != cBCC) {
                            if (debug) System.out.println(String.format("  received BCC = %02x, excpected = %02x", buff[indexETX+1], cBCC));
			    throw new DriverException(Lbl + " checksum incorrect in read-response");
			}
                        break;    // completed successful response
		    } else if (indexETX < MIN_READ_LEN - 2) {
                        throw new DriverException(Lbl + String.format(" read-response too short (ETX found at byte %d", indexETX));
                    }
		}
            } else {                        // response to Write command
                if (getIndex(buff, ACK) == ACK_RESP_LEN - 1) break;
            }
	}
        byte[] response = new byte[curr];
        System.arraycopy(buff,0,response,0,curr);
        // impose delay to protect against next command coming too soon
        if (delay > 0) {
            try {
                Thread.sleep(delay);
            } catch (InterruptedException ex) {
                throw new RuntimeException("Interrupt during delay after MotronaIV251 command",ex);
            }
        }
        return response;
    }

    /**
     *  Find index of first occurrence of specified byte in an array.
     *
     *  @param array      byte array
     *  @param xxx        byte to search for
     *
     *  return            index of byte if found, -1 if not found
     */

    private int getIndex(byte[] array, byte xxx) {

	int index = -1;
        if (array.length > 0) {
            for (int i = 0; i < array.length; i++) {
                if (array[i] == xxx) {
                    index = i;
                    break;
                }
            }
        }
        return index;
    }

    /**
     *  Compute BCC "block check character" checksum (exclusive or) for 
     *  byte array.  
     *
     *  @param array      Byte array
     *  @param omitFirst  Number of bytes at start to omit.
     *  @param omitLast   Number of bytes at end to omit.
     *
     *  @return           BCC
     */

    private byte computeBCC(byte[] array, int omitFirst, int omitLast) {

        byte BCC = 0;
        if (array.length - omitFirst - omitLast >0) {
            for (int i = omitFirst; i < array.length - omitLast; i++) {
                BCC ^= array[i];
            }
        }
        return BCC;
    }
}
