package org.lsst.ccs.drivers.modbus;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *  Communicates via the Modbus protocol on a serial line or network socket
 *
 *  @author Owen Saxton
 */
public class Modbus extends Ascii {

    /**
     *  Private & package data.
     */
    public enum ConnType { NET, SERIAL, ASERIAL }

    private static final Map<ConnType, Ascii.ConnType> connTypeMap = new HashMap<>();
    static {
        connTypeMap.put(ConnType.NET, Ascii.ConnType.NET);
        connTypeMap.put(ConnType.SERIAL, Ascii.ConnType.SERIAL);
        connTypeMap.put(ConnType.ASERIAL, Ascii.ConnType.SERIAL);
    }
    private static final Map<Ascii.ConnType, ConnType> revTypeMap = new HashMap<>();
    static {
        revTypeMap.put(Ascii.ConnType.NET, ConnType.NET);
        revTypeMap.put(Ascii.ConnType.SERIAL, ConnType.SERIAL);
    }
    
    static final byte
        FUNC_READ_COILS      = 1,
        FUNC_READ_DISCRETES  = 2,
        FUNC_READ_REGISTERS  = 3,
        FUNC_READ_INPUTS     = 4,
        FUNC_WRITE_COIL      = 5,
        FUNC_WRITE_REGISTER  = 6,
        FUNC_WRITE_COILS     = 15,
        FUNC_WRITE_REGISTERS = 16;

    static final int
        OFF_TID         = 0,
        OFF_PID         = 2,
        OFF_LENGTH      = 4,
        OFF_UID         = 6,
        OFF_FUNC        = 7,
        OFF_RLENG       = 8,
        OFF_ERROR       = 8,
        OFF_ADDR        = 8,
        OFF_RDATA       = 9,
        OFF_COUNT       = 10,
        OFF_DATA        = 12,
        LENG_HEADER     = 6,
        LENG_CRC        = 2;

    static final int
        EXCP_ILL_FUNC   = 1,
        EXCP_ILL_ADDR   = 2,
        EXCP_ILL_VALUE  = 3,
        EXCP_SRVR_FAIL  = 4,
        EXCP_SRVR_BUSY  = 6;

    private static final Map<Integer, String> stdExcpMap = new HashMap<>();
    static {
        stdExcpMap.put(EXCP_ILL_ADDR, "illegal data address");
        stdExcpMap.put(EXCP_ILL_FUNC, "illegal function");
        stdExcpMap.put(EXCP_ILL_VALUE, "illegal data value");
        stdExcpMap.put(EXCP_SRVR_BUSY, "server device busy");
        stdExcpMap.put(EXCP_SRVR_FAIL, "server device failure");
    }

    static final int
        DEFAULT_PORT_NUMBER  = 502,
        DEFAULT_BAUD_RATE    = 115200;

    private static final double
        READ_TIMEOUT    = 2,
        PACKET_INTERVAL = 0.1;

    private ConnType connType;
    private int nmbrOff = 1;
    private short tranId = 0;
    private final Map<Integer, String> addExcpMap = new HashMap<>();
    private byte[] resp = new byte[1024];


    /**
     *  Opens a connection to the Modbus.
     *
     *  @param  type   The type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   port name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for serial
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   data characteristics for serial
     *  @throws  DriverException
     */
    public synchronized void open(ConnType type, String ident, int parm1, int parm2) throws DriverException
    {
        super.open(connTypeMap.get(type), ident, parm1, parm2);
        connType = type;
        setTimeout(READ_TIMEOUT);
    }


    /**
     *  Opens a connection to the Modbus.
     *
     *  @param  type   The type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   port name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for serial
     *  @throws  DriverException
     */
    public synchronized void open(ConnType type, String ident, int parm1) throws DriverException
    {
        open(type, ident, parm1, 0);
    }


    /**
     *  Opens a connection to the Modbus.
     *
     *  @param  type   The type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   port name for serial
     *  @throws  DriverException
     */
    public void open(ConnType type, String ident) throws DriverException
    {
        open(type, ident, type == ConnType.NET ? DEFAULT_PORT_NUMBER : DEFAULT_BAUD_RATE);
    }


    /**
     *  Opens a connection to the Modbus.
     *
     *  This overrides Ascii.open, so doesn't support Ascii Modbus
     *
     *  @param  type   The type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   port name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for serial
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   data characteristics for serial
     *  @throws  DriverException
     */
    @Override
    public synchronized void open(Ascii.ConnType type, String ident, int parm1, int parm2) throws DriverException
    {
        super.open(type, ident, parm1, parm2);
        connType = revTypeMap.get(type);
        setTimeout(READ_TIMEOUT);
    }


    /**
     *  Sets whether entity numbers are direct addresses.
     *
     *  The usual Modbus convention is that holding register numbers, etc,
     *  start from 1, so that 1 has to be subtracted in order to generate the
     *  actual address.  However, some devices provide the direct address in
     *  their documentation.
     *
     *  @param  mode  Whether or not address mode is to be set
     */
    public void setAddressMode(boolean mode)
    {
        nmbrOff = mode ? 0 : 1;
    }


    /**
     *  Returns whether entity numbers are direct addresses.
     *
     *  @return  Whether or not address mode is set
     */
    public boolean isAddressMode()
    {
        return nmbrOff == 0;
    }


    /**
     *  Sets the additional exception description map.
     *
     *  @param  map  The map of additional codes to descriptions
     */
    public void setExceptionMap(Map<Integer, String> map)
    {
        addExcpMap.putAll(map);
    }


    /**
     *  Reads a set of coils
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  cNmbr  The number of the first coil to read
     *  @param  count  The number of coils to read
     *  @return  The array of read coil values, one per bit
     *  @throws DriverException
     */
    public synchronized byte[] readCoils(short dAddr, short cNmbr, short count) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(cNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_COILS, cmnd);
        int nbyte = resp[OFF_RLENG] & 0xff;
        byte[] value = new byte[nbyte];
        System.arraycopy(resp, OFF_RDATA, value, 0, nbyte);
        return value;
    }


    /**
     *  Reads a set of discrete inputs
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  iNmbr  The number of the first input to read
     *  @param  count  The number of inputs to read
     *  @return  The array of read discrete values, one per bit
     *  @throws DriverException
     */
    public synchronized byte[] readDiscretes(short dAddr, short iNmbr, short count) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(iNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_DISCRETES, cmnd);
        int nbyte = resp[OFF_RLENG] & 0xff;
        byte[] value = new byte[nbyte];
        System.arraycopy(resp, OFF_RDATA, value, 0, nbyte);
        return value;
    }


    /**
     *  Reads a set of holding registers
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  rNmbr  The number of the first register to read
     *  @param  count  The number of registers to read
     *  @return  The array of read register values
     *  @throws DriverException
     */
    public synchronized short[] readRegisters(short dAddr, short rNmbr, short count) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_REGISTERS, cmnd);
        int nvalue = (resp[OFF_RLENG] & 0xff) / 2;
        short[] value = new short[nvalue];
        for (int j = 0; j < nvalue; j++) {
            value[j] = Convert.bytesToShortBE(resp, 2 * j + OFF_RDATA);
        }
        return value;
    }


    /**
     *  Reads a set of input registers
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  rNmbr  The number of the first register to read
     *  @param  count  The number of registers to read
     *  @return  The array of read register values
     *  @throws DriverException
     */
    public synchronized short[] readInputs(short dAddr, short rNmbr, short count) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_INPUTS, cmnd);
        int nvalue = (resp[OFF_RLENG] & 0xff) / 2;
        short[] value = new short[nvalue];
        for (int j = 0; j < nvalue; j++) {
            value[j] = Convert.bytesToShortBE(resp, 2 * j + OFF_RDATA);
        }
        return value;
    }


    /**
     *  Writes a holding register
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  rNmbr  The number of the register to write
     *  @param  value  The (16-bit) value to write
     *  @throws DriverException
     */
    public synchronized void writeRegister(short dAddr, short rNmbr, short value) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(value, cmnd, OFF_COUNT);
        send(dAddr, FUNC_WRITE_REGISTER, cmnd);
    }


    /**
     *  Writes a coil
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  cNmbr  The number of the coil to write
     *  @param  value  The value (false = off, true = on) to write
     *  @throws DriverException
     */
    public synchronized void writeCoil(short dAddr, short cNmbr, boolean value) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(cNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE((short)(value ? 0xff00 : 0), cmnd, OFF_COUNT);
        send(dAddr, FUNC_WRITE_COIL, cmnd);
    }


    /**
     *  Writes a set of holding registers
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  rNmbr  The number of the first register to write
     *  @param  value  The array of register values to write
     *  @throws DriverException
     */
    public synchronized void writeRegisters(short dAddr, short rNmbr, short[] value) throws DriverException
    {
        int count = value.length;
        byte[] cmnd = new byte[2 * count + 1 + OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE((short)count, cmnd, OFF_COUNT);
        cmnd[OFF_DATA] = (byte)(2 * count);
        for (int j = 0; j < count; j++) {
            Convert.shortToBytesBE(value[j], cmnd, 2 * j + 1 + OFF_DATA);
        }
        send(dAddr, FUNC_WRITE_REGISTERS, cmnd);
    }


    /**
     *  Writes a set of coils
     *
     *  @param  dAddr  The Modbus address of the device
     *  @param  cNmbr  The number of the first coil to write
     *  @param  count  The number of coil values to write
     *  @param  value  The array of coil values to write
     *  @throws  DriverException
     */
    public synchronized void writeCoils(short dAddr, short cNmbr, short count, byte[] value) throws DriverException
    {
        int nbyte = (count + 7) / 8;
        byte[] cmnd = new byte[nbyte + 1 + OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(cNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        cmnd[OFF_DATA] = (byte)nbyte;
        System.arraycopy(value, 0, cmnd, OFF_DATA + 1, nbyte);
        send(dAddr, FUNC_WRITE_COILS, cmnd);
    }


    /**
     *  Sends a command to a device on the Modbus and returns its response
     *
     *  @param  addr     The destination Modbus address
     *  @param  func     The function code
     *  @param  cmnd     The command byte array to send.
     *  @return  The number of bytes read
     *  @throws  DriverException
     */
    private int send(short addr, byte func, byte[] cmnd) throws DriverException
    {
        if (!isOpen()) {
            throw new DriverException("Device not connected");
        }
        Convert.shortToBytesBE(tranId++, cmnd, OFF_TID);
        cmnd[OFF_UID] = (byte)addr;
        cmnd[OFF_FUNC] = func;
        writeCmnd(cmnd);
        int leng = readResp();
        if ((resp[OFF_UID] & 0xff) != addr) {
            throw new DriverException("Invalid reply address");
        }
        if ((resp[OFF_FUNC] & 0x7f) != func) {
            throw new DriverException("Invalid reply function code");
        }
        if ((resp[OFF_FUNC] & 0x80) != 0) {
            int code = resp[OFF_ERROR];
            String message = addExcpMap.get(code);
            if (message == null) {
                message = stdExcpMap.get(code);
            }
            if (message == null) {
                message = "unknown code (" + code + ")";
            }
            throw new DriverException("Reply exception: " + message);
        }

        return leng;
    }


    /**
     *  Writes a command.
     *
     *  @param  command  The command to write
     *  @throws  DriverException
     */
    private void writeCmnd(byte[] command) throws DriverException
    {
        int leng;
        switch (connType) {
        case NET:
            Convert.shortToBytesBE((short)0, command, OFF_TID);
            Convert.shortToBytesBE((short)0, command, OFF_PID);
            leng = command.length - LENG_HEADER - LENG_CRC;
            Convert.shortToBytesBE((short)leng, command, OFF_LENGTH);
            writeBytes(command, 0, command.length - LENG_CRC);
            break;

        case SERIAL:
            leng = command.length;
            CRC16.generateStd(command, LENG_HEADER, leng - LENG_HEADER - LENG_CRC,
                              command, leng - LENG_CRC);
            writeBytes(command, LENG_HEADER, leng - LENG_HEADER);
            break;

        case ASERIAL:
            leng = command.length - LENG_CRC;
            command[leng] = generateLRC(command, LENG_HEADER, leng - LENG_HEADER);
            write(bytesToHex(command, LENG_HEADER, leng - LENG_HEADER + 1));
            break;
        }
    }


    /**
     *  Reads response data.
     *
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    private int readResp() throws DriverException
    {
        int posn = 0;
        switch (connType) {
        case NET:
            int rqstLeng = LENG_HEADER;
            while (posn < rqstLeng) {
                posn += readBytes(resp, posn, rqstLeng - posn);
            }
            rqstLeng += Convert.bytesToShortBE(resp, OFF_LENGTH);
            if (rqstLeng > resp.length) {
                resp = Arrays.copyOf(resp, rqstLeng);
            }
            while (posn < rqstLeng) {
                posn += readBytes(resp, posn, rqstLeng - posn);
            }
            break;

        case SERIAL:
            setTimeout(READ_TIMEOUT);
            posn = LENG_HEADER;
            try {
                posn += readBytes(resp, posn);
                setTimeout(PACKET_INTERVAL);
                while (true) {
                    while (posn < resp.length) {
                        posn += readBytes(resp, posn);
                    }
                    resp = Arrays.copyOf(resp, 2 * resp.length);
                }
            }
            catch (DriverTimeoutException e) {
                if (posn == LENG_HEADER) {
                    throw e;
                }
            }
            if (CRC16.generateStd(resp, LENG_HEADER, posn - LENG_HEADER) != 0) {
                throw new DriverException("Reply checksum error");
            }
            break;

        case ASERIAL:
            //TODO add receive code here
            break;
        }
        return posn;
    }


    /**
     *  Generates LRC.
     */
    private static byte generateLRC(byte[] data, int offset, int length)
    {
        int value = 0;
        for (int j = offset; j < offset + length; j++) {
            value += data[j];
        }
        return (byte)(-value);
    }


    /**
     *  Converts binary array to hexadecimal string.
     */
    private static String bytesToHex(byte[] data, int offset, int length)
    {
        StringBuilder str = new StringBuilder(":");
        for (int j = offset; j < offset + length; j++) {
            str.append(Integer.toHexString(data[j]).toUpperCase());
        }
        return str.toString();
    }


    /**
     *  Converts hexadecimal string to binary array.
     */
    private static byte hexToBytes(String str, byte[] data, int offset, int length)
    {
        int value = 0;
        for (int j = offset; j < offset + length; j++) {
            value += data[j];
        }
        return (byte)(-value);
    }

}
