package org.lsst.ccs.drivers.modbus;

/**
 *  Routines for generating 16-bit cyclic redundancy checksums
 *
 *  @author Owen Saxton
 */
public class CRC16 { 

    private static final int stdPoly = 0xa001, TABLE_SIZE = 256;
    private static final int[] stdTable = new int[TABLE_SIZE];
    static {
        makeTable(stdPoly, stdTable);
    }

    private final int[] table = new int[TABLE_SIZE];


    /**
     *  Constructor
     */
    public CRC16(int poly)
    {
        makeTable(poly, table);
    }

    
    /**
     *  Generates the custom CRC for some data
     */
    public int generate(byte[] data)
    {
        return generateCRC(table, data, 0, data.length);
    }

    public int generate(byte[] data, int offset, int length)
    {
        return generateCRC(table, data, offset, length);
    }

    public void generate(byte[] data, byte[] output, int outoff)
    {
        generate(data, 0, data.length, output, outoff);
    }

    public void generate(byte[] data, int offset, int length,
                         byte[] output, int outoff)
    {
        int crc = generateCRC(table, data, offset, length);
        output[outoff] = (byte)crc;
        output[outoff + 1] = (byte)(crc >> 8);
    }


    /**
     *  Generates the standard CRC for some data
     */
    public static int generateStd(byte[] data)
    {
        return generateCRC(stdTable, data, 0, data.length);
    }

    public static int generateStd(byte[] data, int offset, int length)
    {
        return generateCRC(stdTable, data, offset, length);
    }

    public static void generateStd(byte[] data, byte[] output, int outoff)
    {
        generateStd(data, 0, data.length, output, outoff);
    }

    public static void generateStd(byte[] data, int offset, int length,
                                   byte[] output, int outoff)
    {
        int crc = generateCRC(stdTable, data, offset, length);
        output[outoff] = (byte)crc;
        output[outoff + 1] = (byte)(crc >> 8);
    }


    /**
     *  Makes a lookup table for a given polynomial
     */
    private static void makeTable(int poly, int[] table)
    {
        for (int j = 0; j < table.length; j++) {
            int crc = j;
            for (int k = 0; k < 8; k++) {
                crc = (crc >>> 1) ^ ((crc & 1) == 0 ? 0 : poly);
            }
            table[j] = crc;
        }
    }


    /**
     *  Calculates the CRC
     */
    private static int generateCRC(int[] table, byte[] data, int offset,
                                   int length)
    {
        int crc = 0xffff;
        for (int j = 0; j < length; j++) {
            crc = (crc >>> 8) ^ table[(crc ^ data[offset + j]) & 0xff];
        }
        return crc;
    }


    /**
     *  Test program
     */
    public static void main(String[] args) { 
        if (args.length == 0) {
            System.out.println("No argument supplied");
            System.exit(0);
        }
        String arg = args[0];
        int leng = arg.length();
        if ((leng & 1) != 0) {
            System.out.println("Argument must have an even number of digits");
            System.exit(0);
        }
        byte[] data = new byte[leng / 2];
        for (int j = 0; j < leng; j += 2) {
            try {
                int value = Integer.decode("0x" + arg.substring(j, j + 2));
                data[j / 2] = (byte)value;
            }
            catch (NumberFormatException e) {
                System.out.println("Invalid hexadecimal digit found");
                System.exit(0);
            }
        }
        System.out.format("CRC16 = %04x\n", CRC16.generateStd(data));
    }

}
