package org.lsst.ccs.drivers.modbus;

import java.io.IOException;
import java.io.PrintStream;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test a Modbus device
 *
 *  @author Owen Saxton
 */
public class TestModbus {

    /**
     *  Inner class for detecting console input while reading continuously.
     */
    class ConsThread extends Thread {

        private final BlockingQueue<Integer> consQueue = new ArrayBlockingQueue<>(1);
        private boolean[] consDone;

        ConsThread() {
            super();
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                awaitStart();
                awaitTerminal();
                consDone[0] = true;
            }
        }

        public void start(boolean[] done) {
            if (getState() == Thread.State.NEW) {
                start();
            }
            consDone = done;
            consDone[0] = false;
            consQueue.offer(0);
        }

        private void awaitStart() {
            while (true) {
                try {
                    consQueue.take();
                    return;
                }
                catch (InterruptedException e) {
                }
            }
        }

        private void awaitTerminal() {
            while (true) {
                try {
                    if (System.in.available() > 0) {
                        break;
                    }
                    try {
                        Thread.sleep(50);
                    }
                    catch (InterruptedException e) {
                    }
                }
                catch (IOException e) {
                    break;
                }
            }
            try {
                while (System.in.available() > 0) {
                    System.in.read();
                }
            }
            catch (IOException e) {
            }
        }

    }

    protected static enum OnOff { ON, OFF }

    protected final PrintStream out = System.out;
    protected final Modbus mod;
    private final ConsThread consThread = new ConsThread();
    private boolean hexMode;
    private int maxRead = 16;


    public TestModbus()
    {
        mod = new Modbus();
    }


    public TestModbus(Modbus mbus)
    {
        mod = mbus;
    }


    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type", description="Connection type") Modbus.ConnType type,
                     @Argument(name="ident", description="Device identifier") String ident) throws DriverException
    {
        mod.open(type, ident);
    }


    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type", description="Connection type") Modbus.ConnType type,
                     @Argument(name="ident", description="Device identifier") String ident,
                     @Argument(name="param", description="Device parameter") int param) throws DriverException
    {
        mod.open(type, ident, param);
    }


    @Command(name="open", description="Open connection to serial device")
    public void open(@Argument(description="Connection type: net, serial or ftdi") Modbus.ConnType type,
                     @Argument(description="Device identifier") String ident,
                     @Argument(description="Baud rate") int baud,
                     @Argument(description="The number of data bits") Modbus.DataBits dbits,
                     @Argument(description="The number of stop bits") Modbus.StopBits sbits,
                     @Argument(description="The parity") Modbus.Parity parity,
                     @Argument(description="The flow control") Modbus.FlowCtrl flow) throws DriverException
    {
        mod.open(type, ident, baud, Modbus.makeDataCharacteristics(dbits, sbits, parity, flow));
    }


    @Command(name="close", description="Close connection to device")
    public void close() throws DriverException
    {
        mod.close();
    }


    @Command(name="setaddress", description="Set the address mode")
    public void setAddress(@Argument(description="The address mode: off or on") OnOff mode) throws DriverException
    {
        mod.setAddressMode(mode == OnOff.ON);
    }


    @Command(name="showaddress", description="Show the address mode")
    public void showAddress() throws DriverException
    {
        out.println("Address mode is " + (mod.isAddressMode() ? "on" : "off"));
    }


    @Command(name="sethex", description="Set the hex diaplay mode")
    public void setHex(@Argument(description="The hex mode: off or on") OnOff mode)
    {
        hexMode = mode == OnOff.ON;
    }


    @Command(name="showhex", description="Show the hex display mode")
    public void showHex()
    {
        out.println("Hex display mode is " + (hexMode ? "on" : "off"));
    }


    @Command(name="setmaxread", description="Set the maximum device read count")
    public void setMaxRead(@Argument(description="The maximum read count") int count)
    {
        maxRead = count;
    }


    @Command(name="showmaxread", description="Show the maximum device read count")
    public void showMaxRead()
    {
        out.println("Maximum read count is " + maxRead);
    }


    @Command(name="settimeout", description="Set the timeout")
    public void setTimeout(@Argument(description="The timeout (sec)") double timeout) throws DriverException
    {
        mod.setTimeout(timeout);
    }


    @Command(name="coilread", description="Read a set of coils")
    public void coilRead(@Argument(description="The Modbus address") int address,
                         @Argument(description="First coil to read") int number,
                         @Argument(description="Number of coils to read") int count) throws DriverException
    {
        byte[] reply = mod.readCoils((short)address, (short)number, (short)count);
        int nvalue = count <= 8 * reply.length ? count : 8 * reply.length;
        for (int j = 0; j < nvalue; j++) {
            if ((j % 20) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", address + j);
            }
            out.format("%2s", (reply[j >> 3] & (1 << (j & 7))) != 0 ? 1 : 0);
        }
        out.println();
    }


    @Command(name="discread", description="Read a set of discretes")
    public void discRead(@Argument(description="The Modbus address") int address,
                         @Argument(description="First discrete to read") int number,
                         @Argument(description="Number of discretes to read") int count) throws DriverException
    {
        byte[] reply = mod.readDiscretes((short)address, (short)number, (short)count);
        int nvalue = count <= 8 * reply.length ? count : 8 * reply.length;
        for (int j = 0; j < nvalue; j++) {
            if ((j % 20) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", number + j);
            }
            out.format("%2s", (reply[j >> 3] & (1 << (j & 7))) != 0 ? 1 : 0);
        }
        out.println();
    }


    @Command(name="regread", description="Read a set of registers")
    public void regRead(@Argument(description="The Modbus address") int address,
                        @Argument(description="First register to read") int number,
                        @Argument(description="Number of registers to read") int count) throws DriverException
    {
        int lineSize = hexMode ? 8 : 10;
        String addrFmt = hexMode ? "%04x:" : "%4s:";
        int nRead = 0;
        try {
            while (nRead < count) {
                short[] reply = mod.readRegisters((short)address, (short)number, (short)Math.min(count - nRead, maxRead));
                for (int j = 0; j < reply.length; j++, nRead++, number++) {
                    if ((nRead % lineSize) == 0) {
                        if (nRead != 0) {
                            out.println();
                        }
                        out.format(addrFmt, number);
                    }
                    out.format(" %04x", reply[j]);
                }
            }
        }
        finally {
            if (nRead != 0) {
                out.println();
            }
        }
    }


    @Command(name="regmap", description="Read a set of registers, mapping their existence")
    public void regMap(@Argument(description="The Modbus address") int address,
                       @Argument(description="First register to read") int number,
                       @Argument(description="Number of registers to read") int count) throws DriverException
    {
        int lineSize = hexMode ? 8 : 10;
        String addrFmt = hexMode ? "%04x:" : "%4s:";
        for (int nRead = 0; nRead < count; nRead++, number++) {
            String value;
            try {
                short[] reply = mod.readRegisters((short)address, (short)number, (short)1);
                value = String.format(" %04x", reply[0]);
            }
            catch (DriverException e) {
                value = " ....";
            }
            if ((nRead % lineSize) == 0) {
                if (nRead != 0) {
                    out.println();
                }
                out.format(addrFmt, number);
            }
            out.print(value);
        }
        out.println();
    }


    @Command(name="inpread", description="Read a set of inputs")
    public void inpRead(@Argument(description="The Modbus address") int address,
                        @Argument(description="First input to read") int number,
                        @Argument(description="Number of inputs to read") int count) throws DriverException
    {
        short[] reply = mod.readInputs((short)address, (short)number, (short)count);
        int lineSize = hexMode ? 8 : 10;
        String addrFmt = hexMode ? "%04x:" : "%4s:";
        for (int j = 0; j < reply.length; j++) {
            if ((j % lineSize) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format(addrFmt, number + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


    @Command(name="coilwrite", description="Write a set of coils")
    public void coilWrite(@Argument(description="The Modbus address") int address,
                          @Argument(description="First coil to write") int number,
                          @Argument(description="Values to write") int... values) throws DriverException
    {
        short count = (short)values.length;
        if (count == 0) {
            throw new DriverException("No values supplied");
        }
        if (count == 1) {
            mod.writeCoil((short)address, (short)number, values[0] != 0);
        }
        else {
            byte[] coils = new byte[(count + 7) / 8];
            int value = 0, index = 0;
            for (int j = 0; j < count; j++) {
                if ((j & 7) == 0 && j > 0) {
                    coils[index++] = (byte)value;
                    value = 0;
                }
                if (values[j] != 0) {
                    value |= 1 << (j & 7);
                }
            }
            coils[index] = (byte)value;
            mod.writeCoils((short)address, (short)number, count, coils);
        }
    }


    @Command(name="regwrite", description="Write a set of registers")
    public void regWrite(@Argument(description="The Modbus address") int address,
                         @Argument(description="First register to write") int number,
                         @Argument(description="Values to write") int... values) throws DriverException
    {
        short count = (short)values.length;
        if (count == 0) {
            throw new DriverException("No values supplied");
        }
        if (count == 1) {
            mod.writeRegister((short)address, (short)number, (short)values[0]);
        }
        else {
            short[] sValues = new short[count];
            for (int j = 0; j < count; j++) {
                sValues[j] = (short)values[j];
            }
            mod.writeRegisters((short)address, (short)number, sValues);
        }
    }


    @Command(name="contregread", description="Continuously read a set of registers")
    public void contRegRead(@Argument(description="The Modbus address") int address,
                            @Argument(description="First register to read") int number,
                            @Argument(description="Number of registers to read") int count) throws DriverException
    {
        System.out.println("Press any key to terminate...");
        boolean[] done = {false};
        consThread.start(done);
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        while (!done[0]) {
            try {
                mod.readRegisters((short)address, (short)number, (short)count);
                nRead += count;
            }
            catch (DriverException e) {
                out.println("Terminated by exception: " + e.getMessage());
                break;
            }
        }
        double rate = (1000.0 * nRead) / (System.currentTimeMillis() - startTime);
        out.format("Read rate (Hz) = %.4g\n", rate);
    }

}
