package org.lsst.ccs.drivers.modbus;

import java.io.PrintStream;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ***************************************************************************
 **
 **  Program to test a Modbus device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestModbus {

    protected final PrintStream out = System.out;
    protected final Modbus mod;

    protected static enum ConnType {

        NET(Modbus.CONN_TYPE_NETWORK),
        FTDI(Modbus.CONN_TYPE_FTDI);

        int value;

        ConnType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    protected static enum OnOff {

        ON, OFF

    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor.
    **
    *************************************************************************** 
    */
    public TestModbus()
    {
        mod = new Modbus();
    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor.
    **
    **  @param  mbus  The Modbus object
    **
    *************************************************************************** 
    */
    public TestModbus(Modbus mbus)
    {
        mod = mbus;
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    **
    **  @param  type   The connection type
    **
    **  @param  ident  The device identification
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type",
                               description="Connection type: net, serial or ftdi")
                     ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident) throws DriverException
    {
        mod.open(type.getValue(), ident);
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    **
    **  @param  type   The connection type
    **
    **  @param  ident  The device identification
    **
    **  @param  param  The device parameter
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type",
                               description="Connection type: net or ftdi")
                     ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="param", description="Device parameter")
                     int param) throws DriverException
    {
        mod.open(type.getValue(), ident, param);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes connection to a device.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="close", description="Close connection to device")
    public void close() throws DriverException
    {
        mod.close();
    }


   /**
    ***************************************************************************
    **
    **  Sets the address mode.
    **
    ***************************************************************************
    */
    @Command(name="setaddress", description="Set the address mode")
    public void setAddress(@Argument(name="mode",
                                     description="The address mode: off or on")
                           OnOff mode) throws DriverException
    {
        mod.setAddressMode(mode == OnOff.ON);
    }


   /**
    ***************************************************************************
    **
    **  Shows the address mode.
    **
    ***************************************************************************
    */
    @Command(name="showaddress", description="Show the address mode")
    public void showAddress() throws DriverException
    {
        out.println("Address mode is " + (mod.isAddressMode() ? "on" : "off"));
    }


   /**
    ***************************************************************************
    **
    **  Sets the timeout.
    **
    ***************************************************************************
    */
    @Command(name="settimeout", description="Set the timeout")
    public void setTimeout(@Argument(name="timeout",
                                     description="The timeout (sec)")
                           double timeout) throws DriverException
    {
        mod.setTimeout(timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of coil values.
    **
    ***************************************************************************
    */
    @Command(name="coilread", description="Read a set of coils")
    public void coilRead(@Argument(name="address",
                                   description="The Modbus address")
                         int addr,
                         @Argument(name="number",
                                   description="First coil to read")
                         int nmbr,
                         @Argument(name="count",
                                   description="Number of coils to read")
                         int count) throws DriverException
    {
        byte[] reply = mod.readCoils((short)addr, (short)nmbr, (short)count);
        int nvalue = count <= 8 * reply.length ? count : 8 * reply.length;
        for (int j = 0; j < nvalue; j++) {
            if ((j % 20) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", addr + j);
            }
            out.format("%2s", (reply[j >> 3] & (1 << (j & 7))) != 0 ? 1 : 0);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of discrete values.
    **
    ***************************************************************************
    */
    @Command(name="discread", description="Read a set of discretes")
    public void discRead(@Argument(name="address",
                                   description="The Modbus address")
                         int addr,
                         @Argument(name="number",
                                   description="First discrete to read")
                         int nmbr,
                         @Argument(name="count",
                                   description="Number of discretes to read")
                         int count) throws DriverException
    {
        byte[] reply = mod.readDiscretes((short)addr, (short)nmbr, (short)count);
        int nvalue = count <= 8 * reply.length ? count : 8 * reply.length;
        for (int j = 0; j < nvalue; j++) {
            if ((j % 20) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", nmbr + j);
            }
            out.format("%2s", (reply[j >> 3] & (1 << (j & 7))) != 0 ? 1 : 0);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of register values.
    **
    ***************************************************************************
    */
    @Command(name="regread", description="Read a set of registers")
    public void regRead(@Argument(name="address",
                                  description="The Modbus address")
                        int addr,
                        @Argument(name="number",
                                  description="First register to read")
                        int nmbr,
                        @Argument(name="count",
                                  description="Number of registers to read")
                        int count) throws DriverException
    {
        short[] reply = mod.readRegisters((short)addr, (short)nmbr, (short)count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", nmbr + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of input values.
    **
    ***************************************************************************
    */
    @Command(name="inpread", description="Read a set of inputs")
    public void inpRead(@Argument(name="address",
                                  description="The Modbus address")
                        int addr,
                        @Argument(name="number",
                                  description="First input to read")
                        int nmbr,
                        @Argument(name="count",
                                  description="Number of inputs to read")
                        int count) throws DriverException
    {
        short[] reply = mod.readInputs((short)addr, (short)nmbr, (short)count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", nmbr + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of coils.
    **
    ***************************************************************************
    */
    @Command(name="coilwrite", description="Write a set of coils")
    public void coilWrite(@Argument(name="address",
                                   description="The Modbus address")
                          int addr,
                          @Argument(name="number",
                                    description="First coil to write")
                          int nmbr,
                          @Argument(name="values",
                                    description="Values to write")
                          int... values) throws DriverException
    {
        short count = (short)values.length;
        if (count == 0) {
            throw new DriverException("No values supplied");
        }
        if (count == 1) {
            mod.writeCoil((short)addr, (short)nmbr, values[0] != 0);
        }
        else {
            byte[] coils = new byte[(count + 7) / 8];
            int value = 0, index = 0;
            for (int j = 0; j < count; j++) {
                if ((j & 7) == 0 && j > 0) {
                    coils[index++] = (byte)value;
                    value = 0;
                }
                if (values[j] != 0) {
                    value |= 1 << (j & 7);
                }
            }
            coils[index] = (byte)value;
            mod.writeCoils((short)addr, (short)nmbr, count, coils);
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of registers.
    **
    ***************************************************************************
    */
    @Command(name="regwrite", description="Write a set of registers")
    public void regWrite(@Argument(name="address",
                                   description="The Modbus address")
                         int addr,
                         @Argument(name="number",
                                   description="First register to write")
                         int nmbr,
                         @Argument(name="values",
                                   description="Values to write")
                         int... values) throws DriverException
    {
        short count = (short)values.length;
        if (count == 0) {
            throw new DriverException("No values supplied");
        }
        if (count == 1) {
            mod.writeRegister((short)addr, (short)nmbr, (short)values[0]);
        }
        else {
            short[] sValues = new short[count];
            for (int j = 0; j < count; j++) {
                sValues[j] = (short)values[j];
            }
            mod.writeRegisters((short)addr, (short)nmbr, sValues);
        }
    }

}
