package org.lsst.ccs.drivers.modbus;

import java.io.PrintStream;
import java.util.concurrent.ArrayBlockingQueue;
import org.lsst.ccs.drivers.ftdi.Ftdi;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Handles Modbus RTU messages on a serial line
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class ModbusSlave {

    /**
     **  Private constants
     */
    private final static int
        NUM_COILS     = 96,
        NUM_DISCRETES = 16,
        NUM_REGISTERS = 200,
        NUM_INPUTS    = 10;

    private final static int
        PACKET_TIMEOUT = 50;

    /**
     **  Private fields
     */
    private final static PrintStream out = System.out;
    private final Reader reader = new Reader();
    private final Thread readMb = new Thread(reader);
    private Ftdi ftd = new Ftdi();
    private ArrayBlockingQueue replyQ = new ArrayBlockingQueue(1);
    private int busAddr;
    private byte[] coil = new byte[NUM_COILS >> 3];
    private byte[] discrete = new byte[NUM_DISCRETES >> 3];
    private short[] register = new short[NUM_REGISTERS];
    private short[] input = new short[NUM_INPUTS];

   /**
    ***************************************************************************
    **
    **  Implements the Modbus-reading thread
    **
    ***************************************************************************
    */
    private class Reader implements Runnable {

        @Override
        public void run()
        {
            byte[] buff = new byte[256];
            while (true) {
                try {
                    int leng = 0, nread;
                    ftd.setTimeouts(0, 0);
                    nread = ftd.read(buff, 0, 1);
                    leng += nread;
                    ftd.setTimeouts(PACKET_TIMEOUT, 0);
                    nread = ftd.read(buff, leng, buff.length - leng);
                    leng += nread;
                    byte[] data = new byte[leng];
                    System.arraycopy(buff, 0, data, 0, leng);
                    replyQ.offer(data);
                }
                catch (DriverException e) {
                    System.out.println(new DriverException(e.getMessage()));
                    break;
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public ModbusSlave(int index, String serial, String node)
        throws DriverException
    {
        for (int j = 0; j < coil.length; j++) {
            coil[j] = (byte)0xaa;
        }
        for (int j = 0; j < discrete.length; j++) {
            discrete[j] = 0x55;
        }
        for (short j = 0; j < register.length; j++) {
            register[j] = j;
        }
        for (short j = 0; j < input.length; j++) {
            input[j] = j;
        }

        try {
            ftd.open(node, index, serial);
            ftd.setBaudrate(Modbus.BAUD_RATE);
            ftd.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                       Ftdi.PARITY_NONE);
        }
        catch (DriverException e) {
            throw new DriverException(e.getMessage());
        }

        readMb.setDaemon(true);
        readMb.start();
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args) throws DriverException
    {
        int index = 0;
        if (args.length >= 1) {
            try {
                index = Integer.decode(args[0]);
            }
            catch (NumberFormatException e) {
                out.println("Invalid device index");
                System.exit(0);
            }
        }
        String serial = null;
        if (args.length >= 2) {
            serial = args[1];
        }
        int addr = 1;
        if (args.length >= 3) {
            try {
                addr = Integer.decode(args[2]);
            }
            catch (NumberFormatException e) {
                out.println("Invalid Modbus address");
                System.exit(0);
            }
        }
        String node = null;
        if (args.length >= 4) {
            node = args[3];
        }

        ModbusSlave slave = new ModbusSlave(index, serial, node);
        slave.setAddress(addr);

        out.println("Slave started");
        int nexcptn = 0;
        while (true) {
            try {
                slave.receive();
                nexcptn = 0;
            }
            catch (DriverException e) {
                out.println(e);
                if (++nexcptn >= 5) {
                    out.println("Exiting: exception loop");
                    break;
                }
            }
            catch (Exception e) {
                e.printStackTrace();
                break;
            }
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Sets the Modbus address being used
    **
    ***************************************************************************
    */
    public void setAddress(int addr)
    {
        busAddr = addr;
    }


   /**
    ***************************************************************************
    **
    **  Receives a command on the Modbus and dispatches it
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void receive() throws DriverException
    {
        checkOpen();
        byte[] cmnd = null;
        while (cmnd == null) {
            try {
                cmnd = (byte[])replyQ.take();
            }
            catch (InterruptedException e) {
            }
        }
        int addr = cmnd[0] & 0xff, func = cmnd.length >= 2 ? cmnd[1] & 0xff : 0;
        out.format("Received command: leng = %s, addr = %s, func = %s\n",
                   cmnd.length, addr, func);
        if (CRC16.generateStd(cmnd) != 0) {
            out.println("Command checksum error");
            return;
        }
        if (addr != busAddr) return;

        switch (func) {
        case Modbus.FUNC_READ_COILS:
            readCoils(cmnd);
            break;
        case Modbus.FUNC_READ_DISCRETES:
            readDiscretes(cmnd);
            break;
        case Modbus.FUNC_READ_REGISTERS:
            readRegisters(cmnd);
            break;
        case Modbus.FUNC_READ_INPUTS:
            readInputs(cmnd);
            break;
        case Modbus.FUNC_WRITE_COIL:
            writeCoil(cmnd);
            break;
        case Modbus.FUNC_WRITE_REGISTER:
            writeRegister(cmnd);
            break;
        case Modbus.FUNC_WRITE_COILS:
            writeCoils(cmnd);
            break;
        case Modbus.FUNC_WRITE_REGISTERS:
            writeRegisters(cmnd);
            break;
        default:
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_FUNC);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ COILS command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void readCoils(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd), count = getCount(cmnd);
        if (addr + count > NUM_COILS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        int nbyte = (count + 7) / 8;
        byte[] resp = createReadResp(cmnd, nbyte);
        for (int j = 0; j < nbyte; j++) {
            resp[j + 3] = 0;
        }
        for (int j = 0; j < count; j++, addr++) {
            if ((coil[addr >> 3] & (1 << (addr & 7))) != 0) {
                resp[(j >> 3) + 3] |= 1 << (j & 7);
            }
        }
        send(resp);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ DISCRETES command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void readDiscretes(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd), count = getCount(cmnd);
        if (addr + count > NUM_DISCRETES) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        int nbyte = (count + 7) / 8;
        byte[] resp = createReadResp(cmnd, nbyte);
        for (int j = 0; j < nbyte; j++) {
            resp[j + 3] = 0;
        }
        for (int j = 0; j < count; j++, addr++) {
            if ((discrete[addr >> 3] & (1 << (addr & 7))) != 0) {
                resp[(j >> 3) + 3] |= 1 << (j & 7);
            }
        }
        send(resp);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ REGISTERS command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void readRegisters(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd), count = getCount(cmnd);
        if (addr + count > NUM_REGISTERS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        byte[] resp = createReadResp(cmnd, 2 * count);
        for (int j = 0; j < count; j++, addr++) {
            Convert.shortToBytesBE(register[addr], resp, 2 * j + 3);
        }
        send(resp);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ INPUTS command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void readInputs(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd), count = getCount(cmnd);
        if (addr + count > NUM_INPUTS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        byte[] resp = createReadResp(cmnd, 2 * count);
        for (int j = 0; j < count; j++, addr++) {
            Convert.shortToBytesBE(input[addr], resp, 2 * j + 3);
        }
        send(resp);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITE COIL command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void writeCoil(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd);
        if (addr >= NUM_COILS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        short value = Convert.bytesToShortBE(cmnd, 4);
        if ((value & 0xffff) == 0xff00) {
            coil[addr >> 3] |= (byte)(1 << (addr & 7));
        }
        else if (value == 0) {
            coil[addr >> 3] &= (byte)~(1 << (addr & 7));
        }
        sendWriteResp(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITE REGISTER command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void writeRegister(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd);
        if (addr >= NUM_REGISTERS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        register[addr] = Convert.bytesToShortBE(cmnd, 4);
        sendWriteResp(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITE COILS command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void writeCoils(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd), count = getCount(cmnd);
        if (addr + count > NUM_COILS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        for (int j = 0; j < count; j++, addr++) {
            if ((cmnd[7 + (j >> 3)] & (1 << (j & 7))) != 0) {
                coil[addr >> 3] |= (byte)(1 << (addr & 7));
            }
            else {
                coil[addr >> 3] &= (byte)~(1 << (addr & 7));
            }
        }
        sendWriteResp(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITE REGISTERS command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void writeRegisters(byte[] cmnd) throws DriverException
    {
        int addr = getAddress(cmnd), count = getCount(cmnd);
        if (addr + count > NUM_REGISTERS) {
            sendExceptionResp(cmnd, Modbus.EXCP_ILL_ADDR);
            return;
        }
        for (int j = 0; j < count; j++, addr++) {
            register[addr] = Convert.bytesToShortBE(cmnd, 2 * j + 7);
        }
        sendWriteResp(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Sends a response to a write command
    **
    **  @param  cmnd  The received command
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void sendWriteResp(byte[] cmnd) throws DriverException
    {
        byte[] resp = new byte[8];
        System.arraycopy(cmnd, 0, resp, 0, 6);
        send(resp);
    }


   /**
    ***************************************************************************
    **
    **  Sends an exception response
    **
    **  @param  cmnd  The received command
    **
    **  @param  code  The exception code
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void sendExceptionResp(byte[] cmnd, int code) throws DriverException
    {
        out.println("Sending exception " + code);
        byte[] resp = new byte[5];
        resp[0] = cmnd[0];
        resp[1] = (byte)(cmnd[1] | 0x80);
        resp[2] = (byte)code;
        send(resp);
    }


   /**
    ***************************************************************************
    **
    **  Sends a response on the Modbus
    **
    **  @param  resp  The response to send
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void send(byte[] resp) throws DriverException
    {
        checkOpen();
        int leng = resp.length;
        CRC16.generateStd(resp, 0, leng - 2, resp, leng - 2);
        try {
            ftd.write(resp);
        }
        catch (DriverException e) {
            throw new DriverException(e.getMessage());
        }
        out.println("Sent response: leng = " + leng);
    }


   /**
    ***************************************************************************
    **
    **  Creates a read response
    **
    **  @param  cmnd   The received command
    **
    **  @param  nbyte  The number of bytes to be returned
    **
    ***************************************************************************
    */
    private byte[] createReadResp(byte[] cmnd, int nbyte)
    {
        byte[] resp = new byte[nbyte + 5];
        resp[0] = cmnd[0];
        resp[1] = cmnd[1];
        resp[2] = (byte)nbyte;

        return resp;
    }


   /**
    ***************************************************************************
    **
    **  Gets the address field of a command
    **
    ***************************************************************************
    */
    private static int getAddress(byte[] cmnd)
    {
        return Convert.bytesToShortBE(cmnd, 2) & 0xffff;
    }


   /**
    ***************************************************************************
    **
    **  Gets the count field of a command
    **
    ***************************************************************************
    */
    private static int getCount(byte[] cmnd)
    {
        return Convert.bytesToShortBE(cmnd, 4) & 0xffff;
    }


   /**
    ***************************************************************************
    **
    **  Checks that the port is open
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void checkOpen() throws DriverException
    {
        if (ftd == null)
            throw new DriverException("Port not open");
    }

}
