package org.lsst.ccs.drivers.modbus;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *****************************************************************************
 **
 **  Communications I/O routines for network devices.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class ModbusIONet implements ModbusIO {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    private Socket sock;
    private InputStream in;
    private OutputStream out;


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  host   The host name or IP address
    **
    **  @param  port   The port number
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String host, int port) throws DriverException
    {
        try {
            sock = new Socket(host, port);
            in = sock.getInputStream();
            out = sock.getOutputStream();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        try {
            sock.close();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Writes a command and reads the response.
    **
    **  @param  cmnd     The command to write
    **
    **  @param  resp     The buffer to receive the response data
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int send(byte[] cmnd, byte[] resp, int timeout)
        throws DriverException
    {
        write(cmnd);
        int leng = read(resp, timeout);
        if (Convert.bytesToShortBE(resp, Modbus.OFF_TID)
              != Convert.bytesToShortBE(cmnd, Modbus.OFF_TID)) {
            throw new DriverException("Transaction IDs don't match");
        }

        return leng;
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    private void write(byte[] command) throws DriverException
    {
        Convert.shortToBytesBE((short)0, command, Modbus.OFF_TID);
        Convert.shortToBytesBE((short)0, command, Modbus.OFF_PID);
        int length = command.length - Modbus.LENG_HEADER - Modbus.LENG_CRC;
        Convert.shortToBytesBE((short)length, command, Modbus.OFF_LENGTH);
        try {
            out.write(command, 0, command.length - Modbus.LENG_CRC);
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads response data.
    **
    **  @param  buff     The buffer to receive the response data
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    private int read(byte[] buff, int timeout) throws DriverException
    {
        setTimeout(timeout);
        int posn = 0, rqstLeng = Modbus.LENG_HEADER;
        while (true) {
            while (posn < rqstLeng) {
                if (buff.length < rqstLeng) {
                    throw new DriverException("Read buffer too short");
                }
                try {
                    int leng = in.read(buff, posn, rqstLeng - posn);
                    if (leng <= 0) {
                        throw new DriverException("Connection closed");
                    }
                    posn += leng;
                }
                catch (SocketTimeoutException e) {
                    throw new DriverTimeoutException("Read timed out");
                }
                catch (IOException e) {
                    throw new DriverException(e);
                }
            }
            if (rqstLeng == Modbus.LENG_HEADER) {
                rqstLeng += Convert.bytesToShortBE(buff, Modbus.OFF_LENGTH);
                if (rqstLeng > Modbus.LENG_HEADER) continue;
            }
            return posn;
        }
    }


   /**
    **************************************************************************
    **
    **  Flushes any available response data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void flush() throws DriverException
    {
        try {
            int count = in.available();
            if (count > 0) {
                byte[] buff = new byte[count];
                in.read(buff, 0, count);
            }
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }

    
   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    private void setTimeout(int timeout) throws DriverException
    {
        try {
            sock.setSoTimeout(timeout);
        }
        catch (SocketException e) {
            throw new DriverException(e);
        }
    }

}
