package org.lsst.ccs.drivers.modbus;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ftdi.Ftdi;

/**
 *****************************************************************************
 **
 **  Communications I/O routines for serial devices connected via FTDI.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class ModbusIOFtdi implements ModbusIO {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    private static final int PACKET_INTERVAL = 100;
    private final Ftdi ftdi = new Ftdi();


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  serial  The serial number string of the interface
    **
    **  @param  baud    The baud rate
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String serial, int baud) throws DriverException
    {

        ftdi.open(0, serial);
        ftdi.setBaudrate(baud);
        ftdi.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                    Ftdi.PARITY_NONE);
        ftdi.enableEvents(Ftdi.EVENT_RXCHAR);
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        ftdi.close();
    }


   /**
    **************************************************************************
    **
    **  Writes a command and reads the response.
    **
    **  @param  cmnd     The command to write
    **
    **  @param  resp     The buffer to receive the response data
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int send(byte[] cmnd, byte[] resp, int timeout)
        throws DriverException
    {
        write(cmnd);
        return read(resp, timeout);
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    private void write(byte[] command) throws DriverException
    {
        int leng = command.length;
        CRC16.generateStd(command, Modbus.LENG_HEADER,
                          leng - Modbus.LENG_HEADER - Modbus.LENG_CRC,
                          command, leng - Modbus.LENG_CRC);
        ftdi.write(command, Modbus.LENG_HEADER,
                   command.length - Modbus.LENG_HEADER);
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff     The buffer to receive the response data
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    private int read(byte[] buff, int timeout) throws DriverException
    {
        int posn = Modbus.LENG_HEADER;
        while (ftdi.awaitEvent(timeout) != 0) {
            int leng = ftdi.getQueueStatus();
            if (posn + leng > buff.length) {
                throw new DriverException("Read buffer is too short");
            }
            posn += ftdi.read(buff, posn, leng);
            timeout = PACKET_INTERVAL;
        }
        if (posn == Modbus.LENG_HEADER) {
            throw new DriverTimeoutException("Read timed out");
        }
        if (CRC16.generateStd(buff, Modbus.LENG_HEADER,
                              posn - Modbus.LENG_HEADER) != 0) {
            throw new DriverException("Reply checksum error");
        }

        return posn;
    }


   /**
    **************************************************************************
    **
    **  Flushes any available response data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void flush() throws DriverException
    {
        int count = ftdi.getQueueStatus();
        if (count > 0) {
            byte[] buff = new byte[count];
            ftdi.read(buff, 0, count);
        }
    }

}
