package org.lsst.ccs.drivers.modbus;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *****************************************************************************
 **
 **  Communicates via the Modbus protocol on a serial line or network socket
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */
public class Modbus {

   /**
    **************************************************************************
    **
    **  Public constants.
    **
    **************************************************************************
    */
    /** Connection type - network */
    public final static int CONN_TYPE_NETWORK = 0;

    /** Connection type - serial via FTDI chip */
    public final static int CONN_TYPE_FTDI    = 1;

   /**
    **************************************************************************
    **
    **  Private & package data.
    **
    **************************************************************************
    */
    final static byte
        FUNC_READ_COILS      = 1,
        FUNC_READ_DISCRETES  = 2,
        FUNC_READ_REGISTERS  = 3,
        FUNC_READ_INPUTS     = 4,
        FUNC_WRITE_COIL      = 5,
        FUNC_WRITE_REGISTER  = 6,
        FUNC_WRITE_COILS     = 15,
        FUNC_WRITE_REGISTERS = 16;

    final static int
        OFF_TID         = 0,
        OFF_PID         = 2,
        OFF_LENGTH      = 4,
        OFF_UID         = 6,
        OFF_FUNC        = 7,
        OFF_RLENG       = 8,
        OFF_ERROR       = 8,
        OFF_ADDR        = 8,
        OFF_RDATA       = 9,
        OFF_COUNT       = 10,
        OFF_DATA        = 12,
        LENG_HEADER     = 6,
        LENG_CRC        = 2;

    final static int
        EXCP_ILL_FUNC   = 1,
        EXCP_ILL_ADDR   = 2,
        EXCP_ILL_VALUE  = 3,
        EXCP_SRVR_FAIL  = 4,
        EXCP_SRVR_BUSY  = 6;

    final static int
        PORT_NUMBER  = 502,
        BAUD_RATE    = 115200;

    private final static int
        READ_TIMEOUT    = 2000;

    private ModbusIO io;
    private int nmbrOff = 1;
    private short tranId = 0;
    private final byte[] resp = new byte[1024];
    protected int timeout = READ_TIMEOUT;


   /**
    **************************************************************************
    **
    **  Opens a connection to the device.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network;
    **                   serial number for FTDI device;
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void open(int type, String ident) throws DriverException
    {
        open(type, ident, -1);
    }


   /**
    **************************************************************************
    **
    **  Opens a connection to the Modbus.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network;
    **                   serial number for FTDI device
    **
    **  @param  parm   The device parameter:
    **                   port number for network;
    **                   baud rate for FTDI
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void open(int type, String ident, int parm)
        throws DriverException
    {
        if (io != null) {
            throw new DriverException("Device already open");
        }

        ModbusIO newIo;

        switch (type) {

        case CONN_TYPE_NETWORK:
            newIo = new ModbusIONet();
            parm = (parm < 0) ? PORT_NUMBER : parm;
            break;

        case CONN_TYPE_FTDI:
            newIo = new ModbusIOFtdi();
            parm = (parm < 0) ? BAUD_RATE : parm;
            break;

        default:
            throw new DriverException("Invalid connection type: " + type);
        }

        newIo.open(ident, parm);
        io = newIo;
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the Modbus
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized void close() throws DriverException
    {
        checkOpen();
        try {
            io.close();
        }
        finally {
            io = null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether connection is open.
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void checkOpen() throws DriverException
    {
        if (io == null) {
            throw new DriverException("Device not open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets whether entity numbers are direct addresses.
    **
    **  The usual Modbus convention is that holding register numbers, etc,
    **  start from 1, so that 1 has to be subtracted in order to generate the
    **  actual address.  However, some devices provide the direct address in
    **  their documentation.
    **
    **  @param  mode  Whether or not address mode is to be set
    **
    ***************************************************************************
    */
    public void setAddressMode(boolean mode)
    {
        nmbrOff = mode ? 0 : 1;
    }


   /**
    ***************************************************************************
    **
    **  Returns whether entity numbers are direct addresses.
    **
    **  @return  Whether or not address mode is set
    **
    ***************************************************************************
    */
    public boolean isAddressMode()
    {
        return nmbrOff == 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the response timeout.
    **
    **  @param  tmo  The timeout (sec)
    **
    ***************************************************************************
    */
    public void setTimeout(double tmo)
    {
        timeout = (int)(1000 * tmo);
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of coils
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  cNmbr  The number of the first coil to read
    **
    **  @param  count  The number of coils to read
    **
    **  @return  The array of read coil values, one per bit
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized byte[] readCoils(short dAddr, short cNmbr, short count)
        throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(cNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_COILS, cmnd, resp, timeout);
        int nbyte = resp[OFF_RLENG] & 0xff;
        byte[] value = new byte[nbyte];
        System.arraycopy(resp, OFF_RDATA, value, 0, nbyte);

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of discrete inputs
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  iNmbr  The number of the first input to read
    **
    **  @param  count  The number of inputs to read
    **
    **  @return  The array of read discrete values, one per bit
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized byte[] readDiscretes(short dAddr, short iNmbr,
                                             short count) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(iNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_DISCRETES, cmnd, resp, timeout);
        int nbyte = resp[OFF_RLENG] & 0xff;
        byte[] value = new byte[nbyte];
        System.arraycopy(resp, OFF_RDATA, value, 0, nbyte);

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of holding registers
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number of the first register to read
    **
    **  @param  count  The number of registers to read
    **
    **  @return  The array of read register values
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized short[] readRegisters(short dAddr, short rNmbr,
                                              short count)
        throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_REGISTERS, cmnd, resp, timeout);
        int nvalue = (resp[OFF_RLENG] & 0xff) / 2;
        short[] value = new short[nvalue];
        for (int j = 0; j < nvalue; j++) {
            value[j] = Convert.bytesToShortBE(resp, 2 * j + OFF_RDATA);
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of input registers
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number of the first register to read
    **
    **  @param  count  The number of registers to read
    **
    **  @return  The array of read register values
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized short[] readInputs(short dAddr, short rNmbr,
                                           short count) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        send(dAddr, FUNC_READ_INPUTS, cmnd, resp, timeout);
        int nvalue = (resp[OFF_RLENG] & 0xff) / 2;
        short[] value = new short[nvalue];
        for (int j = 0; j < nvalue; j++) {
            value[j] = Convert.bytesToShortBE(resp, 2 * j + OFF_RDATA);
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a holding register
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number of the register to write
    **
    **  @param  value  The (16-bit) value to write
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized void writeRegister(short dAddr, short rNmbr,
                                           short value) throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(value, cmnd, OFF_COUNT);
        send(dAddr, FUNC_WRITE_REGISTER, cmnd, resp, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes a coil
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  cNmbr  The number of the coil to write
    **
    **  @param  value  The value (false = off, true = on) to write
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized void writeCoil(short dAddr, short cNmbr, boolean value)
        throws DriverException
    {
        byte[] cmnd = new byte[OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(cNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE((short)(value ? 0xff00 : 0), cmnd, OFF_COUNT);
        send(dAddr, FUNC_WRITE_COIL, cmnd, resp, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of holding registers
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number of the first register to
    **                 write
    **
    **  @param  value  The array of register values to write
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized void writeRegisters(short dAddr, short rNmbr,
                                            short[] value)
        throws DriverException
    {
        int count = value.length;
        byte[] cmnd = new byte[2 * count + 1 + OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(rNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE((short)count, cmnd, OFF_COUNT);
        cmnd[OFF_DATA] = (byte)(2 * count);
        for (int j = 0; j < count; j++) {
            Convert.shortToBytesBE(value[j], cmnd, 2 * j + 1 + OFF_DATA);
        }
        send(dAddr, FUNC_WRITE_REGISTERS, cmnd, resp, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of coils
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  cNmbr  The number of the first coil to write
    **
    **  @param  count  The number of coil values to write
    **
    **  @param  value  The array of coil values to write
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized void writeCoils(short dAddr, short cNmbr, short count,
                                        byte[] value) throws DriverException
    {
        int nbyte = (count + 7) / 8;
        byte[] cmnd = new byte[nbyte + 1 + OFF_DATA + LENG_CRC];
        Convert.shortToBytesBE((short)(cNmbr - nmbrOff), cmnd, OFF_ADDR);
        Convert.shortToBytesBE(count, cmnd, OFF_COUNT);
        cmnd[OFF_DATA] = (byte)nbyte;
        System.arraycopy(value, 0, cmnd, OFF_DATA + 1, nbyte);
        send(dAddr, FUNC_WRITE_COILS, cmnd, resp, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Sends a command to a device on the Modbus and returns its response
    **
    **  @param  addr     The destination Modbus address
    **
    **  @param  func     The function code
    **
    **  @param  cmnd     The command byte array to send.
    **
    **  @param  resp     A byte array to receive the response.
    **
    **  @param  timeout  The timeout period (ms)
    **
    **  @return  The number of bytes read
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private int send(short addr, byte func, byte[] cmnd, byte[] resp,
                     int timeout) throws DriverException
    {
        checkOpen();
        Convert.shortToBytesBE(tranId++, cmnd, OFF_TID);
        cmnd[OFF_UID] = (byte)addr;
        cmnd[OFF_FUNC] = func;
        int leng = io.send(cmnd, resp, timeout);
        if ((resp[OFF_UID] & 0xff) != addr) {
            throw new DriverException("Invalid reply address");
        }
        if ((resp[OFF_FUNC] & 0x7f) != func) {
            throw new DriverException("Invalid reply function code");
        }
        if ((resp[OFF_FUNC] & 0x80) != 0) {
            String message;
            int code = resp[OFF_ERROR];
            switch (code) {
            case EXCP_ILL_FUNC:
                message = "illegal function";
                break;
            case EXCP_ILL_ADDR:
                message = "illegal data address";
                break;
            case EXCP_ILL_VALUE:
                message = "illegal data value";
                break;
            case EXCP_SRVR_FAIL:
                message = "server device failure";
                break;
            case EXCP_SRVR_BUSY:
                message = "server device busy";
                break;
            default:
                message = "unknown code (" + code + ")";
            }
            throw new DriverException("Reply exception: " + message);
        }

        return leng;
    }

}
