package org.lsst.ccs.drivers.modbus;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.lsst.ccs.drivers.ftdi.Ftdi;
import org.lsst.ccs.drivers.ftdi.FtdiException;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to test a Modbus device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestModbus implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_OPEN        = 0,
        CMD_CLOSE       = 1,
        CMD_COILREAD    = 2,
        CMD_DISCREAD    = 3,
        CMD_REGREAD     = 4,
        CMD_INPREAD     = 5,
        CMD_COILWRITE   = 6,
        CMD_REGWRITE    = 7,
        CMD_SHOWMODEM   = 8,
        NUM_CMDS        = 9;

    /*
    **  Command help
    */
    private final static String[] helpOpen = {
        "Open device connection",
        "open [<index>] [<serial>] [<addr>] [<node>]",
        "index   The index of the FTDI device to connect to (default 0)",
        "serial  The serial number of the FTDI device (default none)",
        "addr    The Modbus address of the device (default 1)",
        "node    The node from which Modbus is being served (default local)",
    };

    private final static String[] helpClose = {
        "Close device connection",
        "close",
    };

    private final static String[] helpCoilread = {
        "Reads and displays a set of coils",
        "coilread <nmbr> [<count>]",
        "nmbr   The number of the first coil to read",
        "count  The number of coils to read (default 1)",
    };

    private final static String[] helpDiscread = {
        "Reads and displays a set of discrete inputs",
        "discread <nmbr> [<count>]",
        "nmbr   The number of the first discrete input to read",
        "count  The number of discrete inputs to read (default 1)",
    };

    private final static String[] helpRegread = {
        "Reads and displays a set of holding registers",
        "regread <nmbr> [<count>]",
        "nmbr   The number of the first holding register to read",
        "count  The number of holding registers to read (default 1)",
    };

    private final static String[] helpInpread = {
        "Reads and displays a set of input registers",
        "regread <nmbr> [<count>]",
        "nmbr   The number of the first input register to read",
        "count  The number of input registers to read (default 1)",
    };

    private final static String[] helpCoilwrite = {
        "Writes a set of coils",
        "coilwrite <nmbr> <value1> [<value2>] ... [<value8>]",
        "nmbr    The number of the first coil to write",
        "valuen  The nth value to write, 0 (off), ~0 (on)",
    };

    private final static String[] helpRegwrite = {
        "Writes a set of holding registers",
        "regwrite <nmbr> <value1> [<value2>] ... [<value8>]",
        "nmbr    The number of the first holding register to write",
        "valuen  The nth value to write",
    };

    private final static String[] helpShowmodem = {
        "Displays the status of the modem lines",
        "showmodem",
    };

    /*
    **  Command definition table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("open",       CMD_OPEN,       helpOpen,       "isws");
        cmnd.add("close",      CMD_CLOSE,      helpClose,      "");
        cmnd.add("coilread",   CMD_COILREAD,   helpCoilread,   "Ww");
        cmnd.add("discread",   CMD_DISCREAD,   helpDiscread,   "Ww");
        cmnd.add("regread",    CMD_REGREAD,    helpRegread,    "Ww");
        cmnd.add("inpread",    CMD_INPREAD,    helpInpread,    "Ww");
        cmnd.add("coilwrite",  CMD_COILWRITE,  helpCoilwrite,  "WWwwwwwww");
        cmnd.add("regwrite",   CMD_REGWRITE,   helpRegwrite,   "WWwwwwwww");
        cmnd.add("showmodem",  CMD_SHOWMODEM,  helpShowmodem,  "");
    }

    /*
    **  Private fields
    */
    private final static PrintStream out = System.out;
    private final CmndProcess proc = new CmndProcess();
    private Modbus mod = new Modbus();
    private short busAddr;
    private boolean devOpen;


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestModbus()).run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    **  <p>
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null || !proc.process(line)) break;
            }
            if (devOpen) mod.close();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_OPEN:
                procOpen(found, args); break;
            case CMD_CLOSE:
                procClose(found, args); break;
            case CMD_COILREAD:
                procCoilread(found, args); break;
            case CMD_DISCREAD:
                procDiscread(found, args); break;
            case CMD_REGREAD:
                procRegread(found, args); break;
            case CMD_INPREAD:
                procInpread(found, args); break;
            case CMD_COILWRITE:
                procCoilwrite(found, args); break;
            case CMD_REGWRITE:
                procRegwrite(found, args); break;
            case CMD_SHOWMODEM:
                procShowmodem(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (ModbusException e) {
            out.println(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the OPEN command
    **
    ***************************************************************************
    */
    private void procOpen(int found, Object[] args) throws ModbusException
    {
        int index = (found & 0x01) != 0 ? (Integer)args[0] : 0;
        String serial = (found & 0x02) != 0 ? (String)args[1] : null;
        String node = (found & 0x08) != 0 ? (String)args[3] : null;
        mod.open(node, index, serial);
        devOpen = true;
        busAddr = (found & 0x04) != 0 ? (Short)args[2] : 1;
    }


   /**
    ***************************************************************************
    **
    **  Processes the CLOSE command
    **
    ***************************************************************************
    */
    private void procClose(int found, Object[] args) throws ModbusException
    {
        devOpen = false;
        mod.close();
    }


   /**
    ***************************************************************************
    **
    **  Processes the COILREAD command
    **
    ***************************************************************************
    */
    private void procCoilread(int found, Object[] args) throws ModbusException
    {
        short addr = (Short)args[0];
        short count = (found & 0x02) != 0 ? (Short)args[1] : 1;
        byte[] reply = mod.readCoils(busAddr, addr, count);
        int nvalue = count <= 8 * reply.length ? count : 8 * reply.length;
        for (int j = 0; j < nvalue; j++) {
            if ((j % 20) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", addr + j);
            }
            out.format("%2s", (reply[j >> 3] & (1 << (j & 7))) != 0 ? 1 : 0);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the DISCREAD command
    **
    ***************************************************************************
    */
    private void procDiscread(int found, Object[] args) throws ModbusException
    {
        short nmbr = (Short)args[0];
        short count = (found & 0x02) != 0 ? (Short)args[1] : 1;
        byte[] reply = mod.readDiscretes(busAddr, nmbr, count);
        int nvalue = count <= 8 * reply.length ? count : 8 * reply.length;
        for (int j = 0; j < nvalue; j++) {
            if ((j % 20) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", nmbr + j);
            }
            out.format("%2s", (reply[j >> 3] & (1 << (j & 7))) != 0 ? 1 : 0);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGREAD command
    **
    ***************************************************************************
    */
    private void procRegread(int found, Object[] args) throws ModbusException
    {
        short nmbr = (Short)args[0];
        short count = (found & 0x02) != 0 ? (Short)args[1] : 1;
        short[] reply = mod.readRegisters(busAddr, nmbr, count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", nmbr + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the INPREAD command
    **
    ***************************************************************************
    */
    private void procInpread(int found, Object[] args) throws ModbusException
    {
        short nmbr = (Short)args[0];
        short count = (found & 0x02) != 0 ? (Short)args[1] : 1;
        short[] reply = mod.readInputs(busAddr, nmbr, count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%4s:", nmbr + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the COILWRITE command
    **
    ***************************************************************************
    */
    private void procCoilwrite(int found, Object[] args) throws ModbusException
    {
        short nmbr = (Short)args[0];
        short count = 0, value = 0;
        for (int j = 0; j < 8; j++, found >>= 1) {
            if ((found & 2) != 0) {
                if ((Short)args[j + 1] != 0) {
                    value |= (1 << count);
                }
                count++;
            }
        }
        if (count == 1) {
            mod.writeCoil(busAddr, nmbr, value != 0);
        }
        else {
            mod.writeCoils(busAddr, nmbr, count, new byte[]{(byte)value});
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGWRITE command
    **
    ***************************************************************************
    */
    private void procRegwrite(int found, Object[] args) throws ModbusException
    {
        short nmbr = (Short)args[0];
        short count = 0;
        short[] value = new short[8];
        for (int j = 0; j < 8; j++, found >>= 1) {
            if ((found & 2) != 0) {
                value[count++] = (Short)args[j + 1];
            }
        }
        if (count == 1) {
            mod.writeRegister(busAddr, nmbr, value[0]);
        }
        else {
            short[] values = new short[count];
            System.arraycopy(value, 0, values, 0, count);
            mod.writeRegisters(busAddr, nmbr, values);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWMODEM command
    **
    ***************************************************************************
    */
    private void procShowmodem(int found, Object[] args) throws ModbusException
    {
        try {
            Ftdi ftd = mod.getFtdi();
            int status = ftd.getModemStatus();
            out.format("Modem status: 0x%04x\n", status);
        }
        catch (FtdiException e) {
            throw new ModbusException(e.getMessage());
        }
    }

}
