package org.lsst.ccs.drivers.modbus;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.drivers.ftdi.Ftdi;
import org.lsst.ccs.drivers.ftdi.FtdiException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Handles Modbus RTU messages on a serial line
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Modbus {

    /**
     **  Package constants
     */
    final static byte
        FUNC_READ_COILS      = 1,
        FUNC_READ_DISCRETES  = 2,
        FUNC_READ_REGISTERS  = 3,
        FUNC_READ_INPUTS     = 4,
        FUNC_WRITE_COIL      = 5,
        FUNC_WRITE_REGISTER  = 6,
        FUNC_WRITE_COILS     = 15,
        FUNC_WRITE_REGISTERS = 16;

    final static int
        EXCP_ILL_FUNC   = 1,
        EXCP_ILL_ADDR   = 2,
        EXCP_ILL_VALUE  = 3,
        EXCP_SRVR_FAIL  = 4;

    final static int
        BAUD_RATE    = 115200;

    /**
     **  Private constants
     */
    private final static int
        READ_TIMEOUT    = 2000,
        CLOSE_TIMEOUT   = 5000,
        READ_INTERVAL   = 1000,
        PACKET_INTERVAL = 50;

    /**
     **  Private fields
     */
    private final Reader reader = new Reader();
    private Thread readMb;
    private Ftdi ftd = new Ftdi();
    private ArrayBlockingQueue replyQ = new ArrayBlockingQueue(1),
                               closeQ = new ArrayBlockingQueue(1);
    private boolean closed = true;

   /**
    ***************************************************************************
    **
    **  Implements the Modbus-reading thread
    **
    ***************************************************************************
    */
    private class Reader implements Runnable {

        @Override
        public void run()
        {
            FtdiException excp = null;
            byte[] buff = new byte[256];

            // Loop reading packets until close request or read error
            while (true) {
                int leng = 0, nread = 0;

                // Wait for data to arrive and read packet
                try {
                    ftd.setTimeouts(READ_INTERVAL, 0);
                    while (!closed && nread == 0) {
                        nread = ftd.read(buff, 0, 1);
                    }
                    if (closed) break;
                    leng += nread;
                    ftd.setTimeouts(PACKET_INTERVAL, 0);
                    nread = ftd.read(buff, leng, buff.length - leng);
                    leng += nread;
                }
                catch (FtdiException e) {
                    excp = e;
                    break;
                }

                // Queue read packer to requestor
                byte[] data = new byte[leng];
                System.arraycopy(buff, 0, data, 0, leng);
                replyQ.offer(data);
            }

            // Got error or close request: close the connection
            try {
                ftd.close();
            }
            catch (FtdiException e) {
                if (closed) {
                    excp = e;
                }
            }

            // If error, queue exception back to requestor
            if (!closed) {
                closed = true;
                replyQ.offer(new ModbusException(excp.getMessage()));
            }

            // Otherwise queue close result back to requestor
            else {
                if (excp == null) {
                    closeQ.offer(new Object());
                }
                else {
                    closeQ.offer(new ModbusException(excp.getMessage()));
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the Modbus
    **
    **  @param  index   The zero-based index of the Modbus adapter within the
    **                  list of FTDI devices selected by the serial argument
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void open(int index, String serial) throws ModbusException
    {
        open(null, index, serial);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the Modbus
    **
    **  @param  node    The name of the node from which the Modbus is being
    **                  served, or null for the local node
    **
    **  @param  index   The zero-based index of the Modbus adapter within the
    **                  list of FTDI devices selected by the serial argument
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void open(String node, int index, String serial)
        throws ModbusException
    {
        try {
            ftd.open(node, index, serial);
            ftd.setBaudrate(BAUD_RATE);
            ftd.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                       Ftdi.PARITY_NONE);
        }
        catch (FtdiException e) {
            throw new ModbusException(e.getMessage());
        }
        closed = false;
        readMb = new Thread(reader);
        readMb.setDaemon(true);
        readMb.start();
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the Modbus
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void close() throws ModbusException
    {
        if (closed) {
            throw new ModbusException("Device not open");
        }
        closeQ.clear();
        closed = true;
        Object result = null;
        try {
            result = closeQ.poll(CLOSE_TIMEOUT, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
        }
        if (result == null) {
            throw new ModbusException("Close timeout");
        }
        if (result instanceof ModbusException) {
            throw (ModbusException)result;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the associated Ftdi object
    **
    ***************************************************************************
    */
    public Ftdi getFtdi()
    {
        return ftd;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of coils
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  cNmbr  The number (starting at 1) of the first coil to read
    **
    **  @param  count  The number of coils to read
    **
    **  @return  The array of read coil values, one per bit
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public byte[] readCoils(short dAddr, short cNmbr, short count)
        throws ModbusException
    {
        byte[] cmnd = new byte[8], resp;
        cmnd[1] = FUNC_READ_COILS;
        Convert.shortToBytesBE((short)(cNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE(count, cmnd, 4);
        resp = send(dAddr, cmnd, READ_TIMEOUT);
        int nbyte = resp[2] & 0xff;
        byte[] value = new byte[nbyte];
        System.arraycopy(resp, 3, value, 0, nbyte);

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of discrete inputs
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  iNmbr  The number (starting at 1) of the first input to read
    **
    **  @param  count  The number of inputs to read
    **
    **  @return  The array of read discrete values, one per bit
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public byte[] readDiscretes(short dAddr, short iNmbr, short count)
        throws ModbusException
    {
        byte[] cmnd = new byte[8];
        cmnd[1] = FUNC_READ_DISCRETES;
        Convert.shortToBytesBE((short)(iNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE(count, cmnd, 4);
        byte[] resp = send(dAddr, cmnd, READ_TIMEOUT);
        int nbyte = resp[2] & 0xff;
        byte[] value = new byte[nbyte];
        System.arraycopy(resp, 3, value, 0, nbyte);

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of holding registers
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number (starting at 1) of the first register to read
    **
    **  @param  count  The number of registers to read
    **
    **  @return  The array of read register values
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public short[] readRegisters(short dAddr, short rNmbr, short count)
        throws ModbusException
    {
        byte[] cmnd = new byte[8];
        cmnd[1] = FUNC_READ_REGISTERS;
        Convert.shortToBytesBE((short)(rNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE(count, cmnd, 4);
        byte [] resp = send(dAddr, cmnd, READ_TIMEOUT);
        int nvalue = (resp[2] & 0xff) / 2;
        short[] value = new short[nvalue];
        for (int j = 0; j < nvalue; j++) {
            value[j] = Convert.bytesToShortBE(resp, 2 * j + 3);
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a set of input registers
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number (starting at 1) of the first register to read
    **
    **  @param  count  The number of registers to read
    **
    **  @return  The array of read register values
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public short[] readInputs(short dAddr, short rNmbr, short count)
        throws ModbusException
    {
        byte[] cmnd = new byte[8];
        cmnd[1] = FUNC_READ_INPUTS;
        Convert.shortToBytesBE((short)(rNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE(count, cmnd, 4);
        byte[] resp = send(dAddr, cmnd, READ_TIMEOUT);
        int nvalue = (resp[2] & 0xff) / 2;
        short[] value = new short[nvalue];
        for (int j = 0; j < nvalue; j++) {
            value[j] = Convert.bytesToShortBE(resp, 2 * j + 3);
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a holding register
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number (starting at 1) of the register to write
    **
    **  @param  value  The (16-bit) value to write
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void writeRegister(short dAddr, short rNmbr, short value)
        throws ModbusException
    {
        byte[] cmnd = new byte[8];
        cmnd[1] = FUNC_WRITE_REGISTER;
        Convert.shortToBytesBE((short)(rNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE(value, cmnd, 4);
        send(dAddr, cmnd, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Writes a coil
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  cNmbr  The number (starting at 1) of the coil to write
    **
    **  @param  value  The value (false = off, true = on) to write
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void writeCoil(short dAddr, short cNmbr, boolean value)
        throws ModbusException
    {
        byte[] cmnd = new byte[8];
        cmnd[1] = FUNC_WRITE_COIL;
        Convert.shortToBytesBE((short)(cNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE((short)(value ? 0xff00 : 0), cmnd, 4);
        send(dAddr, cmnd, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of holding registers
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  rNmbr  The number (starting at 1) of the first register to
    **                 write
    **
    **  @param  value  The array of register values to write
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void writeRegisters(short dAddr, short rNmbr, short[] value)
        throws ModbusException
    {
        int count = value.length;
        byte[] cmnd = new byte[2 * count + 9];
        cmnd[1] = FUNC_WRITE_REGISTERS;
        Convert.shortToBytesBE((short)(rNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE((short)count, cmnd, 4);
        cmnd[6] = (byte)(2 * count);
        for (int j = 0; j < count; j++) {
            Convert.shortToBytesBE(value[j], cmnd, 2 * j + 7);
        }
        send(dAddr, cmnd, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of coils
    **
    **  @param  dAddr  The Modbus address of the device
    **
    **  @param  cNmbr  The number (starting at 1) of the first coil to write
    **
    **  @param  count  The number of coil values to write
    **
    **  @param  value  The array of coil values to write
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public void writeCoils(short dAddr, short cNmbr, short count, byte[] value)
        throws ModbusException
    {
        int nbyte = (count + 7) / 8;
        byte[] cmnd = new byte[nbyte + 9];
        cmnd[1] = FUNC_WRITE_COILS;
        Convert.shortToBytesBE((short)(cNmbr - 1), cmnd, 2);
        Convert.shortToBytesBE(count, cmnd, 4);
        cmnd[6] = (byte)nbyte;
        System.arraycopy(value, 0, cmnd, 7, nbyte);
        send(dAddr, cmnd, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Sends a command to a device on the Modbus and returns its response
    **
    **  @param  addr     The destination Modbus address
    **
    **  @param  cmnd     The command byte array to send.
    **
    **  @param  timeout  The timeout period (ms)
    **
    **  @return  The response byte array
    **
    **  @throws ModbusException
    **
    ***************************************************************************
    */
    public byte[] send(short addr, byte[] cmnd, int timeout)
        throws ModbusException
    {
        replyQ.clear();
        cmnd[0] = (byte)addr;
        int leng = cmnd.length;
        CRC16.generateStd(cmnd, 0, leng - 2, cmnd, leng - 2);
        try {
            ftd.write(cmnd);
        }
        catch (FtdiException e) {
            throw new ModbusException(e.getMessage());
        }
        Object respObj = null;
        try {
            respObj = (byte[])replyQ.poll(timeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
        }
        if (respObj == null) {
            throw new ModbusException("Read timeout");
        }
        if (respObj instanceof ModbusException) {
            throw (ModbusException)respObj;
        }

        byte[] resp = (byte[])respObj;
        if (CRC16.generateStd(resp) != 0) {
            throw new ModbusException("Reply checksum error");
        }
        if ((resp[0] & 0xff) != addr) {
            throw new ModbusException("Invalid reply address");
        }
        if ((resp[1] & 0x7f) != cmnd[1]) {
            throw new ModbusException("Invalid reply function code");
        }
        if ((resp[1] & 0x80) != 0) {
            String message;
            int code = resp[2];
            switch (code) {
            case EXCP_ILL_FUNC:
                message = "illegal function";
                break;
            case EXCP_ILL_ADDR:
                message = "illegal data address";
                break;
            case EXCP_ILL_VALUE:
                message = "illegal data value";
                break;
            case EXCP_SRVR_FAIL:
                message = "server device failure";
                break;
            default:
                message = "unknown code (" + code + ")";
            }
            throw new ModbusException("Reply exception: " + message);
        }

        return resp;
    }

}
