package org.lsst.ccs.drivers.mmr;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ****************************************************************************** 
 ** 
 **  Program to test the MMR controller device driver.
 ** 
 **  @author Owen Saxton
 ** 
 ****************************************************************************** 
 */
public class TestMmr implements Mmr.Listener {

    protected Mmr mmr = new Mmr();
    int nSkip, nLeft;

    public enum ConnType {

        FTDI(Scpi.CONN_TYPE_FTDI),
        SERIAL(Scpi.CONN_TYPE_SERIAL);

        int value;

        ConnType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    public enum OnOff {

        ON(true),
        OFF(false);

        boolean value;

        OnOff(boolean value)
        {
            this.value = value;
        }

        public boolean getValue()
        {
            return value;
        }

    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type",
                               description="Connection type: serial or ftdi")
                     ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident
                     ) throws DriverException
    {
        mmr.open(type.getValue(), ident);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="close", description="Close connection to device")
    public void close() throws DriverException
    {
        mmr.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the listener on or off.
    ** 
    *************************************************************************** 
    */
    @Command(name="listen", description="Turns listener on or off")
    public void write(@Argument(name="onoff", description="state: on or off")
                      OnOff onoff
                      )
    {
        if (onoff.getValue()) {
            mmr.addListener(this);
        }
        else {
            mmr.removeListener();
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the data skip count.
    ** 
    *************************************************************************** 
    */
    @Command(name="skip", description="Sets data message skip count")
    public void write(@Argument(name="count", description="Items to skip")
                      int count
                      )
    {
        nSkip = count;
        nLeft = 0;
    }


   /**
    ***************************************************************************
    **
    **  Processes logged data
    **
    **  @param  data  An 8-element double array containing the data.
    **
    ***************************************************************************
    */
    @Override
    public void processData(double[] data)
    {
        if (nLeft-- > 0) return;
        System.out.format("A = %s, T0 = %s, T1 = %s, P0 = %s, P1 = %s, "
                            + "D0 = %s, D1 = %s, I = %s\n",
                          data[Mmr.FLD_A], data[Mmr.FLD_T0], data[Mmr.FLD_T1],
                          data[Mmr.FLD_P0], data[Mmr.FLD_P1], data[Mmr.FLD_D0],
                          data[Mmr.FLD_D1], data[Mmr.FLD_I]);
        nLeft = nSkip;
    }


   /**
    ***************************************************************************
    **
    **  Handles power on/off transitions
    **
    **  @param  on  Whether the power is on
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        System.out.println("Power turned " + (on ? "on" : "off"));
    }


   /**
    ***************************************************************************
    **
    **  Handles error-induced device closure
    **
    ***************************************************************************
    */
    @Override
    public void setClosed(Exception e)
    {
        System.out.println("Connection closed: " + e);
    }

}
