package org.lsst.ccs.drivers.mmr;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ******************************************************************************
 **
 **  Accesses an MMR compressor control chassis.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Mmr {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    /** Data field - address (ID) */
    public final static int FLD_A  = 0;

    /** Data field - thermocouple 0 */
    public final static int FLD_T0 = 1;

    /** Data field - thermocouple 1 */
    public final static int FLD_T1 = 2;

    /** Data field - pressure transducer 0 */
    public final static int FLD_P0 = 3;

    /** Data field - pressure transducer 1 */
    public final static int FLD_P1 = 4;

    /** Data field - digital input line 0 */
    public final static int FLD_D0 = 5;

    /** Data field - digital input line 1 */
    public final static int FLD_D1 = 6;

    /** Data field - current */
    public final static int FLD_I  = 7;

    /** Number of data fields */
    public final static int N_FIELDS = 8;

   /**
    ***************************************************************************
    **
    **  Private constants & fields
    **
    ***************************************************************************
    */
    private final static int
        BAUD_RATE       = 9600;
    private final static double
        READ_TIMEOUT    = 2;
    private final static int[]
        UNITS_LENGTH = {0, 1, 1, 2, 2, 0, 0, 3};

    private Reader reader;
    private Scpi scpi = new Scpi();
    private boolean open, powered;
    private Listener listener;


   /**
    ***************************************************************************
    **
    **  Implements the reading thread
    **
    ***************************************************************************
    */
    private class Reader extends Thread {

        private Exception excp;

        @Override
        public void run()
        {
            // Loop to read and process each record; break if error
            while (open) {
                try {
                    String data = scpi.read();
                    if (!powered) {
                        powered = true;
                        if (listener != null) {
                            listener.setPowered(powered);
                        }
                    }
                    sendListener(data);
                }
                catch (DriverTimeoutException e) {
                    if (powered) {
                        powered = false;
                        if (listener != null) {
                            listener.setPowered(powered);
                        }
                    }
                }
                catch (Exception e) {
                    excp = e;
                    break;
                }
            }

            // Done if not open
            if (!open) return;

            // Error: close the connection and report to listener
            try {
                close();
            }
            catch (DriverException e) {
            }
            if (listener != null) {
                listener.setClosed(excp);
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Defines an event listener
    **
    ***************************************************************************
    */
    public interface Listener {

       /**
        ***********************************************************************
        **
        **  Processes logged data
        **
        **  @param  data  An 8-element double array containing the data.
        **
        ***********************************************************************
        */
        public void processData(double[] data);

       /**
        ***********************************************************************
        **
        **  Handles power on/off transitions
        **
        **  @param  on  Whether the power is on
        **
        ***********************************************************************
        */
        public void setPowered(boolean on);

       /**
        ***********************************************************************
        **
        **  Handles error-induced device closure
        **
        **  @param  e  The exception
        **
        ***********************************************************************
        */
        public void setClosed(Exception e);

    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the controller.
    **
    **  @param  type    The type of connection, serial or FTDI.
    **
    **  @param  ident   Device identifier: device name for serial, USB serial
    **                  number for SCPI.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void open(int type, String ident) throws DriverException
    {
        if (type == Scpi.CONN_TYPE_NETWORK) {
            throw new DriverException("Invalid connection type");
        }
        scpi.open(type, ident, BAUD_RATE);
        scpi.setTimeout(READ_TIMEOUT);
        open = true;
        reader = new Reader();
        reader.setDaemon(true);
        reader.start();
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the controller.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void close() throws DriverException
    {
        if (!open) {
            throw new DriverException("Device not open");
        }
        open = false;
        powered = false;
        scpi.close();
    }


   /**
    ***************************************************************************
    **
    **  Adds an event listener.
    **
    **  @param  listener  The event listener object, containing the
    **                    {@code processData}, {@code setPowered} and
    **                    {@code setClosed} methods.
    **
    ***************************************************************************
    */
    public void addListener(Listener listener)
    {
        this.listener = listener;
    }


   /**
    ***************************************************************************
    **
    **  Removes the event listener.
    **
    ***************************************************************************
    */
    public void removeListener()
    {
        listener = null;
    }


   /**
    ***************************************************************************
    **
    **  Converts and sends logged data to the listener.
    **
    **  The reply string, which consists of a set of comma-delimited
    **  labelled fields, is processed into an array of doubles.
    **
    **  A sample string:
    **
    **    >A=1,T0=30.5C,T1=24.6C,P0=3201mV,P1=2767mV,D0=1,D1=0,I=865mV<
    **
    **  @param  reply  The reply string received from the controller.
    **
    ***************************************************************************
    */
    private void sendListener(String reply)
    {
        if (listener == null) return;

        String[] fields = reply.split(",");
        double[] values = new double[N_FIELDS];

        for (int j = 0; j < N_FIELDS; j++) {
            String[] items = fields[j].split("=");
            int leng = items[1].length() - UNITS_LENGTH[j];
            values[j] = Double.valueOf(items[1].substring(0, leng));
        }
        listener.processData(values);
    }

}
