package org.lsst.ccs.drivers.mks;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.ascii.Multidrop;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  Routines for communicating with an MKS model 972, 974 or 925 vacuum gauge
 *
 *  @author  Owen Saxton
 */
public class Model9XX extends Multidrop {

    /**
     *  Public constants.
     */
    public enum Sensor {

        COMB(3), COMB4(4), MP(1), CC(5), PZ(2);

        private final int value;
        
        Sensor(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }
    }

    public enum Unit {TORR, MBAR, PASCAL}
    public enum Direction {ABOVE, BELOW}
    public enum Enable {CC, PIR, CMB, ON, OFF}
    public enum Gas {NITROGEN, AIR, ARGON, HELIUM, HYDROGEN, H2O, NEON, CO2, XENON}
    public enum Default {ALL, VAC, VAC3, ATM, CFS, MZL}

    public static final int
        DEFAULT_BAUDRATE = 9600,
        DEFAULT_ADDRESS = 253,
        RELAY_MIN   = 1,
        RELAY_MAX   = 3,
        OUTPUT_MIN  = 1,
        OUTPUT_MAX  = 2;

    /**
     *  Private constants & data.
     */
    private static final int
        DEFAULT_TIMEOUT = 100,
        LEVEL_925 = 0,
        LEVEL_972 = 1,
        LEVEL_974 = 2,
        LEVEL_UNKNOWN = 3;
    private static final double
        MAX_PRESSURE = 1500;  // Maximum allowed pressure (Torr)

    private static final Map<String, Unit> unitRespMap = new HashMap<>();
    static {
        for (Unit unit : Unit.values()) {
            unitRespMap.put(unit.name(), unit);
        }
    }
    private static final Map<Unit, Double> convFromTorrMap = new HashMap<>();
    static {
        convFromTorrMap.put(Unit.TORR, 1.0);
        convFromTorrMap.put(Unit.MBAR, 1.333);
        convFromTorrMap.put(Unit.PASCAL, 133.3);
    }
    private static final Map<String, Direction> dirRespMap = new HashMap<>();
    static {
        for (Direction dir : Direction.values()) {
            dirRespMap.put(dir.name(), dir);
        }
    }
    private static final Map<String, Enable> enabRespMap = new HashMap<>();
    static {
        for (Enable enab : Enable.values()) {
            enabRespMap.put(enab.name(), enab);
        }
    }
    private static final Map<String, Gas> gasRespMap = new HashMap<>();
    static {
        for (Gas gas : Gas.values()) {
            gasRespMap.put(gas.name(), gas);
        }
    }
    private static final Map<String, Boolean> setRespMap = new HashMap<>();
    static {
        setRespMap.put("ON", true);
        setRespMap.put("OFF", false);
    }
    private static final Map<String, Boolean> activeRespMap = new HashMap<>();
    static {
        activeRespMap.put("SET", true);
        activeRespMap.put("CLEAR", false);
    }
    private static final Map<String, Integer> delayRespMap = new HashMap<>();
    static {
        delayRespMap.put("ON", 120);
        delayRespMap.put("OFF", -1);
    }
    private static final Map<String, Integer> levelMap = new HashMap<>();
    static {
        levelMap.put("925", LEVEL_925);
        levelMap.put("972", LEVEL_972);
        levelMap.put("974", LEVEL_974);
    }

    private static final byte RESP_INTR = '@';
    private static final byte[] RESP_TERM = {';', 'F', 'F'};
    private static final String CMND_SFX = ";FF";
    private String addrStr;
    private String cmndPfx;
    private final byte[] readBuff = new byte[256];
    private int readBuffLeng = 0;
    private String model = "";
    private int cmndLevel = LEVEL_UNKNOWN;
    private Object cmndSync = this;
    private double maxPressure = MAX_PRESSURE;
    private int retryLimit = 0, retryCount;


    /**
     *  Constructor
     */
    public Model9XX()
    {
        setDefaultBaud(DEFAULT_BAUDRATE);
        setDefaultAddress(DEFAULT_ADDRESS);
    }


    /**
     *  Opens a multi-drop (RS-485) connection.
     * 
     *  Overrides the main (5-argument) Multidrop open method
     *
     *  @param  ident     The serial port name
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The data characteristics (not used)
     *  @param  addr      The device address
     *  @throws  DriverException
     */
    @Override
    public void open(String ident, int baudRate, int dataChar, int addr) throws DriverException
    {
        super.open(ident, baudRate, 0, addr);
        setAddress(address);
        setTimeout(DEFAULT_TIMEOUT);
        retryCount = 0;
        cmndSync = getSyncObject();
        try {
            model = getModel();
            cmndLevel = LEVEL_UNKNOWN;
            if (model.length() >= 3) {
                cmndLevel = levelMap.get(model.substring(0, 3));
            }
            if (cmndLevel == LEVEL_UNKNOWN) {
                throw new DriverException("Unrecognized gauge model: " + model);
            }
            getPressureUnit();
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


    /**
     *  Gets the transducer device type.
     *
     *  @return  The device type name
     *  @throws  DriverException
     */
    public String getDeviceType() throws DriverException
    {
        return readString("DT?");
    }


    /**
     *  Gets the firmware version.
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    public String getFirmwareVersion() throws DriverException
    {
        return readString("FV?");
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     *  @throws  DriverException
     */
    public String getHardwareVersion() throws DriverException
    {
        return readString("HV?");
    }


    /**
     *  Gets the manufacturer.
     *
     *  @return  The manufacturer
     *  @throws  DriverException
     */
    public String getManufacturer() throws DriverException
    {
        return readString("MF?");
    }


    /**
     *  Gets the model.
     *
     *  @return  The model
     *  @throws  DriverException
     */
    public String getModel() throws DriverException
    {
        return readString("MD?");
    }


    /**
     *  Gets the part number.
     *
     *  @return  The part number
     *  @throws  DriverException
     */
    public String getPartNumber() throws DriverException
    {
        return readString("PN?");
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getSerialNumber() throws DriverException
    {
        return readString("SN?");
    }


    /**
     *  Gets the transducer time on.
     *
     *  @return  The transducer time on (hours)
     *  @throws  DriverException
     */
    public int getTimeOn() throws DriverException
    {
        return readInt("TIM?");
    }

    /**
     *  Gets the sensor time on.
     *
     *  @return  The sensor time on (hours)
     *  @throws  DriverException
     */
    public int getSensorTimeOn() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readInt("TIM2?");
    }


    /**
     *  Gets the sensor pressure dose.
     *
     *  @return  The sensor pressure dose
     *  @throws  DriverException
     */
    public double getSensorDose() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("TIM3?");
    }


    /**
     *  Gets the transducer status.
     *
     *  @return  The status string
     *  @throws  DriverException
     */
    public String getTransStatus() throws DriverException
    {
        return readString("T?");
    }


    /**
     *  Sets the pressure unit.
     *
     *  @param  unit  The pressure unit: TORR, MBAR or PASCAL
     *  @throws  DriverException
     */
    public void setPressureUnit(Unit unit) throws DriverException
    {
        writeCmnd("U!" + unit.name());
        getPressureUnit();
    }


    /**
     *  Gets the pressure unit.
     *
     *  @return  The pressure unit
     *  @throws  DriverException
     */
    public Unit getPressureUnit() throws DriverException
    {
        String resp = readString("U?");
        Unit unit = unitRespMap.get(resp);
        if (unit == null) {
            throwBadResponse("unit", resp);
        }
        maxPressure = MAX_PRESSURE * convFromTorrMap.get(unit);
        return unit;
    }


    /**
     *  Reads a pressure sensor.
     *
     *  @param  sensor  The pressure sensor enum
     *  @return  The pressure
     *  @throws  DriverException
     */
    public double readPressure(Sensor sensor) throws DriverException
    {
        if (cmndLevel == LEVEL_925 && sensor != Sensor.MP && sensor != Sensor.COMB4) {
            throw new DriverException("Invalid sensor (" + sensor + ") for model " + model);
        }
        double value = readDouble("PR" + sensor.getValue() + "?");
        if (value > maxPressure) {
            throw new DriverException("Invalid (> " + maxPressure + ") pressure value: " + value);
        }
        return value;
    }


    /**
     *  Reads the combined pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    public double readPressure() throws DriverException
    {
        return readPressure(cmndLevel == LEVEL_925 ? Sensor.MP : Sensor.COMB);
    }


    /**
     *  Reads the MicroPirani temperature.
     *
     *  @return  The temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return readDouble("TEM?");
    }


    /**
     *  Sets the state of the protected function lock.
     *
     *  @param  on  Whether lock is to be set on
     *  @throws  DriverException
     */
    public void setLock(boolean on) throws DriverException
    {
        writeCmnd(on ? "FD!LOCK" : "FD!UNLOCK");
    }


    /**
     *  Sets the baud rate.
     *
     *  @param  baud  The baud rate: 4800, 9600, 19200, 38400, 57600, 115200 or 230400
     *  @throws  DriverException
     */
    public void setBaudRate(int baud) throws DriverException
    {
        writeCmnd("BR!" + baud);
    }


    /**
     *  Gets the baud rate.
     *
     *  @return  The baud rate: 4800, 9600, 19200, 38400, 57600, 115200 or 230400
     *  @throws  DriverException
     */
    public int getBaudRate() throws DriverException
    {
        return readInt("BR?");
    }


    /**
     *  Changes the device address.
     *
     *  @param  addr  The new address
     *  @throws  DriverException
     */
    public void changeAddress(int addr) throws DriverException
    {
        if (addr < 1 || addr > 253) {
            throw new DriverException("Invalid device address: " + addr);
        }
        writeCmnd("AD!" + String.format("%03d", addr));
        setAddress(addr);
    }


    /**
     *  Sets the user tag.
     *
     *  @param  tag  The tag string (15 characters maximum)
     *  @throws  DriverException
     */
    public void setUserTag(String tag) throws DriverException
    {
        writeCmnd("UT!" + tag);
    }


    /**
     *  Gets the user tag.
     *
     *  @return  The user tag string
     *  @throws  DriverException
     */
    public String getUserTag() throws DriverException
    {
        return readString("UT?");
    }


    /**
     *  Sets the switch enabled state.
     *
     *  @param  on  Whether enabled or not
     *  @throws  DriverException
     */
    public void setSwitchEnable(boolean on) throws DriverException
    {
        writeCmnd("SW!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets the switch enabled state.
     *
     *  @return  Whether the switch is enabled
     *  @throws  DriverException
     */
    public boolean isSwitchEnabled() throws DriverException
    {
        String resp = readString("SW?");
        Boolean on = setRespMap.get(resp);
        if (on == null) {
            throwBadResponse("switch enabled", resp);
        }
        return on;
    }


    /**
     *  Sets the delayed response state.
     *
     *  @param  on  Whether enabled or not
     *  @throws  DriverException
     */
    public void setDelayedResponse(boolean on) throws DriverException
    {
        writeCmnd("RSD!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets the RS-485 delayed response state.
     *
     *  @return  Whether delayed response enabled
     *  @throws  DriverException
     */
    public boolean isResponseDelayed() throws DriverException
    {
        String resp = readString("RSD?");
        Boolean on = setRespMap.get(resp);
        if (on == null) {
            throwBadResponse("delayed response", resp);
        }
        return on;
    }


    /**
     *  Sets the relay trip delayed state.
     *
     *  @param  on  Whether enabled or not
     *  @throws  DriverException
     */
    public void setRelayDelayed(boolean on) throws DriverException
    {
        writeCmnd("SPD!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets the relay trip delayed state.
     *
     *  @return  Whether relay trip delay is enabled
     *  @throws  DriverException
     */
    public boolean isRelayDelayed() throws DriverException
    {
        String resp = readString("SPD?");
        Boolean on = setRespMap.get(resp);
        if (on == null) {
            throwBadResponse("relay trip delayed", resp);
        }
        return on;
    }


    /**
     *  Sets the identify state.
     *
     *  @param  on  Whether enabled or not
     *  @throws  DriverException
     */
    public void setIdentify(boolean on) throws DriverException
    {
        writeCmnd("TST!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets the identify state.
     *
     *  @return  Whether the identify light is flashing
     *  @throws  DriverException
     */
    public boolean isIdentifying() throws DriverException
    {
        String resp = readString("TST?");
        Boolean on = setRespMap.get(resp);
        if (on == null) {
            throwBadResponse("identify", resp);
        }
        return on;
    }


    /**
     *  Sets the calibration gas.
     *
     *  @param  gas  The gas enum
     *  @throws  DriverException
     */
    public void setCalibrationGas(Gas gas) throws DriverException
    {
        writeCmnd("GT!" + gas);
    }


    /**
     *  Gets the calibration gas.
     *
     *  @return  The calibration gas enum
     *  @throws  DriverException
     */
    public Gas getCalibrationGas() throws DriverException
    {
        String resp = readString("GT?");
        Gas gas = gasRespMap.get(resp);
        if (gas == null) {
            throwBadResponse("calibration gas", resp);
        }
        return gas;
    }


    /**
     *  Calibrates MicroPirani at atmospheric pressure.
     *
     *  @param  press  The atmospheric pressure value to set
     *  @throws  DriverException
     */
    public void calibrateAtmospheric(double press) throws DriverException
    {
        writeCmnd("ATM!" + press);
    }


    /**
     *  Gets the atmospheric pressure calibration factor.
     *
     *  @return  The atmospheric pressure factor
     *  @throws  DriverException
     */
    public double getAtmospheric() throws DriverException
    {
        return readDouble("ATM?");
    }


    /**
     *  Calibrates MicroPirani at vacuum pressure.
     *
     *  @throws  DriverException
     */
    public void calibrateVacuumMP() throws DriverException
    {
        writeCmnd("VAC!");
    }


    /**
     *  Sets the MicroPirani auto zero limit.
     *
     *  @param  press  The limit pressure value to set
     *  @throws  DriverException
     */
    public void setAutoZeroLimit(double press) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("MZL!" + press);
    }


    /**
     *  Gets the MicroPirani auto zero limit.
     *
     *  @return  The auto zero limit pressure
     *  @throws  DriverException
     */
    public double getAutoZeroLimit() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("MZL?");
    }


    /**
     *  Calibrates cold cathode at full scale.
     *
     *  @param  press  The full-scale pressure value to set
     *  @throws  DriverException
     */
    public void calibrateFullScale(double press) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("CFS!" + press);
    }


    /**
     *  Gets the cold cathode full-scale pressure calibration factor.
     *
     *  @return  The full-scale pressure factor
     *  @throws  DriverException
     */
    public double getFullScale() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("CFS?");
    }


    /**
     *  Calibrates cold cathode at vacuum pressure.
     *
     *  @throws  DriverException
     */
    public void calibrateVacuumCC() throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("VAC3!");
    }


    /**
     *  Sets cold cathode turn-on pressure.
     *
     *  @param  press  The turn-on pressure to set
     *  @throws  DriverException
     */
    public void setOnPressureCC(double press) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("SLC!" + press);
    }


    /**
     *  Gets cold cathode turn-on pressure.
     *
     *  @return  The turn-on pressure
     *  @throws  DriverException
     */
    public double getOnPressureCC() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("SLC?");
    }


    /**
     *  Sets cold cathode turn-off pressure.
     *
     *  @param  press  The turn-off pressure to set
     *  @throws  DriverException
     */
    public void setOffPressureCC(double press) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("SHC!" + press);
    }


    /**
     *  Gets cold cathode turn-off pressure.
     *
     *  @return  The turn-off pressure
     *  @throws  DriverException
     */
    public double getOffPressureCC() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("SHC?");
    }


    /**
     *  Sets low integration pressure.
     *
     *  @param  press  The pressure to set
     *  @throws  DriverException
     */
    public void setLowIntPressure(double press) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("SLP!" + press);
    }


    /**
     *  Gets low integration pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    public double getLowIntPressure() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("SLP?");
    }


    /**
     *  Sets high integration pressure.
     *
     *  @param  press  The pressure to set
     *  @throws  DriverException
     */
    public void setHighIntPressure(double press) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("SHP!" + press);
    }


    /**
     *  Gets high integration pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    public double getHighIntPressure() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("SHP?");
    }


    /**
     *  Sets cold cathode auto power on/off.
     *
     *  @param  on  Whether on or off
     *  @throws  DriverException
     */
    public void setCCAutoPower(boolean on) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("ENC!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets cold cathode auto power state.
     *
     *  @return  Whether auto power state is on
     *  @throws  DriverException
     */
    public boolean isCCAutoPowerOn() throws DriverException
    {
        checkLevel(LEVEL_972);
        String resp = readString("ENC?");
        Boolean on = setRespMap.get(resp);
        if (on == null) {
            throwBadResponse("cold cathode auto power", resp);
        }
        return on;
    }


    /**
     *  Sets cold cathode power on/off.
     *
     *  @param  on  Whether on or off
     *  @throws  DriverException
     */
    public void setCCPower(boolean on) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("FP!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets cold cathode power state.
     *
     *  @return  Whether power state is on
     *  @throws  DriverException
     */
    public boolean isCCPowerOn() throws DriverException
    {
        checkLevel(LEVEL_972);
        String resp = readString("FP?");
        Boolean on = setRespMap.get(resp);
        if (on == null) {
            throwBadResponse("cold cathode power", resp);
        }
        return on;
    }


    /**
     *  Sets cold cathode protection delay.
     *
     *  @param  delay  The delay before turning off (sec)
     *  @throws  DriverException
     */
    public void setCCProtDelay(int delay) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("PRO!" + delay);
    }


    /**
     *  Sets cold cathode protection on/off.
     *
     *  @param  on  Whether to enable protection (true = 120 sec)
     *  @throws  DriverException
     */
    public void setCCProtection(boolean on) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("PRO!" + (on ? "ON" : "OFF"));
    }


    /**
     *  Gets cold cathode protection delay.
     *
     *  @return  Protection delay time (-1 = OFF)
     *  @throws  DriverException
     */
    public int getCCProtDelay() throws DriverException
    {
        checkLevel(LEVEL_972);
        String resp = readString("PRO?");
        Integer delay = delayRespMap.get(resp);
        if (delay == null) {
            try {
                delay = Integer.valueOf(resp);
            }
            catch (NumberFormatException e) {
                throwBadResponse("cold cathode protection", resp);
            }
        }
        return delay;
    }


    /**
     *  Sets cold cathode pressure dose limit.
     *
     *  @param  dose  The pressure dose to set
     *  @throws  DriverException
     */
    public void setSensorDoseLimit(double dose) throws DriverException
    {
        checkLevel(LEVEL_972);
        writeCmnd("PD!" + dose);
    }


    /**
     *  Gets cold cathode pressure dose limit.
     *
     *  @return  press  The pressure dose
     *  @throws  DriverException
     */
    public double getSensorDoseLimit() throws DriverException
    {
        checkLevel(LEVEL_972);
        return readDouble("PD?");
    }


    /**
     *  Sets a relay set point.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  value  The set point value
     *  @throws  DriverException
     */
    public void setRelayTrip(int relay, double value) throws DriverException
    {
        checkRelayNumber(relay);
        writeCmnd("SP" + relay + "!" + value);
    }


    /**
     *  Gets a relay set point.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The set point value
     *  @throws  DriverException
     */
    public double getRelayTrip(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        return readDouble("SP" + relay + "?");
    }


    /**
     *  Sets a relay hysteresis value.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  value  The hysteresis value
     *  @throws  DriverException
     */
    public void setRelayHyst(int relay, double value) throws DriverException
    {
        checkRelayNumber(relay);
        writeCmnd("SH" + relay + "!" + value);
    }


    /**
     *  Gets a relay hysteresis value.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The hysteresis value
     *  @throws  DriverException
     */
    public double getRelayHyst(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        return readDouble("SH" + relay + "?");
    }


    /**
     *  Sets a relay trip direction.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  dirn   The trip direction, ABOVE or BELOW
     *  @throws  DriverException
     */
    public void setRelayDirection(int relay, Direction dirn) throws DriverException
    {
        checkRelayNumber(relay);
        writeCmnd("SD" + relay + "!" + dirn);
    }


    /**
     *  Gets a relay trip direction.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The trip direction
     *  @throws  DriverException
     */
    public Direction getRelayDirection(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        String resp = readString("SD" + relay + "?");
        Direction dirn = dirRespMap.get(resp);
        if (dirn == null) {
            throwBadResponse("relay direction", resp);
        }
        return dirn;
    }


    /**
     *  Sets the enabled state of a relay.
     *
     *  @param  relay   The relay number (1 - 3)
     *  @param  enable  The enable type (CC, PIR, CMB, ON, OFF)
     *  @throws  DriverException
     */
    public void setRelayEnable(int relay, Enable enable) throws DriverException
    {
        checkRelayNumber(relay);
        writeCmnd("EN" + relay + "!" + enable);
    }


    /**
     *  Gets the enabled state of a relay.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The enabled state (CC, PIR, ON (= CMB), OFF)
     *  @throws  DriverException
     */
    public Enable getRelayEnable(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        String resp = readString("EN" + relay + "?");
        Enable enab = enabRespMap.get(resp);
        if (enab == null) {
            throwBadResponse("relay enable", resp);
        }
        return enab;
    }


    /**
     *  Gets whether a relay is active.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  Whether active
     *  @throws  DriverException
     */
    public boolean isRelayActive(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        String resp = readString("SS" + relay + "?");
        Boolean active = activeRespMap.get(resp);
        if (active == null) {
            throwBadResponse("relay active", resp);
        }
        return active;
    }


    /**
     *  Sets analog output parameters.
     *
     *  @param  chan    The analog channel (1 or 2)
     *  @param  sensor  The pressure sensor to use
     *  @param  curve   The output curve number (0 - 32)
     *  @throws  DriverException
     */
    public void setAnalogOut(int chan, Sensor sensor, int curve) throws DriverException
    {
        checkOutputChannel(chan);
        if (cmndLevel == LEVEL_925 && sensor != Sensor.MP) {
            throw new DriverException("Invalid sensor (" + sensor + ") for model " + model);
        }
        writeCmnd("AO" + chan + "!" + sensor.getValue() + curve);
    }


    /**
     *  Gets an analog output string.
     *
     *  @param  chan  The analog channel (1 or 2)
     *  @return  The output string
     *  @throws  DriverException
     */
    public String getAnalogOut(int chan) throws DriverException
    {
        checkOutputChannel(chan);
        return readString("AO" + chan + "?");
    }


    /**
     *  Sets factory defaults.
     *
     *  @throws  DriverException
     */
    public void setDefaults() throws DriverException
    {
        writeCmnd("FD!");
    }


    /**
     *  Sets factory defaults.
     *
     *  @param  item  The item for which to set default value(s)
     *  @throws  DriverException
     */
    public void setDefaults(Default item) throws DriverException
    {
        if (cmndLevel == LEVEL_925 && item != Default.ALL && item != Default.ATM && item != Default.VAC) {
            throw new DriverException("Invalid default item (" + item + ") for model " + model);
        }
        writeCmnd("FD!" + item);
    }


    /**
     *  Sets the command retry limit
     * 
     *  @param limit 
     */
    public void setRetryLimit(int limit) {
        retryLimit = limit;
    }


    /**
     *  Gets the command retry limit
     * 
     *  @return  The retry limit
     */
    public int getRetryLimit() {
        return retryLimit;
    }


    /**
     *  Clears the command retry count.
     */
    public void clearRetryCount() {
        retryCount = 0;
    }


    /**
     *  Gets the command retry count
     * 
     *  @return  The retry count
     */
    public int getRetryCount() {
        return retryCount;
    }


    /**
     *  Gets the contents of the read buffer.
     *
     *  @return  The buffer contents
     */
    public synchronized String getReadBuffer()
    {
        return new String(readBuff, 0, readBuffLeng);
    }


    /**
     *  Formats the contents of the read buffer.
     *
     *  @return  The buffer contents
     */
    public synchronized String formatReadBuffer()
    {
        return formatBuffer(readBuff, 0, readBuffLeng);
    }


    /**
     *  Sets the device address.
     * 
     *  @param  addr  The address (0 - 255)
     */
    private void setAddress(int addr)
    {
        addrStr = String.format("%03d", addr);
        cmndPfx = "@" + addrStr;
    }


    /**
     *  Sends a command.
     *
     *  @param  cmnd  The command
     *  @throws  DriverException
     */
    void writeCmnd(String cmnd) throws DriverException
    {
        readString(cmnd);
    }


    /**
     *  Sends a command and returns the response.
     *
     *  If a timeout occurs and the response is empty, the command is retried the specified number of times
     *
     *  @param  cmnd  The command
     *  @return  The response, stripped of its header & trailer
     *  @throws  DriverException
     */
    String readString(String cmnd) throws DriverException
    {
        String response = "";
        for (int rCount = 0; rCount <= retryLimit; rCount++) {
            try {
                response = readString1(cmnd);
                break;
            }
            catch (DriverTimeoutException e) {
                if (readBuffLeng > 0 || rCount >= retryLimit) {
                    throw new DriverTimeoutException("Read timeout: buffer contents = " + formatReadBuffer());
                }
                retryCount++;
            }
        }
        return response;
    }


    /**
     *  Sends a command and returns the response.
     *
     *  @param  cmnd  The command
     *  @return  The response, stripped of its header & trailer
     *  @throws  DriverException
     */
    private synchronized String readString1(String cmnd) throws DriverException
    {
        readBuffLeng = 0;
        int respEnd = -1, respStart = -1;
        synchronized (cmndSync) {
            flush();
            writeBytes((cmndPfx + cmnd + CMND_SFX).getBytes());
            int inPosn = 0, termIx = 0;
            while (respEnd < 0) {
                readBuffLeng = readBytes(readBuff, inPosn) + inPosn;
                if (respStart < 0) {
                    for (; inPosn < readBuffLeng; inPosn++) {
                        if (readBuff[inPosn] == RESP_INTR) {
                            respStart = ++inPosn;
                            break;
                        }
                    }
                }
                if (respStart >= 0) {
                    for (; inPosn < readBuffLeng; inPosn++) {
                        if (readBuff[inPosn] == RESP_TERM[termIx]) {
                            if (++termIx == RESP_TERM.length) {
                                respEnd = inPosn + 1 - RESP_TERM.length;
                                break;
                            }
                        }
                        else {
                            termIx = 0;
                        }
                    }
                }
            }
        }
        String reply = new String(readBuff, respStart, respEnd - respStart).replace("\0", "");
        if (reply.length() < 6) {
            throw new DriverException("Reply (" + reply + ") is too short");
        }
        String rAddr = reply.substring(0, 3);
        if (!rAddr.equals(addrStr)) {
            throw new DriverException("Invalid reply address: " + rAddr);
        }
        String acknak = reply.substring(3, 6);
        String resp = reply.substring(6);

        switch (acknak) {
        case "ACK":
            return resp;

        case "NAK":
            switch (resp) {
            case "8":
                throw new DriverException("Zero adjustment at too high pressure");
            case "9":
                throw new DriverException("Atmospheric adjustment at too low pressure");
            case "160":
                throw new DriverException("Unrecognized command");
            case "169":
                throw new DriverException("Invalid argument");
            case "172":
                throw new DriverException("Value out of range");
            case "175":
                throw new DriverException("Invalid command character");
            case "180":
                throw new DriverException("Protected setting");
            case "195":
                throw new DriverException("Control setpoint enabled");
            default:
                throw new DriverException("Unrecognized error: " + resp);
            }

        default:
            throw new DriverException("Unrecognized reply: " + formatBuffer(readBuff, respStart, respEnd));
        }
    }


    /**
     *  Sends a command and returns the integer response.
     *
     *  @param  cmnd  The command
     *  @return  The response, as an integer
     *  @throws  DriverException
     */
    private int readInt(String cmnd) throws DriverException
    {
        String resp = readString(cmnd);
        try {
            return Integer.valueOf(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid integer value: " + resp);
        }
    }


    /**
     *  Sends a command and returns the double response.
     *
     *  @param  cmnd  The command
     *  @return  The response, as a double
     *  @throws  DriverException
     */
    private double readDouble(String cmnd) throws DriverException
    {
        String resp = readString(cmnd);
        try {
            return Double.valueOf(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid double value: " + resp);
        }
    }


    /**
     *  Throws an "unrecognized response" exception.
     *
     *  @param  cmnd  Command identifier
     *  @param  resp  The unrecognized response
     *  @throws  DriverException
     */
    private void throwBadResponse(String cmnd, String resp) throws DriverException {
        throw new DriverException("Unrecognized " + cmnd + " response: " + resp);
    }


    /**
     *  Checks whether it's the correct model level.
     *
     *  @param  type  The required model type
     *  @throws  DriverException
     */
    private void checkLevel(int level) throws DriverException {
        if (cmndLevel < level) {
            throw new DriverException("Operation not available on a model " + model);
        }
    }


    /**
     *  Checks a relay number for validity.
     *
     *  @param  relay  The relay number
     *  @throws  DriverException
     */
    private void checkRelayNumber(int relay) throws DriverException {
        if (relay < RELAY_MIN || relay > RELAY_MAX) {
            throw new DriverException("Invalid relay number: " + relay);
        }
    }


    /**
     *  Checks an analog output channel number for validity.
     *
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    private void checkOutputChannel(int chan) throws DriverException {
        if (chan < OUTPUT_MIN || chan > OUTPUT_MAX) {
            throw new DriverException("Invalid output channel number: " + chan);
        }
    }


    /**
     *  Formats the contents of a buffer.
     *
     *  @param  buff   The buffer
     *  @param  start  The start position
     *  @param  end    The end position
     *  @return  The formatted buffer contents
     */
    private String formatBuffer(byte[] buff, int start, int end)
    {
        StringBuilder rBuff = new StringBuilder("'");
        for (int j = start; j < end; j++) {
            char ch = (char)buff[j];
            rBuff.append(ch >= ' ' && ch <= '~' ? ch : '?');
        }
        rBuff.append("' (");
        String space = "";
        for (int j = start; j < end; j++) {
            byte ch = buff[j];
            rBuff.append(space);
            rBuff.append(String.format("%02x", ch));
            space = " ";
        }
        rBuff.append(")");
        return rBuff.toString();
    }

}
