package org.lsst.ccs.drivers.mks;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ** General access routines for the GP835 device
 *
 ** @author Homer Neal
 */
public class GP835 extends Scpi implements GP390.HardComm {

    /**
     * Private fields
     */
    private static final double UNAVAIL_VALUE = 9.91e37;
    private static final double SHORT_TIMEOUT = 1.0, LONG_TIMEOUT = 10.0;
    private final int gp390Address = 2;
    private final String gp390Format = "SYST:COMM:SER:ACOM? " + gp390Address + ",\"%s\"";
    private boolean DEBUG = false;
    private final GP390 gp390 = new GP390();

    /**
     * Constructor
     */
    public GP835() {
        setOptions(Option.NO_NET);
    }

    /**
     *  Opens a connection.
     *
     *  Overrides the SCPI open and provides any additional actions.
     *
     *  @param  connType  The connection type (SERIAL)
     *  @param  ident     The serial port name
     *  @param  parm1     The first parameter (baud rate)
     *  @param  parm2     The second parameter (data characteristics)
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int parm1, int parm2) throws DriverException {
        super.open(connType, ident, parm1, parm2);
        setTerminator(Terminator.CR);
        setTimeout(SHORT_TIMEOUT);
        checkIdentification(null, CHECK_CONTAINS, "835", CHECK_CONTAINS);
        gp390.setCommHandler(this);
        gp390.setAddress(gp390Address);
    }

    /**
     * Opens a serial connection.
     *
     * @param serialname The device name of the serial port
     * @throws DriverException
     */
    public void open(String serialname) throws DriverException {
        open(ConnType.SERIAL, serialname, 0);
    }

    /**
     * Returns the pressure
     *
     * @return The pressure
     * @throws DriverException
     * 9.5.6 External Total Pressure To get the total
     * pressure from an external instrument: INST ETPR OUTP ON MEAS:PRES?
     */
    public double getPressure() throws DriverException {
	writeCommand("INST ETPR");
        double value = readDouble("MEAS:PRES?");
        return value == UNAVAIL_VALUE ? Double.NaN : value;
    }

    /**
     * Returns the pressure directly from the GP390
     *
     * @return The pressure
     * @throws DriverException
     */
    public double getPressureGP390() throws DriverException {
        return gp390.readPressure();
    }

    /**
     * Returns the filament power
     *
     * @return The power
     * @throws DriverException
     */
    public double getFilamentPower() throws DriverException {
	writeCommand("INST FIL");
        return readDouble("MEAS:POW?");
    }

    /**
     * Gets the mass spectrometer scan results
     *
     * @return The scan data
     * @throws DriverException
     */
    public String getMSP() throws DriverException {
	writeCommand("INST MSP");
        return readString("FORM:AMU ON;FORM:COUNT ON;OUTP ON ;INIT;FETC?");
    }

    /**
     * Gets the test report
     *
     * @return The test report, as a string
     * @throws DriverException
     */
    public String getTestReport() throws DriverException {
        return readString("TEST:REP?");
    }

    /**
     * Gets the currently selected instrument
     *
     * @return The name of the current instrument
     * @throws DriverException
     */
    public String getInstrument() throws DriverException {
        return readString("INST?");
    }

    /**
     * Sets the whole vacuum gauge on.
     * 
     * This takes 7-8 seconds
     *
     * @throws DriverException
     */
    public void setGaugeOn() throws DriverException {
        //write("INST ETPR;OUTP ON");  // This doesn't work!!!
        setTimeout(LONG_TIMEOUT);
       	writeCommand("INST ETPR");
	writeCommand("OUTP ON");
        setTimeout(SHORT_TIMEOUT);
    }

    /**
     * Sets the whole vacuum gauge off --- usually to avoid light
     *
     * @throws DriverException *
     */
    public void setGaugeOff() throws DriverException {
        //write("INST ETPR;OUTP OFF");  // This doesn't work!!!
       	writeCommand("INST ETPR");
	writeCommand("OUTP OFF");
    }

    /**
     * Gets whether the vacuum gauge is on
     *
     * @return Whether the gauge is on
     * @throws DriverException
     */
    public boolean isGaugeOn() throws DriverException {
       	writeCommand("INST ETPR");
        return readInteger("OUTP?") == 1;
    }

    /**
     * Sets the vacuum gauge micro-ion filament on.
     * 
     * This takes 7-8 seconds
     *
     * @throws DriverException
     */
    public void setFilamentOn() throws DriverException {
        gp390.setMiGaugeOn(true);
    }

    /**
     * Sets the vacuum gauge micro-ion filament off --- usually to avoid light
     *
     * @throws DriverException *
     */
    public void setFilamentOff() throws DriverException {
        gp390.setMiGaugeOn(false);
    }

    /**
     * Gets whether the vacuum gauge micro-ion filament is on
     *
     * @return Whether the gauge is on
     * @throws DriverException
     */
    public boolean isFilamentOn() throws DriverException {
        return gp390.isMiGaugeOn();
    }

    /**
     *  Sets a trip relay set point.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  type   The set point type (1 = activation, 0 = deactivation)
     *  @param  value  The set point value
     *  @throws  DriverException
     */
    public void setRelayTrip(int relay, int type, double value) throws DriverException
    {
        gp390.setRelayTrip(relay, type, value);
    }

    /**
     *  Gets a trip relay set point.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  type   The set point type (1 = activation, 0 = deactivation)
     *  @return  The set point value
     *  @throws  DriverException
     */
    public double getRelayTrip(int relay, int type) throws DriverException
    {
        return gp390.getRelayTrip(relay, type);
    }

    /**
     *  Assigns a trip relay to a vacuum type.
     *
     *  @param  relay   The relay number (1 - 3)
     *  @param  assign  The assignment: 0 = absolute, 1 = differential
     *  @throws  DriverException
     */
    public void assignRelay(int relay, int assign) throws DriverException
    {
        gp390.assignRelay(relay, assign);
    }

    /**
     *  Gets the assignment for a trip relay.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The assignment: 0 = absolute, 1 = differential
     *  @throws  DriverException
     */
    public int getRelayAssignment(int relay) throws DriverException
    {
        return gp390.getRelayAssignment(relay);
    }

    /**
     *  Enables/disables a trip relay.
     *
     *  @param  relay   The relay number (1 - 3)
     *  @param  enable  Whether to enable
     *  @throws  DriverException
     */
    public void enableRelay(int relay, boolean enable) throws DriverException
    {
        gp390.enableRelay(relay, enable);
    }

    /**
     *  Gets whether a trip relay is enabled.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  Whether enabled
     *  @throws  DriverException
     */
    public boolean isRelayEnabled(int relay) throws DriverException
    {
        return gp390.isRelayEnabled(relay);
    }

    /**
     *  Gets whether a trip relay is active.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  Whether active
     *  @throws  DriverException
     */
    public boolean isRelayActive(int relay) throws DriverException
    {
        return gp390.isRelayActive(relay);
    }

    /**
     * Sets the mass spectrometer on.
     * 
     * This takes ? seconds
     *
     * @throws DriverException
     */
    public void setMSPOn() throws DriverException {
        //write("INST MSP;OUTP ON");  // This doesn't work!!!
        setTimeout(LONG_TIMEOUT);
       	writeCommand("INST MSP");
	writeCommand("OUTP ON");
        setTimeout(SHORT_TIMEOUT);
    }

    /**
     * Sets the mass spectrometer off --- usually to avoid light
     *
     * @throws DriverException *
     */
    public void setMSPOff() throws DriverException {
        //write("INST MSP;OUTP OFF");  // This doesn't work!!!
       	writeCommand("INST MSP");
	writeCommand("OUTP OFF");
    }

    /**
     * Reads the AMU data
     *
     * @return Array of double pairs of data
     * @throws DriverException 
     */
    public double[][] readAMU() throws DriverException {
        double[][] amuarray = new double[5000][2];
        String buff;
        buff = getMSP();

        int blen = buff.length();

        int ichar = 0;

        int icrv = buff.indexOf(" CURVE ");
        if (icrv > 0) {
            int istart = buff.indexOf(" VALues ");
            if (istart > 0) {
                istart += 8;
            }
            ichar = istart;
            ByteBuffer curv;
            curv = ByteBuffer.wrap(buff.substring(istart).getBytes());
            curv.order(ByteOrder.LITTLE_ENDIAN);
            ShortBuffer mspdat = curv.asShortBuffer();

            double val;
            int idx = 0;
            
            while ((ichar + 1) < blen) {
                char thischar = buff.charAt(ichar);
                char nextchar = buff.charAt(ichar + 1);
                if (thischar != '\n' && nextchar != '\n') {
                    val = mspdat.get();
                    amuarray[idx][1] = val;
                    idx++;
                    ichar += 2;
                } else {
                    break;
                }
                System.out.println("ADC count = " + val);

            }
        
        }
        return(amuarray);
    }
    /*
     public double[][] readAMU() throws DriverException {
     readIntegerGP835("*OPC?");  // any other operations to complete
     write("FORM:ALL ON,ON");
     write("INIT");       // get measurements

     int krdy = 0;
     System.out.println("Beginning of readBuffer: krdy = " + krdy);
     while (krdy != 1) { // this should have been set false by accumbuffer
     System.out.println("In loop of readBuffer: krdy = " + krdy);
     try {
     krdy = readIntegerGP835("*OPC?");  // wait for read operations to complete
     } catch (Exception ex) {
     krdy = 0;                         // obviously we are not ready yet
     }
     System.out.println("krdy = " + krdy);
     if (krdy != 1) {
     System.out.println("sleep for a second then recheck.");
     try {
     Thread.currentThread().sleep(1000);//sleep for 1000 ms
     } catch (InterruptedException ex) {
     System.out.println("Cannot sleep because of exception: " + ex);
     }
     }
     // in case conditions are never satisfied, count on the timeout to free us
     }
     if (DEBUG) {
     System.out.println("About to do double pair array read.");
     }
     double[][] buff = readDoublePairArray("FETCH?"); // request all stored readings
     if (DEBUG) {
     System.out.println("Fetched data.");
     }
     return buff;
     }
     */

    /**
     * Reads a double pair array with SCPI error checking after writing a
     * command.
     *
     * @param command The command to write, excluding terminator
     * @return The returned array of double float values
     * @throws DriverException
     */
    public double[][] readDoublePairArray(String command) throws DriverException {
        try {
            if (DEBUG) {
                System.out.println("Doing String Array read.");
            }
            String[] reply = readStringArray(command);
            if (DEBUG) {
                /*
                 for (String str : reply) {
                 System.out.println("reply string = " + str);
                 }
                 */
                System.out.println("Retrieved string of values from buffer. length = " + reply.length);
            }
            double[][] dReply = new double[2][reply.length / 2];
            int ridx = 0;
            for (int j = 0; j < reply.length; j += 2) {
//                System.out.println("reply = (" + reply[j] + ")");
                dReply[0][ridx] = Double.valueOf(reply[j]);
                dReply[1][ridx] = Double.valueOf(reply[j + 1]);
                ridx++;
            }
            if (DEBUG) {
                System.out.println("Binary array prepared.");
            }
            return dReply;
        } catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }

    /**
     * Checks that GP390 connection is open.
     * 
     * This is a no-op.
     */
    @Override
    public void checkOpenGP390() {
    }

    /**
     * Sends a GP390 command and returns the result
     * 
     * @param cmnd The command to send
     * @return The raw response
     * @throws DriverException
     */
    @Override
    public String readGP390(String cmnd) throws DriverException {
        String value = readString(String.format(gp390Format, cmnd)).split(" *!")[0];
        if (value.isEmpty()) {
            throw new DriverException("Vacuum gauge is turned off");
        }
        return value;
    }

}
