package org.lsst.ccs.drivers.mks;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for communicating with an MKS GP390 series vacuum gauge
 *
 *  @author  Owen Saxton
 */
public class GP390 {

    /**
     *  Inner hardware communications interface.
     * 
     *  This is needed because the GP390 is available both as a stand-alone RS485
     *  device, and as a part of the GP835 VQM system.
     */
    static interface HardComm {

        String readGP390(String command) throws DriverException;

        void checkOpenGP390() throws DriverException;

    }

    /**
     *  Inner class for storing open connection data.
     */
    static class OpenDesc {

        Ascii asc;
        int refCount = 0;

    }

    /**
     *  Inner class for implementing hardware communications
     */
    class Comm implements HardComm {

        /**
         *  Sends a command and returns the response.
         *
         *  @param  cmnd  The command
         *  @return  The raw response
         *  @throws  DriverException
         */
        @Override
        public String readGP390(String cmnd) throws DriverException
        {
            return desc.asc.read(cmndPfx + cmnd);

        }

        /**
         *  Checks whether a connection is open.
         *
         *  @throws  DriverException
         */
        @Override
        public void checkOpenGP390() throws DriverException
        {
            if (desc == null) {
                throw new DriverException("Connection not open");
            }
        }

    }

    /**
     *  Constants & data.
     */
    @Deprecated
    public enum ConnType { SERIAL; }

    public static final int
        UNIT_TORR = 0,
        UNIT_MBAR = 1,
        UNIT_PA   = 2,
        DISP_ABS  = 0,
        DISP_DIFF = 1,
        DISP_BOTH = 2,
        FMODE_MAN  = 0,
        FMODE_ALT  = 1,
        FMODE_AUTO = 2,
        GMODE_OFF   = 0,
        GMODE_DEGAS = 1,
        GMODE_HIGH  = 2,
        GMODE_LOW   = 3,
        RELAY_MIN   = 1,
        RELAY_MAX   = 3,
        RELAY_ABS   = 0,
        RELAY_DIFF  = 1,
        RELAY_ACT   = 1,
        RELAY_DEACT = 0;

    private static final double UNAVAIL_VALUE = 9.99e09;

    private static final Map<String, Integer> ruRespMap = new HashMap<>();
    static {
        ruRespMap.put("TORR", UNIT_TORR);
        ruRespMap.put("MBAR", UNIT_MBAR);
        ruRespMap.put("PASCAL", UNIT_PA);
    }
    private static final Map<String, Boolean> tlRespMap = new HashMap<>();
    static {
        tlRespMap.put("0 UL OFF", false);
        tlRespMap.put("1 UL  ON", true);
    }
    private static final Map<String, Integer> rfRespMap = new HashMap<>();
    static {
        rfRespMap.put("FIL  SF1", 1);
        rfRespMap.put("FIL  SF2", 2);
        rfRespMap.put("FIL  SFB", 3);
    }
    private static final Map<String, Boolean> igRespMap = new HashMap<>();
    static {
        igRespMap.put("0 IG OFF", false);
        igRespMap.put("1 IG  ON", true);
    }
    private static final Map<String, Boolean> igmRespMap = new HashMap<>();
    static {
        igmRespMap.put("0    ALL", false);
        igmRespMap.put("1     IG", true);
    }
    private static final Map<String, Boolean> iodRespMap = new HashMap<>();
    static {
        iodRespMap.put("0   OFF", false);
        iodRespMap.put("1   ON", true);
    }
    private static final Map<String, Boolean> dgRespMap = new HashMap<>();
    static {
        dgRespMap.put("0 DG OFF", false);
        dgRespMap.put("1 DG  ON", true);
    }
    private static final Map<String, Integer> reRespMap = new HashMap<>();
    static {
        reRespMap.put("0 IG OFF", GMODE_OFF);
        reRespMap.put("15MA EM", GMODE_DEGAS);
        reRespMap.put("4.0MA EM", GMODE_HIGH);
        reRespMap.put(".02MA EM", GMODE_LOW);
    }
    private static final Map<Integer, String> unitMap = new HashMap<>();
    static {
        unitMap.put(UNIT_TORR, "SUT");
        unitMap.put(UNIT_MBAR, "SUM");
        unitMap.put(UNIT_PA, "SUP");
    }
    private static final Map<Integer, String> dispMap = new HashMap<>();
    static {
        dispMap.put(DISP_ABS, "SDA");
        dispMap.put(DISP_DIFF, "SDD");
        dispMap.put(DISP_BOTH, "SDB");
    }
    private static final Map<Integer, String> fmodeMap = new HashMap<>();
    static {
        fmodeMap.put(FMODE_MAN, "SFMAN");
        fmodeMap.put(FMODE_ALT, "SFALT");
        fmodeMap.put(FMODE_AUTO, "SFAUTO");
    }

    private static final Map<String, OpenDesc> openMap = new HashMap<>();
    private OpenDesc desc;
    private String addr;
    private String cmndPfx;
    private HardComm comm = new Comm();


    /**
     *  Opens a connection.
     *
     *  @param  ident     The serial port name
     *  @param  param     The parameter (baud rate)
     *  @param  address   The device address
     *  @throws  DriverException
     */
    public void open(String ident, int param, int address) throws DriverException
    {
        if (desc != null) {
            throw new DriverException("Connection already open");
        }
        desc = getOpen(ident, param);
        setAddress(address);
        cmndPfx = "#" + addr;
    }

    @Deprecated
    public void open(ConnType connType, String ident, int param, int address) throws DriverException
    {
        open(ident, param, address);
    }


    /**
     *  Closes the connection.
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException
    {
        checkOpen();
        dropOpen(desc);
        desc = null;
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  timeout  The timeout (secs).  0 means no timeout.
     *  @throws  DriverException
     */
    public void setTimeout(double timeout) throws DriverException
    {
        checkOpen();
        desc.asc.setTimeout(timeout);
    }


    /**
     *  Sets the pressure unit.
     *
     *  @param  unit  The pressure unit: TORR, MBAR or PA
     *  @throws  DriverException
     */
    public void setPressureUnit(int unit) throws DriverException
    {
        checkOpen();
        String cmnd = unitMap.get(unit);
        if (cmnd == null) {
            throw new DriverException("Invalid pressure unit: " + unit);
        }
        write(cmnd);
    }


    /**
     *  Gets the pressure unit.
     *
     *  @return  The pressure unit
     *  @throws  DriverException
     */
    public int getPressureUnit() throws DriverException
    {
        checkOpen();
        String resp = read("RU").trim();
        Integer unit = ruRespMap.get(resp);
        if (unit == null) {
            throwBadResponse(resp);
        }
        return unit;
    }


    /**
     *  Sets the pressure display type.
     *
     *  @param  disp  The pressure display type: ABS, DIFF or BOTH
     *  @throws  DriverException
     */
    public void setPressureDisp(int disp) throws DriverException
    {
        checkOpen();
        String cmnd = dispMap.get(disp);
        if (cmnd == null) {
            throw new DriverException("Invalid pressure display type: " + disp);
        }
        write(cmnd);
    }


    /**
     *  Reads the vacuum pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    public double readPressure() throws DriverException
    {
        checkOpen();
        double value = readDouble("RD");
        return value == UNAVAIL_VALUE ? Double.NaN : value;
    }


    /**
     *  Reads the differential pressure.
     *
     *  @return  The pressure
     *  @throws  DriverException
     */
    public double readDiffPressure() throws DriverException
    {
        checkOpen();
        double value = readDouble("RDD");
        return value == UNAVAIL_VALUE ? Double.NaN : value;
    }


    /**
     *  Toggles protected function lock.
     *
     *  @return  Whether functions are locked
     *  @throws  DriverException
     */
    public boolean toggleLock() throws DriverException
    {
        checkOpen();
        String resp = read("TLU").trim();
        Boolean locked = tlRespMap.get(resp);
        if (locked == null) {
            throwBadResponse(resp);
        }
        return locked;
    }


    /**
     *  Sets the state of the protected function lock.
     *
     *  @param  on  Whether lock is to be set on
     *  @throws  DriverException
     */
    public void setLock(boolean on) throws DriverException
    {
        if (toggleLock() != on) {
            toggleLock();
        }
    }


    /**
     *  Unlocks protected functions (one time).
     *
     *  @throws  DriverException
     */
    public void unlock() throws DriverException
    {
        checkOpen();
        write("UNL");
    }


    /**
     *  Sets the address offset.
     *
     *  @param  offset  The first hex digit of the address: 0 - 3
     *  @throws  DriverException
     */
    public void setAddressOffset(int offset) throws DriverException
    {
        checkOpen();
        write("SA" + offset + "0");
    }


    /**
     *  Sets the baud rate.
     *
     *  @param  baud  The baud rate: 1200, 2400, 4800, 9600, 19200 or 38400
     *  @throws  DriverException
     */
    public void setBaudRate(int baud) throws DriverException
    {
        checkOpen();
        write("SB" + baud);
    }


    /**
     *  Sets the filament mode.
     *
     *  @param  mode  The filament mode: MAN, ALT or AUTO
     *  @throws  DriverException
     */
    public void setFilamentMode(int mode) throws DriverException
    {
        checkOpen();
        String cmnd = fmodeMap.get(mode);
        if (cmnd == null) {
            throw new DriverException("Invalid filament mode: " + mode);
        }
        write(cmnd);
    }


    /**
     *  Gets the filament status.
     *
     *  @return  Which filament(s) are on: 1, 2 or 3 (both)
     *  @throws  DriverException
     */
    public int getFilamentStatus() throws DriverException
    {
        checkOpen();
        String resp = read("RF").trim();
        Integer status = rfRespMap.get(resp);
        if (status == null) {
            throwBadResponse(resp);
        }
        return status;
    }


    /**
     *  Turns the micro-ion gauge on or off.
     *
     *  @param  on  Whether to turn the gauge on
     *  @throws  DriverException
     */
    public void setMiGaugeOn(boolean on) throws DriverException
    {
        checkOpen();
        write("IG" + (on ? "1" : "0"));
    }


    /**
     *  Tests whether the micro-ion gauge is on or off.
     *
     *  @return  Whether the gauge is on
     *  @throws  DriverException
     */
    public boolean isMiGaugeOn() throws DriverException
    {
        checkOpen();
        String resp = read("IGS").trim();
        Boolean on = igRespMap.get(resp);
        if (on == null) {
            throwBadResponse(resp);
        }
        return on;
    }


    /**
     *  Turns on or off the pressure indication when the micro-ion gauge is off.
     *
     *  @param  on  Whether to turn the pressure indication on
     *  @throws  DriverException
     */
    public void setPressureIndOn(boolean on) throws DriverException
    {
        checkOpen();
        write("IGM" + (on ? "1" : "0"));
    }


    /**
     *  Tests whether the pressure indication is on or off.
     *
     *  @return  Whether the pressure indication is on
     *  @throws  DriverException
     */
    public boolean isPressureIndOn() throws DriverException
    {
        checkOpen();
        String resp = read("IGMS").trim();
        Boolean on = igmRespMap.get(resp);
        if (on == null) {
            throwBadResponse(resp);
        }
        return on;
    }


    /**
     *  Turns the micro-ion gauge delay on or off.
     *
     *  @param  on  Whether to turn the gauge delay on
     *  @throws  DriverException
     */
    public void setMiDelayOn(boolean on) throws DriverException
    {
        checkOpen();
        write("IOD" + (on ? "1" : "0"));
    }


    /**
     *  Tests whether the micro-ion gauge delay is on or off.
     *
     *  @return  Whether the gauge delay is on
     *  @throws  DriverException
     */
    public boolean isMiDelayOn() throws DriverException
    {
        checkOpen();
        String resp = read("IOD").trim();
        Boolean on = iodRespMap.get(resp);
        if (on == null) {
            throwBadResponse(resp);
        }
        return on;
    }


    /**
     *  Sets the micro-ion gauge delay time.
     *
     *  @param  time  The degas time (0 - 600 secs)
     *  @throws  DriverException
     */
    public void setMiDelayTime(int time) throws DriverException
    {
        checkOpen();
        write("IDT" + time);
    }


    /**
     *  Gets the micro-ion gauge delay time.
     *
     *  @return  The degas time
     *  @throws  DriverException
     */
    public int getMiDelayTime() throws DriverException
    {
        checkOpen();
        Integer value = null;
        String resp = read("IDT");
        if (resp.substring(4).equals("  IDT")) {
            try {
                value = Integer.valueOf(resp.substring(0, 4).trim());
            }
            catch (NumberFormatException e) {
            }
        }
        if (value == null) {
            throwBadResponse(resp);
        }
        return value;
    }


    /**
     *  Turns a micro-ion gauge degas cycle on or off.
     *
     *  @param  on  Whether to turn the gauge degas on
     *  @throws  DriverException
     */
    public void setMiDegasOn(boolean on) throws DriverException
    {
        checkOpen();
        write("DG" + (on ? "1" : "0"));
    }


    /**
     *  Tests whether a micro-ion gauge degas is on or off.
     *
     *  @return  Whether the gauge degas is on
     *  @throws  DriverException
     */
    public boolean isMiDegasOn() throws DriverException
    {
        checkOpen();
        String resp = read("DGS").trim();
        Boolean on = dgRespMap.get(resp);
        if (on == null) {
            throwBadResponse(resp);
        }
        return on;
    }


    /**
     *  Sets the micro-ion gauge degas time.
     *
     *  @param  time  The degas time (10 - 120 secs)
     *  @throws  DriverException
     */
    public void setMiDegasTime(int time) throws DriverException
    {
        checkOpen();
        write("DGT" + time);
    }


    /**
     *  Gets the micro-ion gauge degas time.
     *
     *  @return  The degas time
     *  @throws  DriverException
     */
    public int getMiDegasTime() throws DriverException
    {
        checkOpen();
        Integer value = null;
        String resp = read("DGT");
        if (resp.substring(4).equals("  DGT")) {
            try {
                value = Integer.valueOf(resp.substring(0, 4).trim());
            }
            catch (NumberFormatException e) {
            }
        }
        if (value == null) {
            throwBadResponse(resp);
        }
        return value;
    }


    /**
     *  Sets the micro-ion current switch pressure.
     *
     *  @param  press  The switch pressure
     *  @throws  DriverException
     */
    public void setMiSwitchPressure(double press) throws DriverException
    {
        checkOpen();
        write(String.format("SER %.2E", press));
    }


    /**
     *  Gets the micro-ion current switch pressure.
     *
     *  @return  The switch pressure
     *  @throws  DriverException
     */
    public double getMiSwitchPressure() throws DriverException
    {
        checkOpen();
        return readDouble("SER");
    }


    /**
     *  Gets the micro-ion gauge current.
     *
     *  @return  The gauge current (mA)
     *  @throws  DriverException
     */
    public double getMiGaugeCurrent() throws DriverException
    {
        checkOpen();
        Double value = null;
        String resp = read("RE");
        if (resp.substring(4).equals("MA EM")) {
            try {
                value = Double.valueOf(resp.substring(0, 4).trim());
            }
            catch (NumberFormatException e) {
            }
        }
        else if (resp.equals(" 0 IG OFF")) {
            value = 0.0;
        }
        if (value == null) {
            throwBadResponse(resp);
        }
        return value;
    }


    /**
     *  Gets the micro-ion gauge mode.
     *
     *  @return  The gauge mode (OFF, DEGAS, HIGH or LOW)
     *  @throws  DriverException
     */
    public int getMiGaugeMode() throws DriverException
    {
        checkOpen();
        String resp = read("RE").trim();
        Integer value = reRespMap.get(resp);
        if (value == null) {
            throwBadResponse(resp);
        }
        return value;
    }


    /**
     *  Calibrates at atmospheric pressure.
     *
     *  @throws  DriverException
     */
    public void calibrateAtmospheric() throws DriverException
    {
        checkOpen();
        write("TS");
    }


    /**
     *  Calibrates at vacuum pressure.
     *
     *  @throws  DriverException
     */
    public void calibrateVacuum() throws DriverException
    {
        checkOpen();
        write("TZ");
    }


    /**
     *  Sets the atmospheric pressure.
     *
     *  @param  press  The atmospheric pressure value to set, or negative for actual
     *  @throws  DriverException
     */
    public void setAtmospheric(double press) throws DriverException
    {
        checkOpen();
        write("ATM " + (press < 0 ? "ACTUAL" : press));
    }


    /**
     *  Gets the atmospheric pressure.
     *
     *  @return  The atmospheric pressure or -1.0 for actual
     *  @throws  DriverException
     */
    public double getAtmospheric() throws DriverException
    {
        checkOpen();
        String resp = read("ATMS").trim();
        Double value = null;
        if (resp.equals("ACTUAL")) {
            value = -1.0;
        }
        else {
            try {
                value = Double.valueOf(resp);
            }
            catch (NumberFormatException e) {
            }
        }
        if (value == null) {
            throwBadResponse(resp);
        }
        return value;
    }


    /**
     *  Sets a trip relay set point.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  type   The set point type (1 = activation, 0 = deactivation)
     *  @param  value  The set point value
     *  @throws  DriverException
     */
    public void setRelayTrip(int relay, int type, double value) throws DriverException
    {
        checkRelayNumber(relay);
        write("PC" + relay + ((type & RELAY_ACT) != 0 ? "A" : "D") + value);
    }


    /**
     *  Gets a trip relay set point.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @param  type   The set point type (1 = activation, 0 = deactivation)
     *  @return  The set point value
     *  @throws  DriverException
     */
    public double getRelayTrip(int relay, int type) throws DriverException
    {
        checkRelayNumber(relay);
        return readDouble("PC" + relay + ((type & RELAY_ACT) != 0 ? "A" : "D"));
    }


    /**
     *  Assigns a trip relay to a vacuum type.
     *
     *  @param  relay   The relay number (1 - 3)
     *  @param  assign  The assignment: 0 = absolute, 1 = differential
     *  @throws  DriverException
     */
    public void assignRelay(int relay, int assign) throws DriverException
    {
        checkRelayNumber(relay);
        char[] assigns = getRelayAssigns();
        assigns[relay - 1] = (assign & RELAY_DIFF) == 0 ? 'A' : 'D';
        write("PCG" + new String(assigns));
    }


    /**
     *  Gets the assignment for a trip relay.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  The assignment: 0 = absolute, 1 = differential
     *  @throws  DriverException
     */
    public int getRelayAssignment(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        return getRelayAssigns()[relay - 1] == 'A' ? RELAY_ABS : RELAY_DIFF;
    }


    /**
     *  Enables/disables a trip relay.
     *
     *  @param  relay   The relay number (1 - 3)
     *  @param  enable  Whether to enable
     *  @throws  DriverException
     */
    public void enableRelay(int relay, boolean enable) throws DriverException
    {
        checkRelayNumber(relay);
        char[] states = getRelayStates("PCE");
        states[relay - 1] = enable ? '1' : '0';
        write("PCE" + new String(states));
    }


    /**
     *  Gets whether a trip relay is enabled.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  Whether enabled
     *  @throws  DriverException
     */
    public boolean isRelayEnabled(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        return getRelayStates("PCE")[relay - 1] == '1';
    }


    /**
     *  Gets whether a trip relay is active.
     *
     *  @param  relay  The relay number (1 - 3)
     *  @return  Whether active
     *  @throws  DriverException
     */
    public boolean isRelayActive(int relay) throws DriverException
    {
        checkRelayNumber(relay);
        return getRelayStates("RPCS")[relay - 1] == '1';
    }


    /**
     *  Gets the array of status strings.
     *
     *  @return  The status strings
     *  @throws  DriverException
     */
    public String[] getStatusStrings() throws DriverException
    {
        checkOpen();
        List<String> list = new ArrayList<>();
        String first = null;
        while (true) {
            String resp = read("RS").trim();
            String code = resp.substring(0, 2);
            if (code.equals("08")) continue;  // power on
            if (first == null) {
                first = resp;
            }
            else {
                if (resp.equals(first)) break;
            }
            if (!code.equals("00")) {
                list.add(resp);
            }
        }
        return list.toArray(new String[list.size()]);
    }


    /**
     *  Gets the status word.
     *
     *  @return  The status word
     *  @throws  DriverException
     */
    public int getStatusWord() throws DriverException
    {
        checkOpen();
        String resp = read("RSX").trim();
        Integer value = null;
        try {
            value = Integer.parseInt(resp, 16);
        }
        catch (NumberFormatException e) {
        }
        if (value == null) {
            throwBadResponse(resp);
        }
        return value;
    }


    /**
     *  Gets the firmware version.
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    public String getFwVersion() throws DriverException
    {
        checkOpen();
        return read("VER").trim();
    }


    /**
     *  Resets to power-up state.
     *
     *  @throws  DriverException
     */
    public void reset() throws DriverException
    {
        checkOpen();
        desc.asc.write("RST");
    }


    /**
     *  Sets communications defaults.
     *
     *  @throws  DriverException
     */
    public void setCommDefaults() throws DriverException
    {
        checkOpen();
        desc.asc.write("yuiop");
    }


    /**
     *  Sets factory defaults.
     *
     *  @throws  DriverException
     */
    public void setDefaults() throws DriverException
    {
        checkOpen();
        write("FAC");
    }


    /**
     *  Sets the hardware communications handler.
     * 
     *  Used by GP835
     * 
     *  @param  commHand  The communications handler object
     */
    void setCommHandler(HardComm commHand)
    {
        comm = commHand;
    }


    /**
     *  Sets the RS-485 address.
     * 
     *  Used internally and by GP835
     * 
     *  @param  address  The address (0 - 63)
     */
    void setAddress(int address)
    {
        addr = String.format("%02d", address);
    }


    /**
     *  Sends a command.
     *
     *  @param  cmnd  The command
     *  @throws  DriverException
     */
    private void write(String cmnd) throws DriverException
    {
        String resp = read(cmnd);
        if (!resp.equals(" PROGM OK")) {
            throwBadResponse(resp);
        }
    }


    /**
     *  Sends a command and returns the response.
     *
     *  @param  cmnd  The command
     *  @return  The response, stripped of its header
     *  @throws  DriverException
     */
    private String read(String cmnd) throws DriverException
    {
        String reply = comm.readGP390(cmnd);
        String rStart = reply.substring(0, 1);
        if (!rStart.equals("*") && !rStart.equals("?")) {
            throw new DriverException("Invalid reply start: " + rStart);
        }
        String rAddr = reply.substring(1, 3);
        if (!rAddr.equals(addr)) {
            throw new DriverException("Invalid reply address: " + rAddr);
        }
        String resp = reply.substring(3);
        if (rStart.equals("*")) {
            return resp;
        }

        switch (resp.trim()) {

        case "RANGE ER":
            throw new DriverException("Range error");

        case "SYNTX ER":
            throw new DriverException("Syntax error");

        case "LOCKED":
            throw new DriverException("Function locked");

        case "INVALID":
            throw new DriverException("Invalid command");

        default:
            throw new DriverException("Unrecognized error: " + resp);
        }
    }


    /**
     *  Sends a command and returns the double response.
     *
     *  @param  cmnd  The command
     *  @return  The response, as a double
     *  @throws  DriverException
     */
    private double readDouble(String cmnd) throws DriverException
    {
        String resp = read(cmnd);
        try {
            return Double.valueOf(resp.trim());
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid double value: " + resp);
        }
    }


    /**
     *  Throws an "unrecognized response" exception.
     *
     *  @param  resp  The unrecognized response
     *  @throws  DriverException
     */
    void throwBadResponse(String resp) throws DriverException {
        throw new DriverException("Unrecognized response: " + resp);
    }


    /**
     *  Checks a relay number for validity.
     *
     *  @param  relay  The relay number
     *  @throws  DriverException
     */
    void checkRelayNumber(int relay) throws DriverException {
        if (relay < RELAY_MIN || relay > RELAY_MAX) {
            throw new DriverException("Invalid relay number: " + relay);
        }
    }


    /**
     *  Gets relay states.
     *
     *  @param  command  The command to send
     *  @return  The relay state array
     *  @throws  DriverException
     */
    private char[] getRelayStates(String command) throws DriverException
    {
        String resp = read(command);
        if (!resp.matches(" [01][01][01]")) {
            throwBadResponse(resp);
        }
        return resp.substring(1).toCharArray();
    }


    /**
     *  Gets relay assigns.
     *
     *  @return  The relay assignment array
     *  @throws  DriverException
     */
    private char[] getRelayAssigns() throws DriverException
    {
        String resp = read("PCG");
        if (!resp.matches(" [AD][AD][AD]")) {
            throwBadResponse(resp);
        }
        return resp.substring(1).toCharArray();
    }


    /**
     *  Gets the open descriptor.
     *
     *  Multiple RS485 devices may share the same physical connection.  If a
     *  connection is already open for the device, its descriptor is returned;
     *  otherwise a new connection is made.
     *
     *  @param  connType  The connection type
     *  @param  ident     The device identifier
     *  @param  param     The device parameter
     *  @return  The open descriptor
     *  @throws  DriverException
     */
    private static synchronized OpenDesc getOpen(String ident, int param)
        throws DriverException
    {
        String key = ident;
        OpenDesc desc = openMap.get(key);
        if (desc == null) {
            desc = new OpenDesc();
            desc.asc = new Ascii();
            desc.asc.setOptions(Ascii.Option.NO_NET);
            openMap.put(key, desc);
        }
        if (desc.refCount <= 0) {
            desc.asc.open(Ascii.ConnType.SERIAL, ident, param);
            desc.asc.setCommandTerm(Ascii.Terminator.CR);
            desc.asc.setResponseTerm(Ascii.Terminator.CR);
        }
        desc.refCount++;

        return desc;
    }


    /**
     *  Drops an open descriptor.
     *
     *  @param  desc  The open descriptor
     *  @throws  DriverException
     */
    private static synchronized void dropOpen(OpenDesc desc) throws DriverException
    {
        if (--desc.refCount <= 0) {
            desc.asc.close();
        }
    }


    /**
     *  Checks whether a connection is open.
     *
     *  @throws  DriverException
     */
    private void checkOpen() throws DriverException
    {
        comm.checkOpenGP390();
    }

}
