package org.lsst.ccs.drivers.mks;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 * Program to test the GP835 device driver
 *
 * @author Homer
 */
public class TestGP835 extends TestScpi {

    protected GP835 mydev;

    public TestGP835() {
        super(new GP835());
        mydev = (GP835)scpi;
    }

    @Command(description = "Open a connection to the device")
    public void open(@Argument(description = "Type of connection") GP835.ConnType connType,
                     @Argument(description = "The ID of the device to open") String devId) throws DriverException {
        mydev.open(connType, devId, 0);
    }

    @Command(description = "Get the pressure")
    public double getPressure() throws DriverException {
        return mydev.getPressure();
    }

    @Command(description = "Get the pressure directly")
    public double getPressureGP390() throws DriverException {
        return mydev.getPressureGP390();
    }

    @Command(description = "Monitor the pressure")
    public void monPressure() throws DriverException {
	long tstart = System.currentTimeMillis();
        while(true) {
            double pres = mydev.getPressure();
            long time = System.currentTimeMillis();
            System.out.println("time = "+(time/1000.0)+" delta_t = "+(time-tstart)/1000.0+" Pressure = "+pres);
            sleep(0.5);
        }
    }

    private void sleep(double secs) {
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            System.out.println("Rude awakening!" + ex);
        }
    }

    @Command(description = "Get the filament power")
    public double getFilamentPower() throws DriverException {
        return mydev.getFilamentPower();
    }

    @Command(description = "Get the test report")
    public String getTestReport() throws DriverException {
        return mydev.getTestReport();
    }

    @Command(description = "Get the mass spectrometer scan results")
    public String getMSP() throws DriverException {
        return mydev.getMSP();
    }

    @Command(description = "Set the gauge on")
    public void setGaugeOn() throws DriverException {
        mydev.setGaugeOn();
    }

    @Command(description = "Set the gauge off")
    public void setGaugeOff() throws DriverException {
        mydev.setGaugeOff();
    }

    @Command(description = "Get whether the gauge is on")
    public boolean isGaugeOn() throws DriverException {
        return mydev.isGaugeOn();
    }

    @Command(description = "Set the filament on")
    public void setFilamentOn() throws DriverException {
        mydev.setFilamentOn();
    }

    @Command(description = "Set the filement off")
    public void setFilamentOff() throws DriverException {
        mydev.setFilamentOff();
    }

    @Command(description = "Get whether the filament is on")
    public boolean isFilamentOn() throws DriverException {
        return mydev.isFilamentOn();
    }

    @Command(description = "Get the selected instrument")
    public String getInstrument() throws DriverException {
        return mydev.getInstrument();
    }

    @Command(description="Set a relay trip point")
    public void setRelayTrip(@Argument(description="Relay number (1-3)") int relay,
                             @Argument(description="Trip point type: 1=activate, 0=deactivate") int type,
                             @Argument(description="Trip point value") double value) throws DriverException {
        mydev.setRelayTrip(relay, type, value);
    }

    @Command(description="Get a relay trip point")
    public double getRelayTrip(@Argument(description="Relay number (1-3)") int relay,
                               @Argument(description="Trip point type: 1=activate, 0=deactivate") int type) throws DriverException {
        return mydev.getRelayTrip(relay, type);
    }

    @Command(description="Assign a relay to vacuum type")
    public void assignRelay(@Argument(description="Relay number (1-3)") int relay,
                            @Argument(description="Vacuum type: 0=absolute, 1=differential") int assign) throws DriverException {
        mydev.assignRelay(relay, assign);
    }

    @Command(description="Get a relay assignment")
    public int getRelayAssignment(@Argument(description="Relay number (1-3)") int relay) throws DriverException {
        return mydev.getRelayAssignment(relay);
    }

    @Command(description="Enable a relay")
    public void enableRelay(@Argument(description="Relay number (1-3)") int relay,
                            @Argument(description="Whether to enable") boolean enable) throws DriverException {
        mydev.enableRelay(relay, enable);
    }

    @Command(description="Get whether relay is enabled")
    public boolean isRelayEnabled(@Argument(description="Relay number (1-3)") int relay) throws DriverException {
        return mydev.isRelayEnabled(relay);
    }

    @Command(description="Get whether relay is active")
    public boolean isRelayActive(@Argument(description="Relay number (1-3)") int relay) throws DriverException {
        return mydev.isRelayActive(relay);
    }

    @Command(description = "Send command to the GP390")
    public String readGP390(@Argument(description = "Command") String command,
                            @Argument(description = "Arguments") String... args) throws DriverException {
        return mydev.readGP390(makeCommandString(command, args));
    }

}
