package org.lsst.ccs.drivers.mks;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.Model9XX.*;

/**
 *  Program to test the MKS models 972B, 974B & 925 vacuum gauge driver
 *
 *  @author Owen Saxton
 */
public class Test9XX {

    protected enum OnOff {ON, OFF}

    private final Model9XX[] devs = {new Model9XX(), new Model9XX(), new Model9XX(),
                                     new Model9XX(), new Model9XX()};
    private Model9XX dev = devs[0];
    private int devNumber = 0;


    @Command(description="Set the current device number")
    public void setDeviceNum(@Argument(description="The device number") int number) throws DriverException
    {
        if (number < 0 || number >= devs.length) {
            throw new DriverException("Invalid device number");
        }
        devNumber = number;
        dev = devs[devNumber];
    }

    @Command(description="Show the current device number")
    public int showDeviceNum()
    {
        return devNumber;
    }

    @Command(description="Open connection to device")
    public void open(@Argument(description="Connection type") Model9XX.ConnType type,
                     @Argument(description="Identification") String ident) throws DriverException
    {
        dev.open(type, ident, -1);
    }

    @Command(description="Open connection to device")
    public void open(@Argument(description="Connection type") Model9XX.ConnType type,
                     @Argument(description="Identification") String ident,
                     @Argument(description="Device address") int addr) throws DriverException
    {
        dev.open(type, ident, addr);
    }

    @Command(description="Open connection to device")
    public void open(@Argument(description="Connection type") Model9XX.ConnType type,
                     @Argument(description="Identification") String ident,
                     @Argument(description="Baud rate") int baud,
                     @Argument(description="Device address") int addr) throws DriverException
    {
        dev.open(type, ident, baud, addr);
    }

    @Command(description="Close the connection")
    public void close() throws DriverException
    {
        dev.close();
    }

    @Command(description="Write a command")
    public void write(@Argument(description="Command") String cmnd) throws DriverException
    {
        dev.writeCmnd(cmnd);
    }

    @Command(description="Read response from a command")
    public String read(@Argument(description="Command") String cmnd) throws DriverException
    {
        return dev.readString(cmnd);
    }

    @Command(description="Set protected settings lock")
    public void setLock(@Argument(description="ON or OFF") OnOff onoff) throws DriverException
    {
        dev.setLock(onoff == OnOff.ON);
    }

    @Command(description="Set factory defaults")
    public void setDefaults() throws DriverException
    {
        dev.setDefaults();
    }

    @Command(description="Set factory defaults")
    public void setDefaults(@Argument(description="Default item to set") Default item) throws DriverException
    {
        dev.setDefaults(item);
    }

    @Command(description="Show device type")
    public String showDeviceType() throws DriverException
    {
        return dev.getDeviceType();
    }

    @Command(description="Show firmware version")
    public String showFwVersion() throws DriverException
    {
        return dev.getFirmwareVersion();
    }

    @Command(description="Show hardware version")
    public String showHwVersion() throws DriverException
    {
        return dev.getHardwareVersion();
    }

    @Command(description="Show manufacturer")
    public String showManufacturer() throws DriverException
    {
        return dev.getManufacturer();
    }

    @Command(description="Show model")
    public String showModel() throws DriverException
    {
        return dev.getModel();
    }

    @Command(description="Show part number")
    public String showPart() throws DriverException
    {
        return dev.getPartNumber();
    }

    @Command(description="Show serial number")
    public String showSerial() throws DriverException
    {
        return dev.getSerialNumber();
    }

    @Command(description="Set user tag")
    public void setUserTag(@Argument(description="The tag string") String tag) throws DriverException
    {
        dev.setUserTag(tag);
    }

    @Command(description="Show user tag")
    public String showUserTag() throws DriverException
    {
        return dev.getUserTag();
    }

    @Command(description="Show transducer status string")
    public String showTransStatus() throws DriverException
    {
        return dev.getTransStatus();
    }

    @Command(description="Show transducer status word")
    public char showStatus() throws DriverException
    {
        return dev.getStatus();
    }

    @Command(description="Show unit on time")
    public String showOnTime() throws DriverException
    {
        return dev.getTimeOn() + " hours";
    }

    @Command(description="Show cold cathode on time")
    public String showSensorTime() throws DriverException
    {
        return dev.getSensorTimeOn() + " hours";
    }

    @Command(description="Show cold cathose pressure dose")
    public String showSensorDose() throws DriverException
    {
        return dev.getSensorDose() + " " + getUnitsString() + " hours";
    }

    @Command(description="Set cold cathode dose limit")
    public void setDoseLimit(@Argument(description="Dose limit (press-unit hours)") double dose) throws DriverException
    {
        dev.setSensorDoseLimit(dose);
    }

    @Command(description="Show cold cathode dose limit")
    public String showDoseLimit() throws DriverException
    {
        return dev.getSensorDoseLimit() + " " + getUnitsString() + " hours";
    }

    @Command(description="Set pressure units")
    public void setUnits(@Argument(description="Pressure units") Unit units) throws DriverException
    {
        dev.setPressureUnit(units);
    }

    @Command(description="Show pressure units")
    public Unit showUnits() throws DriverException
    {
        return dev.getPressureUnit();
    }

    @Command(description="Set calibration gas type")
    public void setGasType(@Argument(description="Gas type") Gas gas) throws DriverException
    {
        dev.setCalibrationGas(gas);
    }

    @Command(description="Show calibration gas type")
    public Gas showGasType() throws DriverException
    {
        return dev.getCalibrationGas();
    }

    @Command(description="Read vacuum pressure")
    public String readPressure() throws DriverException
    {
        return dev.readPressure() + " " + getUnitsString();
    }

    @Command(description="Read vacuum pressure")
    public String readPressure(@Argument(description="Sensor to use") Sensor sensor) throws DriverException
    {
        return dev.readPressure(sensor) + " " + getUnitsString();
    }

    @Command(description="Set atmospheric pressure")
    public void setAtmospheric(@Argument(description="Atmospheric pressure") double press) throws DriverException
    {
        dev.calibrateAtmospheric(press);
    }

    @Command(description="Show atmospheric pressure difference")
    public String showAtmospheric() throws DriverException
    {
        return dev.getAtmospheric() + " " + getUnitsString();
    }

    @Command(description="Set full-scale pressure")
    public void setFullScale(@Argument(description="Pressure") double press) throws DriverException
    {
        dev.calibrateFullScale(press);
    }

    @Command(description="Show full-scale pressure difference")
    public String showFullScale() throws DriverException
    {
        return dev.getFullScale()+ " " + getUnitsString();
    }

    @Command(description="Read MicroPirani temperature")
    public String readTemperature() throws DriverException
    {
        return dev.readTemperature() + " C";
    }

    @Command(description="Set relay trip point")
    public void setRelayTrip(@Argument(description="Relay number") int relay,
                             @Argument(description="Trip point value") double value) throws DriverException
    {
        dev.setRelayTrip(relay, value);
    }

    @Command(description="Show relay trip point")
    public String showRelayTrip(@Argument(description="Relay number") int relay) throws DriverException
    {
        return dev.getRelayTrip(relay) + " " + getUnitsString();
    }

    @Command(description="Set relay hysteresis")
    public void setRelayHyst(@Argument(description="Relay number") int relay,
                             @Argument(description="Hysteresis value") double value) throws DriverException
    {
        dev.setRelayHyst(relay, value);
    }

    @Command(description="Show relay hysteresis")
    public String showRelayHyst(@Argument(description="Relay number") int relay) throws DriverException
    {
        return String.valueOf(dev.getRelayHyst(relay));
    }

    @Command(description="Set relay trip direction")
    public void setRelayDirn(@Argument(description="Relay number") int relay,
                             @Argument(description="Trip direction") Direction dirn) throws DriverException
    {
        dev.setRelayDirection(relay, dirn);
    }

    @Command(description="Show relay trip direction")
    public Direction showRelayDirn(@Argument(description="Relay number") int relay) throws DriverException
    {
        return dev.getRelayDirection(relay);
    }

    @Command(description="Enable a relay")
    public void setRelayEnable(@Argument(description="Relay number") int relay,
                               @Argument(description="Enable type") Enable enable) throws DriverException
    {
        dev.setRelayEnable(relay, enable);
    }

    @Command(description="Show whether relay enabled")
    public Enable showRelayEnable(@Argument(description="Relay number") int relay) throws DriverException
    {
        return dev.getRelayEnable(relay);
    }

    @Command(description="Show whether relay is active")
    public boolean isRelayActive(@Argument(description="Relay number") int relay) throws DriverException
    {
        return dev.isRelayActive(relay);
    }

    @Command(description="Set relay delayed state")
    public void setRelayDelayed(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setRelayDelayed(onOff == OnOff.ON);
    }

    @Command(description="Show relay delayed state")
    public OnOff showRelayDelayed() throws DriverException
    {
        return getOnOff(dev.isRelayDelayed());
    }

    @Command(description="Set analog output parameters")
    public void setAnalogOut(@Argument(description="Analog channel number (1 - 2)") int chan,
                             @Argument(description="Sensor to use") Sensor sensor,
                             @Argument(description="Analog out curve number") int curve) throws DriverException
    {
        dev.setAnalogOut(chan, sensor, curve);
    }

    @Command(description="Show analog output setting")
    public String showAnalogOut(@Argument(description="Analog channel number (1 - 2)") int chan) throws DriverException
    {
        return dev.getAnalogOut(chan);
    }

    @Command(description="Calibrate cold cathopde vacuum")
    public void calibrateVacuumCC() throws DriverException
    {
        dev.calibrateVacuumCC();
    }

    @Command(description="Calibrate micro-pirani vacuum")
    public void calibrateVacuumMP() throws DriverException
    {
        dev.calibrateVacuumMP();
    }

    @Command(description="Set auto-zero limit")
    public void setAutoZeroLimit(@Argument(description="Pressure") double press) throws DriverException
    {
        dev.setAutoZeroLimit(press);
    }

    @Command(description="Show auto-zero limit")
    public String showAutoZeroLimit() throws DriverException
    {
        return dev.getAutoZeroLimit() + " " + getUnitsString();
    }

    @Command(description="Set cold cathode power state")
    public void setCCPower(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setCCPower(onOff == OnOff.ON);
    }

    @Command(description="Show cold cathods power state")
    public OnOff showCCPower() throws DriverException
    {
        return getOnOff(dev.isCCPowerOn());
    }

    @Command(description="Set cold cathode auto power on state")
    public void setCCAutoPower(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setCCAutoPower(onOff == OnOff.ON);
    }

    @Command(description="Show cold cathods auto power on state")
    public OnOff showCCAutoPower() throws DriverException
    {
        return getOnOff(dev.isCCAutoPowerOn());
    }

    @Command(description="Set cold cathode protection state")
    public void setCCProtection(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setCCProtection(onOff == OnOff.ON);
    }

    @Command(description="Set cold cathode protection delay")
    public void setCCProtDelay(@Argument(description="Delay time (sec)") int delay) throws DriverException
    {
        dev.setCCProtDelay(delay);
    }

    @Command(description="Show cold cathods protection delay")
    public String showCCProtDelay() throws DriverException
    {
        return dev.getCCProtDelay() + " sec";
    }

    @Command(description="Set cold cathode on pressure")
    public void setCCOnPressure(@Argument(description="Pressure") double press) throws DriverException
    {
        dev.setOnPressureCC(press);
    }

    @Command(description="Show cold cathode on pressure")
    public String showCCOnPressure() throws DriverException
    {
        return dev.getOnPressureCC() + " " + getUnitsString();
    }

    @Command(description="Set cold cathode off pressure")
    public void setCCOffPressure(@Argument(description="Pressure") double press) throws DriverException
    {
        dev.setOffPressureCC(press);
    }

    @Command(description="Show cold cathode off pressure")
    public String showCCOffPressure() throws DriverException
    {
        return dev.getOffPressureCC() + " " + getUnitsString();
    }

    @Command(description="Set high integration pressure")
    public void setHighIntPressure(@Argument(description="Pressure") double press) throws DriverException
    {
        dev.setHighIntPressure(press);
    }

    @Command(description="Show high integration pressure")
    public String showHighIntPressure() throws DriverException
    {
        return dev.getHighIntPressure() + " " + getUnitsString();
    }

    @Command(description="Set low integration pressure")
    public void setLowIntPressure(@Argument(description="Pressure") double press) throws DriverException
    {
        dev.setLowIntPressure(press);
    }

    @Command(description="Show low integration pressure")
    public String showLowIntPressure() throws DriverException
    {
        return dev.getLowIntPressure() + " " + getUnitsString();
    }

    @Command(description="Set delayed response state")
    public void setDelayedResp(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setDelayedResponse(onOff == OnOff.ON);
    }

    @Command(description="Show delayed response state")
    public OnOff showDelayedResp() throws DriverException
    {
        return getOnOff(dev.isResponseDelayed());
    }

    @Command(description="Set identifying state")
    public void setIdentify(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setIdentify(onOff == OnOff.ON);
    }

    @Command(description="Show identifying state")
    public OnOff showIdentify() throws DriverException
    {
        return getOnOff(dev.isIdentifying());
    }

    @Command(description="Set switch enabled state")
    public void setSwitchEnable(@Argument(description="ON or OFF") OnOff onOff) throws DriverException
    {
        dev.setSwitchEnable(onOff == OnOff.ON);
    }

    @Command(description="Show switch enabled state")
    public OnOff showSwitchEnable() throws DriverException
    {
        return getOnOff(dev.isSwitchEnabled());
    }

    @Command(description="Set baud rate")
    public void setBaud(@Argument(description="Baud rate") int baud) throws DriverException
    {
        dev.setBaudRate(baud);
    }

    @Command(description="Show baud rate")
    public int showBaud() throws DriverException
    {
        return dev.getBaudRate();
    }

    @Command(description="Set (change) device address")
    public void setAddress(@Argument(description="Address (1 - 253)") int address) throws DriverException
    {
        dev.changeAddress(address);
    }

    @Command(description="Set response timeout")
    public void setTimeout(@Argument(description="Timeout (secs)") double timeout) throws DriverException
    {
        dev.setTimeout(timeout);
    }

    @Command(description="Show read buffer contents")
    public String showReadBuffer()
    {
        return dev.getReadBuffer();
    }

    private String getUnitsString() throws DriverException
    {
        return dev.getPressureUnit().name().toLowerCase();
    }

    private OnOff getOnOff(boolean value)
    {
        return value ? OnOff.ON : OnOff.OFF;
    }

}
