package org.lsst.ccs.drivers.mks;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.Model9XX.*;

/**
 *  Program to test the MKS models 972B, 974B & 925 vacuum gauge driver
 *
 *  @author Owen Saxton
 */
public class Test9XX {

    protected enum OnOff {ON, OFF}

    private final Model9XX[] devs = {new Model9XX(), new Model9XX(), new Model9XX(),
                                     new Model9XX(), new Model9XX()};
    private Model9XX dev = devs[0];
    private int devNumber = 0;


    @Command(name="setdevnum", description="Set the current device number")
    public void setDeviceNum(@Argument(description="The device number")
                             int number) throws DriverException
    {
        if (number < 0 || number >= devs.length) {
            throw new DriverException("Invalid device number");
        }
        devNumber = number;
        dev = devs[devNumber];
    }


    @Command(name="showdevnum", description="Show the current device number")
    public int showDeviceNum()
    {
        return devNumber;
    }


    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Model9XX.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        dev.open(connType, ident, -1);
    }


    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Model9XX.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident,
                     @Argument(name="addr", description="Device address")
                     int addr) throws DriverException
    {
        dev.open(connType, ident, addr);
    }


    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Model9XX.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident,
                     @Argument(name="baud", description="Baud rate")
                     int baud,
                     @Argument(name="addr", description="Device address")
                     int address) throws DriverException
    {
        dev.open(connType, ident, baud, address);
    }


    @Command(name="close", description="Close the connection")
    public void close() throws DriverException
    {
        dev.close();
    }


    @Command(name="write", description="Write a command")
    public void write(@Argument(description="Command")
                      String cmnd) throws DriverException
    {
        dev.writeCmnd(cmnd);
    }


    @Command(name="read", description="Read response from a command")
    public String read(@Argument(description="Command")
                       String cmnd) throws DriverException
    {
        return dev.readString(cmnd);
    }


    @Command(name="lock", description="Set protected settings lock")
    public void setLock(@Argument(description="ON or OFF")
                        OnOff onoff) throws DriverException
    {
        dev.setLock(onoff == OnOff.ON);
    }


    @Command(name="setdefaults", description="Set factory defaults")
    public void setDefaults() throws DriverException
    {
        dev.setDefaults();
    }


    @Command(name="setdefaults", description="Set factory defaults")
    public void setDefaults(@Argument(description="Default item to set")
                            Default item) throws DriverException
    {
        dev.setDefaults(item);
    }


    @Command(name="showdevice", description="Show device type")
    public String showDevice() throws DriverException
    {
        return dev.getDeviceType();
    }


    @Command(name="showfwversion", description="Show firmware version")
    public String showFwVersion() throws DriverException
    {
        return dev.getFirmwareVersion();
    }


    @Command(name="showhwversion", description="Show hardware version")
    public String showHwVersion() throws DriverException
    {
        return dev.getHardwareVersion();
    }


    @Command(name="showmanufacturer", description="Show manufacturer")
    public String showManufacturer() throws DriverException
    {
        return dev.getManufacturer();
    }


    @Command(name="showmodel", description="Show model")
    public String showModel() throws DriverException
    {
        return dev.getModel();
    }


    @Command(name="showpart", description="Show part number")
    public String showPart() throws DriverException
    {
        return dev.getPartNumber();
    }


    @Command(name="showserial", description="Show serial number")
    public String showSerial() throws DriverException
    {
        return dev.getSerialNumber();
    }


    @Command(name="setusertag", description="Set user tag")
    public void setUserTag(@Argument(description="The tag string")
                           String tag) throws DriverException
    {
        dev.setUserTag(tag);
    }


    @Command(name="showusertag", description="Show user tag")
    public String showUserTag() throws DriverException
    {
        return dev.getUserTag();
    }


    @Command(name="showstatus", description="Show status word")
    public char showStatus() throws DriverException
    {
        return dev.getStatus();
    }


    @Command(name="showontime", description="Show unit on time")
    public String showOnTime() throws DriverException
    {
        return dev.getTimeOn() + " hours";
    }


    @Command(name="showsensortime", description="Show cold cathode on time")
    public String showSensorTime() throws DriverException
    {
        return dev.getSensorTimeOn() + " hours";
    }


    @Command(name="showsensordose", description="Show cold cathose pressure dose")
    public String showSensorDose() throws DriverException
    {
        return dev.getSensorDose() + " " + getUnitsString() + " hours";
    }


    @Command(name="setunits", description="Set pressure units")
    public void setUnits(@Argument(description="Pressure units")
                         Unit units) throws DriverException
    {
        dev.setPressureUnit(units);
    }


    @Command(name="showunits", description="Show pressure units")
    public Unit showUnits() throws DriverException
    {
        return dev.getPressureUnit();
    }


    @Command(name="setgastype", description="Set calibration gas type")
    public void setGasType(@Argument(description="Gas type")
                           Gas gas) throws DriverException
    {
        dev.setCalibrationGas(gas);
    }


    @Command(name="showgastype", description="Show calibration gas type")
    public Gas showGasType() throws DriverException
    {
        return dev.getCalibrationGas();
    }


    @Command(name="readpressure", description="Read vacuum pressure")
    public String readPressure() throws DriverException
    {
        return dev.readPressure() + " " + getUnitsString();
    }


    @Command(name="readpressure", description="Read vacuum pressure")
    public String readPressure(@Argument(description="Sensor to use")
                               Sensor sensor) throws DriverException
    {
        return dev.readPressure(sensor) + " " + getUnitsString();
    }


    @Command(name="setatmospheric", description="Set atmospheric pressure")
    public void setAtmospheric(@Argument(description="Atmospheric pressure")
                               double press) throws DriverException
    {
        dev.calibrateAtmospheric(press);
    }


    @Command(name="showatmospheric", description="Show atmospheric pressure difference")
    public String showAtmospheric() throws DriverException
    {
        return String.valueOf(dev.getAtmospheric());
    }


    @Command(name="readtemperature", description="Read MicroPirani temperature")
    public String readTemperature() throws DriverException
    {
        return String.valueOf(dev.readTemperature());
    }


    @Command(name="setrelaytrip", description="Set relay trip point")
    public void setRelayTrip(@Argument(description="Relay number")
                             int relay,
                             @Argument(description="Trip point value")
                             double value) throws DriverException
    {
        dev.setRelayTrip(relay, value);
    }


    @Command(name="showrelaytrip", description="Show relay trip point")
    public String showRelayTrip(@Argument(description="Relay number")
                                int relay) throws DriverException
    {
        return String.valueOf(dev.getRelayTrip(relay));
    }


    @Command(name="setrelayhyst", description="Set relay hysteresis")
    public void setRelayHyst(@Argument(description="Relay number")
                             int relay,
                             @Argument(description="Hysteresis value")
                             double value) throws DriverException
    {
        dev.setRelayHyst(relay, value);
    }


    @Command(name="showrelayhyst", description="Show relay hysteresis")
    public String showRelayHyst(@Argument(description="Relay number")
                                int relay) throws DriverException
    {
        return String.valueOf(dev.getRelayHyst(relay));
    }


    @Command(name="setrelaydirn", description="Set relay trip direction")
    public void setRelayDirn(@Argument(description="Relay number")
                             int relay,
                             @Argument(description="Trip direction")
                             Direction dirn) throws DriverException
    {
        dev.setRelayDirection(relay, dirn);
    }


    @Command(name="showrelaydirn", description="Show relay trip direction")
    public Direction showRelayDirn(@Argument(description="Relay number")
                                   int relay) throws DriverException
    {
        return dev.getRelayDirection(relay);
    }


    @Command(name="setrelayenable", description="Enable a relay")
    public void setRelayEnable(@Argument(description="Relay number")
                               int relay,
                               @Argument(description="Enable type")
                               Enable enable) throws DriverException
    {
        dev.setRelayEnable(relay, enable);
    }


    @Command(name="showrelayenable", description="Show whether relay enabled")
    public Enable showRelayEnable(@Argument(description="Relay number")
                                  int relay) throws DriverException
    {
        return dev.getRelayEnable(relay);
    }


    @Command(name="isrelayactive", description="Show whether relay is active")
    public String isRelayActive(@Argument(description="Relay number")
                                int relay) throws DriverException
    {
        return String.valueOf(dev.isRelayActive(relay));
    }


    @Command(name="setanalogout", description="Set analog output parameters")
    public void setAnalogOut(@Argument(description="Analog channel number (1 - 2)")
                             int chan,
                             @Argument(description="Sensor to use")
                             Sensor sensor,
                             @Argument(description="Analog out curve number")
                             int curve) throws DriverException
    {
        dev.setAnalogOut(chan, sensor, curve);
    }


    @Command(name="showanalogout", description="Show analog output setting")
    public String showAnalogOut(@Argument(description="Analog channel number (1 - 2)")
                                int chan) throws DriverException
    {
        return dev.getAnalogOut(chan);
    }


    @Command(name="setbaud", description="Set baud rate")
    public void setBaud(@Argument(description="Baud rate")
                        int baud) throws DriverException
    {
        dev.setBaudRate(baud);
    }


    @Command(name="showbaud", description="Show baud rate")
    public int showBaud() throws DriverException
    {
        return dev.getBaudRate();
    }


    @Command(name="setaddress", description="Set (change) device address")
    public void setAddress(@Argument(description="Address (1 - 253)")
                           int address) throws DriverException
    {
        dev.changeAddress(address);
    }


    @Command(name="settimeout", description="Set response timeout")
    public void setTimeout(@Argument(description="Timeout (secs)")
                           double timeout) throws DriverException
    {
        dev.setTimeout(timeout);
    }


    /**
     *  Gets the units string.
     *
     *  @throws  DriverException
     */
    private String getUnitsString() throws DriverException
    {
        return dev.getPressureUnit().name().toLowerCase();
    }

}
