package org.lsst.ccs.drivers.mks;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Program to test the MKS GP390 vacuum gauge driver
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TestGP390 {

    protected enum OnOff {

        ON, OFF;

    }

    protected enum Units {

        TORR(GP390.UNIT_TORR),
        MBAR(GP390.UNIT_MBAR),
        PASCAL(GP390.UNIT_PA);

        int value;

        Units(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    protected static enum Fmode {

        MANUAL(GP390.FMODE_MAN),
        ALT(GP390.FMODE_ALT),
        AUTO(GP390.FMODE_AUTO);

        int value;

        Fmode(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    private static final Map<Integer, String> unitsMap = new HashMap<>();
    static {
        unitsMap.put(GP390.UNIT_TORR, "Torr");
        unitsMap.put(GP390.UNIT_MBAR, "Mbar");
        unitsMap.put(GP390.UNIT_PA, "Pascal");
    }
    private static final Map<Integer, String> gmodeMap = new HashMap<>();
    static {
        gmodeMap.put(GP390.GMODE_OFF, "Off");
        gmodeMap.put(GP390.GMODE_DEGAS, "Degas");
        gmodeMap.put(GP390.GMODE_HIGH, "High emission");
        gmodeMap.put(GP390.GMODE_LOW, "Low emission");
    }

    private final GP390 gp = new GP390();


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    **  @param  connType  The type of connection: ftdi or serial
    **
    **  @param  ident     The identification: serial no. (ftdi) or port name
    **                    (serial)
    **
    **  @param  baud      The baud rate
    **
    **  @param  address   The address of the device
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     GP390.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident,
                     @Argument(name="baud", description="Baud rate")
                     int baud,
                     @Argument(name="addr", description="Device address")
                     int address) throws DriverException
    {
        gp.open(connType, ident, baud, address);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes the connection.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="close", description="Close the connection")
    public void close() throws DriverException
    {
        gp.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Unlocks protected settings.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="unlock", description="Unlock protected settings")
    public void unlock() throws DriverException
    {
        gp.unlock();
    }


   /**
    *************************************************************************** 
    ** 
    **  Toggles the protected settings lock.
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="toggle", description="Toggle protected settings lock")
    public String toggleLock() throws DriverException
    {
        return gp.toggleLock() ? "Locked" : "Unlocked";
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the protected settings lock.
    ** 
    **  @param  onoff  ON or OFF
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="lock", description="Set protected settings lock")
    public void setLock(@Argument(name="onoff", description="ON or OFF")
                        OnOff onoff) throws DriverException
    {
        gp.setLock(onoff == OnOff.ON);
    }


   /**
    *************************************************************************** 
    ** 
    **  Resets to power-on state.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="reset", description="Reset to power-on state")
    public void reset() throws DriverException
    {
        gp.reset();
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets values to factory defaults.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setdefaults", description="Set factory defaults")
    public void setDefaults() throws DriverException
    {
        gp.setDefaults();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the firmware version.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showversion", description="Show firmware version")
    public String showFwVersion() throws DriverException
    {
        return gp.getFwVersion();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the status word.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showstatus", description="Show status word")
    public String showStatusWord() throws DriverException
    {
        return String.format("%08x", gp.getStatusWord());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the pressure units.
    **
    **  @param  units
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setunits", description="Set pressure units")
    public void setUnits(@Argument(name="unit", description="Pressure units")
                         Units units) throws DriverException
    {
        gp.setPressureUnit(units.getValue());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the pressure units.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showunits", description="Show pressure units")
    public String showUnits() throws DriverException
    {
        return unitsMap.get(gp.getPressureUnit());
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the vacuum pressure.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvacuum", description="Read vacuum pressure")
    public String readPressure() throws DriverException
    {
        return gp.readPressure() + " " + getUnitsString();
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the differential pressure.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readdiff", description="Read differential pressure")
    public String readDiffPressure() throws DriverException
    {
        return gp.readDiffPressure() + " " + getUnitsString();
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the atmospheric pressure.
    **
    **  @param  press  The atmospheric pressure
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setatmospheric", description="Set atmospheric pressure")
    public void setAtmospheric(@Argument(name="press",
                                         description="Atmospheric pressure")
                               double press) throws DriverException
    {
        gp.setAtmospheric(press);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the atmospheric pressure.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showatmospheric", description="Show atmospheric pressure")
    public String showAtmospheric() throws DriverException
    {
        double press = gp.getAtmospheric();
        return press < 0 ? "Actual" : press + " " + getUnitsString();
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the filament mode.
    **
    **  @param  mode  The filament mode
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setfilament", description="Set filament mode")
    public void setFilament(@Argument(name="mode",
                                      description="Filament mode")
                            Fmode mode) throws DriverException
    {
        gp.setFilamentMode(mode.getValue());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the filament status.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showfilament", description="Show filament status")
    public String showFilament() throws DriverException
    {
        return String.valueOf(gp.getFilamentStatus());
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the micro-ion gauge on or off.
    **
    **  @param  onoff  ON or OFF
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setgauge", description="Set gauge on or off")
    public void setGaugeOn(@Argument(name="onoff",
                                     description="ON or OFF")
                           OnOff onoff) throws DriverException
    {
        gp.setMiGaugeOn(onoff == OnOff.ON);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge state.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showgauge", description="Show gauge state")
    public String showGauge() throws DriverException
    {
        return gp.isMiGaugeOn() ? "On" : "Off";
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the micro-ion gauge degas on or off.
    **
    **  @param  onoff  ON or OFF
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setdegas", description="Turn gauge degas on or off")
    public void setDegasOn(@Argument(name="onoff",
                                     description="ON or OFF")
                           OnOff onoff) throws DriverException
    {
        gp.setMiDegasOn(onoff == OnOff.ON);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge degas state.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showdegas", description="Show gauge degas state")
    public String showDegas() throws DriverException
    {
        return gp.isMiDegasOn() ? "On" : "Off";
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the micro-ion gauge degas time.
    **
    **  @param  time  Degas time (sec)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setdegastime", description="Set gauge degas time")
    public void setDegasTime(@Argument(name="time",
                                       description="Degas time")
                             int time) throws DriverException
    {
        gp.setMiDegasTime(time);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge degas time.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showdegastime", description="Show degas time")
    public String showDegasTime() throws DriverException
    {
        return String.valueOf(gp.getMiDegasTime());
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the micro-ion gauge delay on or off.
    **
    **  @param  onoff  ON or OFF
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setdelay", description="Turn gauge delay on or off")
    public void setDelayOn(@Argument(name="onoff",
                                     description="ON or OFF")
                           OnOff onoff) throws DriverException
    {
        gp.setMiDelayOn(onoff == OnOff.ON);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge delay state.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showdelay", description="Show gauge delay state")
    public String showDelay() throws DriverException
    {
        return gp.isMiDelayOn() ? "On" : "Off";
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the micro-ion gauge delay time.
    **
    **  @param  time  Delay time (sec)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setdelaytime", description="Set gauge delay time")
    public void setDelayTime(@Argument(name="time",
                                       description="Delay time")
                             int time) throws DriverException
    {
        gp.setMiDelayTime(time);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge delay time.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showdelaytime", description="Show gauge delay time")
    public String showDelayTime() throws DriverException
    {
        return String.valueOf(gp.getMiDelayTime());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the micro-ion gauge switch pressure.
    **
    **  @param  press  The switch pressure
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setswitch", description="Set gauge switch pressure")
    public void setSwitchPressure(@Argument(name="pressure",
                                            description="Switch pressure")
                                  double press) throws DriverException
    {
        gp.setMiSwitchPressure(press);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge switch pressure.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showswitch", description="Show gauge switch pressure")
    public String showSwitchPressure() throws DriverException
    {
        return gp.getMiSwitchPressure() + " " + getUnitsString();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the micro-ion gauge current.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show gauge current")
    public String showCurrent() throws DriverException
    {
        return gp.getMiGaugeCurrent() + " mA";
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the address offset.
    **
    **  @param  offset  The first hex digit of the address
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setoffset", description="Set address offset")
    public void setOffset(@Argument(name="offset",
                                    description="Address first digit")
                          int offset) throws DriverException
    {
        gp.setAddressOffset(offset);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the baud rate.
    **
    **  @param  baud  The baud rate
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setbaud", description="Set baud rate")
    public void setUnits(@Argument(name="unit", description="Baud rate")
                         int baud) throws DriverException
    {
        gp.setBaudRate(baud);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the response timeout.
    **
    **  @param  timeout  The timeout (secs)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="settimeout", description="Set response timeout")
    public void setTimeout(@Argument(name="timeout",
                                     description="Timeout (secs)")
                           double timeout) throws DriverException
    {
        gp.setTimeout(timeout);
    }


   /**
    *************************************************************************** 
    ** 
    **  Gets the units string.
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    private String getUnitsString() throws DriverException
    {
        return unitsMap.get(gp.getPressureUnit()).toLowerCase();
    }

}
