package org.lsst.ccs.drivers.microion;

import java.util.Random;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
//import org.lsst.ccs.drivers.serial.SerialPort;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 **************************************************************************
 **
 ** General access routines for the MicroIon device * * @author Homer Neal *
 * **************************************************************************
 */
public class MicroIon extends Scpi {

    /**
     **************************************************************************
     **
     ** Public constants *
     * **************************************************************************
     */

    public enum onOff {

        OFF, ON;
    }

    /**
     **************************************************************************
     **
     ** Private constants *
     * **************************************************************************
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private String terminator = "\n";
    private int timeout = 1000;
//    private byte[] buff = new byte[1024];
//    private int buffIn, buffOut;

//    private SerialPort port = new SerialPort();
    private String saddr = "2";
    
    /**
     **************************************************************************
     **
     ** Constructor * *
     * **************************************************************************
     */
    public MicroIon() {
    }


    /**
     **************************************************************************
     **
     ** Opens a connection. * * @param device The device name of the serial
     * port
     ** @param baud The baud rate
     ** @param dummy Dummy argument * * @throws DriverException *
     * **************************************************************************
     */
    public void open(String serialname, int port, int addr) throws DriverException
    {
        open(Scpi.CONN_TYPE_SERIAL, serialname, port);
        Integer iaddr = addr;
        saddr = iaddr.toString();
        init();
//        setTimeout(timeout); // set SCPI timeout    
    }

      /**
    ***************************************************************************
    **
    **  Opens an FTDI connection.
    **
    ***************************************************************************
    */
    public void openftdi(String serialname, int port, int addr) throws DriverException
    {
        System.out.println("opening connection to the MicroIon Vacuum Gauge");
        open(Scpi.CONN_TYPE_FTDI, serialname, port);
        Integer iaddr = addr;
        saddr = iaddr.toString();
        init();
//        setTimeout(timeout); // set SCPI timeout
    }
    
    public void init() throws DriverException
    {
        System.out.println("Device response to baud setting is:"+readMicroIon("#0"+saddr+"SB19200")); // set device baud to 19200
        System.out.println("Device response to 8,1,N setting is:"+readMicroIon("#0"+saddr+"SPN")); // set 8 bits, no parity
        System.out.println("Device response to address setting is:"+readMicroIon("#0"+saddr+"SA0")); // set address offset (SA)                
    }
    
      /**
    ***************************************************************************
    **
    **  Prints device ID
    **
    ***************************************************************************
    */
    public String printdevid() throws DriverException
    {
        String[] id = getIdentification();
        return "Successfully connected to:" +
               "\nManufacturer:   " + id[Scpi.IDENT_MANUFACTURER] +
               "\nModel name:     " + id[Scpi.IDENT_MODEL_NAME] +
               "\nSerial number:  " + id[Scpi.IDENT_SERIAL_NUMBER] +
               "\nF/W version:    " + id[Scpi.IDENT_FW_VERSION];
    }


    /**
     ***************************************************************************
     **
     ** Gets the MicroIon power state. *
     * **************************************************************************
     */
    public onOff getState() throws DriverException {
        String srm = readMicroIon("#0"+saddr+"IGS");
        System.out.println("response = "+srm);
        String[] response = srm.split(" ");
        return response[3]=="ON" ? onOff.ON : onOff.OFF;
    }

    /**
     ***************************************************************************
     **
     ** Gets the MicroIon pressure reading *
     * **************************************************************************
     */
    public double getPressure() throws DriverException {
        String srm = readMicroIon("#0"+saddr+"RD");
        System.out.println("response = "+srm);
        String[] response = srm.split(" ");
//        System.out.println("number string = "+response[1]);
        return Double.valueOf(response[1]);
    }


    /**
     **************************************************************************
     **
     ** Sets the receive timeout. * * @param timeout The receive timeout (ms).
     * 0 means no timeout. * * @throws DriverException *
     * *************************************************************************
     */
    public void setTimeout(int timeout) throws DriverException {
        this.timeout = timeout;
    }

    /**
     **************************************************************************
     **
     ** Writes a command. * * @param command The command to write, excluding
     * terminator * * @throws DriverException *
     * *************************************************************************
     */
    public synchronized void writeMicroIon(String command) throws DriverException {
//        checkOpen();
        write(command + terminator);
    }

    /**
     **************************************************************************
     **
     ** Reads a response. * * @return The command response string * * @throws
     * DriverException
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public synchronized String readMicroIon() throws DriverException {
//        checkOpen();
        return(read());
    }

    /**
     **************************************************************************
     **
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public synchronized String readMicroIon(String command) throws DriverException {
//        flush();

        writeMicroIon(command);
        return readMicroIon();
    }

}
