package org.lsst.ccs.drivers.mcc;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 *  Program to test a Measurement Computing DAQ device
 *
 *  @author Owen Saxton
 */
public class TestMccUsb {

    /*
     *  Lookup tables
     */
    private final static Map<Integer, String> devNames = new HashMap<>();
    static {
        devNames.put(MccUsb.USB_TC_AI_DID, "tcai");
        devNames.put(MccUsb.USB_TC_DID,    "tc");
        devNames.put(MccUsb.USB_TEMP_DID,  "temp");
    }

    private final static Map<Integer, String> itemNames = new HashMap<>();
    static {
        itemNames.put(0, "adc0");
        itemNames.put(1, "adc1");
        itemNames.put(2, "adc2");
        itemNames.put(3, "adc3");
    }

    private final static Map<Integer, String> subiNames = new HashMap<>();
    static {
        subiNames.put(MccUsb.CSI_SENSOR_TYPE, "senstype");
        subiNames.put(MccUsb.CSI_CONN_TYPE,   "conntype");
        subiNames.put(MccUsb.CSI_FILTER_RATE, "filtrate");
        subiNames.put(MccUsb.CSI_EXCITATION,  "excitation");
        subiNames.put(MccUsb.CSI_VREF,        "vref");
        subiNames.put(MccUsb.CSI_I_VALUE_0,   "ival0");
        subiNames.put(MccUsb.CSI_I_VALUE_1,   "ival1");
        subiNames.put(MccUsb.CSI_I_VALUE_2,   "ival2");
        subiNames.put(MccUsb.CSI_V_VALUE_0,   "vval0");
        subiNames.put(MccUsb.CSI_V_VALUE_1,   "vval1");
        subiNames.put(MccUsb.CSI_V_VALUE_2,   "vval2");
        subiNames.put(MccUsb.CSI_CH_0_TC,     "tc0");
        subiNames.put(MccUsb.CSI_CH_1_TC,     "tc1");
        subiNames.put(MccUsb.CSI_CH_0_GAIN,   "gain0");
        subiNames.put(MccUsb.CSI_CH_1_GAIN,   "gain1");
        subiNames.put(MccUsb.CSI_CH_0_COEF_0, "coef00");
        subiNames.put(MccUsb.CSI_CH_1_COEF_0, "coef01");
        subiNames.put(MccUsb.CSI_CH_0_COEF_1, "coef10");
        subiNames.put(MccUsb.CSI_CH_1_COEF_1, "coef11");
        subiNames.put(MccUsb.CSI_CH_0_COEF_2, "coef20");
        subiNames.put(MccUsb.CSI_CH_1_COEF_2, "coef21");
        subiNames.put(MccUsb.CSI_CH_0_COEF_3, "coef30");
        subiNames.put(MccUsb.CSI_CH_1_COEF_3, "coef31");
        subiNames.put(MccUsb.CSI_CH_0_VCONN,  "vconn0");
        subiNames.put(MccUsb.CSI_CH_1_VCONN,  "vconn1");
    }

    private final static CmndProcess.Lookup stypNames;
    static {
        stypNames = new CmndProcess.Lookup(6);
        stypNames.add("rtd",           MccUsb.STP_RTD);
        stypNames.add("thermistor",    MccUsb.STP_THERMISTOR);
        stypNames.add("thermocouple",  MccUsb.STP_THERMOCOUPLE);
        stypNames.add("semiconductor", MccUsb.STP_SEMICONDUCTOR);
        stypNames.add("disabled",      MccUsb.STP_DISABLED);
        stypNames.add("voltage",       MccUsb.STP_VOLTAGE);
    }

    private final static CmndProcess.Lookup stypsNames;
    static {
        stypsNames = new CmndProcess.Lookup(6);
        stypsNames.add("rtd",     MccUsb.STP_RTD);
        stypsNames.add("thermis", MccUsb.STP_THERMISTOR);
        stypsNames.add("thermoc", MccUsb.STP_THERMOCOUPLE);
        stypsNames.add("semicon", MccUsb.STP_SEMICONDUCTOR);
        stypsNames.add("disabld", MccUsb.STP_DISABLED);
        stypsNames.add("voltage", MccUsb.STP_VOLTAGE);
    }

    private final static CmndProcess.Lookup frateNames;
    static {
        frateNames = new CmndProcess.Lookup(14);
        frateNames.add("500 Hz",  MccUsb.FREQ_500_HZ);
        frateNames.add("250 Hz",  MccUsb.FREQ_250_HZ);
        frateNames.add("125 Hz",  MccUsb.FREQ_125_HZ);
        frateNames.add("62.5 Hz", MccUsb.FREQ_62_5_HZ);
        frateNames.add("50 Hz",   MccUsb.FREQ_50_HZ);
        frateNames.add("39.2 Hz", MccUsb.FREQ_39_2_HZ);
        frateNames.add("33.3 Hz", MccUsb.FREQ_33_3_HZ);
        frateNames.add("19.6 Hz", MccUsb.FREQ_19_6_HZ);
        frateNames.add("16.7 Hz", MccUsb.FREQ_16_7_HZ);
        frateNames.add("12.5 Hz", MccUsb.FREQ_12_5_HZ);
        frateNames.add("10 Hz",   MccUsb.FREQ_10_HZ);
        frateNames.add("8.33 Hz", MccUsb.FREQ_8_33_HZ);
        frateNames.add("6.25 Hz", MccUsb.FREQ_6_25_HZ);
        frateNames.add("4.17 Hz", MccUsb.FREQ_4_17_HZ);
    }

    private final static CmndProcess.Lookup fratsNames;
    static {
        fratsNames = new CmndProcess.Lookup(14);
        fratsNames.add("500Hz", MccUsb.FREQ_500_HZ);
        fratsNames.add("250Hz", MccUsb.FREQ_250_HZ);
        fratsNames.add("125Hz", MccUsb.FREQ_125_HZ);
        fratsNames.add("62Hz",  MccUsb.FREQ_62_5_HZ);
        fratsNames.add("50Hz",  MccUsb.FREQ_50_HZ);
        fratsNames.add("39Hz",  MccUsb.FREQ_39_2_HZ);
        fratsNames.add("33Hz",  MccUsb.FREQ_33_3_HZ);
        fratsNames.add("19Hz",  MccUsb.FREQ_19_6_HZ);
        fratsNames.add("16Hz",  MccUsb.FREQ_16_7_HZ);
        fratsNames.add("12Hz",  MccUsb.FREQ_12_5_HZ);
        fratsNames.add("10Hz",  MccUsb.FREQ_10_HZ);
        fratsNames.add("8Hz",   MccUsb.FREQ_8_33_HZ);
        fratsNames.add("6Hz",   MccUsb.FREQ_6_25_HZ);
        fratsNames.add("4Hz",   MccUsb.FREQ_4_17_HZ);
    }

    private final static CmndProcess.Lookup tctypNames;
    static {
        tctypNames = new CmndProcess.Lookup(8);
        tctypNames.add("J", MccUsb.TC_TYPE_J);
        tctypNames.add("K", MccUsb.TC_TYPE_K);
        tctypNames.add("T", MccUsb.TC_TYPE_T);
        tctypNames.add("E", MccUsb.TC_TYPE_E);
        tctypNames.add("R", MccUsb.TC_TYPE_R);
        tctypNames.add("S", MccUsb.TC_TYPE_S);
        tctypNames.add("B", MccUsb.TC_TYPE_B);
        tctypNames.add("N", MccUsb.TC_TYPE_N);
    }

    private final static CmndProcess.Lookup gainNames;
    static {
        gainNames = new CmndProcess.Lookup(8);
        gainNames.add("1X",   MccUsb.GAIN_1X);
        gainNames.add("2X",   MccUsb.GAIN_2X);
        gainNames.add("4X",   MccUsb.GAIN_4X);
        gainNames.add("8X",   MccUsb.GAIN_8X);
        gainNames.add("16X",  MccUsb.GAIN_16X);
        gainNames.add("32X",  MccUsb.GAIN_32X);
        gainNames.add("64X",  MccUsb.GAIN_64X);
        gainNames.add("128X", MccUsb.GAIN_128X);
    }

    private final static CmndProcess.Lookup rangeNames;
    static {
        rangeNames = new CmndProcess.Lookup(4);
        rangeNames.add("10V",   MccUsb.RANGE_10V);
        rangeNames.add("5V",    MccUsb.RANGE_5V);
        rangeNames.add("2.5V",  MccUsb.RANGE_2_5V);
        rangeNames.add("1.25V", MccUsb.RANGE_1_25V);
    }

    private final static CmndProcess.Lookup vconnNames;
    static {
        vconnNames = new CmndProcess.Lookup(4);
        vconnNames.add("differential", MccUsb.VCT_DIFFERENTIAL);
        vconnNames.add("single-ended", MccUsb.VCT_SINGLE_ENDED);
        vconnNames.add("grounded",     MccUsb.VCT_GROUNDED);
        vconnNames.add("calibration",  MccUsb.VCT_CALIBRATION);
    }

    private final static CmndProcess.Lookup vconsNames;
    static {
        vconsNames = new CmndProcess.Lookup(4);
        vconsNames.add("differ",  MccUsb.VCT_DIFFERENTIAL);
        vconsNames.add("snglend", MccUsb.VCT_SINGLE_ENDED);
        vconsNames.add("ground",  MccUsb.VCT_GROUNDED);
        vconsNames.add("calib",   MccUsb.VCT_CALIBRATION);
    }

    private final static CmndProcess.Lookup excitNames;
    static {
        excitNames = new CmndProcess.Lookup(3);
        excitNames.add("off",    MccUsb.CEX_OFF);
        excitNames.add("10 uA",  MccUsb.CEX_10UA);
        excitNames.add("210 uA", MccUsb.CEX_210UA);
    }

    private final static CmndProcess.Lookup tconnNames;
    static {
        tconnNames = new CmndProcess.Lookup(4);
        tconnNames.add("2-wire, 1-sensor", MccUsb.TCT_2WIRE_1SENSOR);
        tconnNames.add("2-wire, 2-sensor", MccUsb.TCT_2WIRE_2SENSOR);
        tconnNames.add("3-wire",           MccUsb.TCT_3WIRE);
        tconnNames.add("4-wire",           MccUsb.TCT_4WIRE);
    }

    private final static CmndProcess.Lookup tconsNames;
    static {
        tconsNames = new CmndProcess.Lookup(4);
        tconsNames.add("2-wire1", MccUsb.TCT_2WIRE_1SENSOR);
        tconsNames.add("2-wire2", MccUsb.TCT_2WIRE_2SENSOR);
        tconsNames.add("3-wire",  MccUsb.TCT_3WIRE);
        tconsNames.add("4-wire",  MccUsb.TCT_4WIRE);
    }

    /*
     *  Enumerations
     */
    protected static enum DevName {

        TCAI(MccUsb.USB_TC_AI_DID),
        TC(MccUsb.USB_TC_DID),
        TEMP(MccUsb.USB_TEMP_DID);

        int value;

        DevName(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    protected static enum AdUnits {

        CONV(0),
        RAW(1);

        int value;

        AdUnits(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    protected static enum ItemName {

        ADC0, ADC1, ADC2, ADC3;

    }

    protected static enum SubiName {

        SENSTYPE(MccUsb.CSI_SENSOR_TYPE),
        CONNTYPE(MccUsb.CSI_CONN_TYPE),
        FILTRATE(MccUsb.CSI_FILTER_RATE),
        EXCITATION(MccUsb.CSI_EXCITATION),
        VREF(MccUsb.CSI_VREF),
        IVAL0(MccUsb.CSI_I_VALUE_0),
        IVAL1(MccUsb.CSI_I_VALUE_1),
        IVAL2(MccUsb.CSI_I_VALUE_2),
        VVAL0(MccUsb.CSI_V_VALUE_0),
        VVAL1(MccUsb.CSI_V_VALUE_1),
        VVAL2(MccUsb.CSI_V_VALUE_2),
        TC0(MccUsb.CSI_CH_0_TC),
        TC1(MccUsb.CSI_CH_1_TC),
        GAIN0(MccUsb.CSI_CH_0_GAIN),
        GAIN1(MccUsb.CSI_CH_1_GAIN),
        COEF00(MccUsb.CSI_CH_0_COEF_0),
        COEF01(MccUsb.CSI_CH_1_COEF_0),
        COEF10(MccUsb.CSI_CH_0_COEF_1),
        COEF11(MccUsb.CSI_CH_1_COEF_1),
        COEF20(MccUsb.CSI_CH_0_COEF_2),
        COEF21(MccUsb.CSI_CH_1_COEF_2),
        COEF30(MccUsb.CSI_CH_0_COEF_3),
        COEF31(MccUsb.CSI_CH_1_COEF_3),
        VCONN0(MccUsb.CSI_CH_0_VCONN),
        VCONN1(MccUsb.CSI_CH_1_VCONN);

        int value;

        SubiName(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    protected static enum DispType {

        BRIEF(false),
        FULL(true);

        boolean value;

        DispType(boolean value)
        {
            this.value = value;
        }

        public boolean getValue()
        {
            return value;
        }

    }

    protected static enum CalPol {

        POSITIVE, NEGATIVE;

    }

    protected static enum CalPath {

        HIGH, LOW;

    }

    protected static enum CalType {

        TEMPERATURE, VOLTAGE, ABORT;

    }

    /*
     *  Private fields
     */
    private final static PrintStream out = System.out;
    private final MccUsb mcc = new MccUsb();


    /**
     *  Opens a connection to a device.
     *
     *  @param  name    The MCC device name enumeration
     *  @param  serial  The USB serial number
     *  @param  force   Whether to force the interface claim
     *  @throws  DriverException
     */
    @Command(name = "open", description = "Open a connection")
    public void open(@Argument(description = "The MCC device name")
                     DevName name,
                     @Argument(description = "The USB serial number")
                     String serial,
                     @Argument(description = "Force the interface claim")
                     boolean force) throws DriverException
    {
        if (serial.length() == 0) {
            serial = null;
        }
        mcc.open(name.getValue(), serial, force);
    }


    /**
     *  Opens a connection to a device, forcing the interface claim.
     *
     *  @param  name    The MCC device name enumeration
     *  @param  serial  The USB serial number
     *  @throws  DriverException
     */
    @Command(name = "open", description = "Open a connection & force the claim")
    public void open(@Argument(description = "The MCC device name")
                     DevName name,
                     @Argument(description = "The USB serial number")
                     String serial) throws DriverException
    {
        if (serial.length() == 0) {
            serial = null;
        }
        mcc.open(name.getValue(), serial, true);
    }


    /**
     *  Opens a connection to a device, forcing the interface claim.
     *
     *  @param  name    The MCC device name enumeration
     *  @throws  DriverException
     */
    @Command(name = "open", description = "Open a connection & force the claim")
    public void open(@Argument(description = "The MCC device name")
                     DevName name) throws DriverException
    {
        mcc.open(name.getValue(), null, true);
    }


    /**
     *  Closes a connection.
     *
     *  @throws  DriverException
     */
    @Command(name = "close", description = "Close a connection")
    public void close() throws DriverException
    {
        mcc.close();
    }


    /**
     *  Configures the DIO port.
     *
     *  @param  value  The direction for all lines: 0 = out; ~0 = in
     *  @throws  DriverException
     */
    @Command(name = "diocfg", description = "Configure the DIO port")
    public void dioCfg(@Argument(description = "The value: 0 = out; ~0 = in")
                       int value) throws DriverException
    {
        mcc.dioConfig(value);
    }


    /**
     *  Configures one line of the DIO port.
     *
     *  @param  bitnum  The line number (0 - 7)
     *  @param  value   The direction to set: 0 = out; ~0 = in
     *  @throws  DriverException
     */
    @Command(name = "diocfgbit", description = "Configure one DIO line")
    public void dioCfgBit(@Argument(description = "The line to configure")
                          int bitnum,
                          @Argument(description = "The value: 0 = out; ~0 = in")
                          int value) throws DriverException
    {
        mcc.dioConfigBit(bitnum, value);
    }


    /**
     *  Reads the DIO port.
     *
     *  @throws  DriverException
     */
    @Command(name = "dioin", description = "Read the DIO port")
    public void dioIn() throws DriverException
    {
        out.format("Value = 0x%02x\n", mcc.dioIn());
    }


    /**
     *  Reads a DIO line
     *
     *  @param  bitnum  The line number (0 - 7)
     *  @throws  DriverException
     */
    @Command(name = "dioinbit", description = "Read a DIO line")
    public void dioInBit(@Argument(description = "The line to read")
                         int bitnum) throws DriverException
    {
        out.println("Value = " + mcc.dioInBit(bitnum));
    }


    /**
     *  Writes to the DIO port.
     *
     *  @param  value   The value to write
     *  @throws  DriverException
     */
    @Command(name = "dioout", description = "Write to the DIO port")
    public void dioOut(@Argument(description = "The value to write")
                       int value) throws DriverException
    {
        mcc.dioOut(value);
    }


    /**
     *  Writes a DIO line.
     *
     *  @param  bitnum  The line number (0 - 7)
     *  @param  value   The value to write
     *  @throws  DriverException
     */
    @Command(name = "diooutbit", description = "Write a DIO line")
    public void dioOutBit(@Argument(description = "The line to write")
                          int bitnum,
                          @Argument(description = "The value to write")
                          int value) throws DriverException
    {
        mcc.dioOutBit(bitnum, value);
    }


    /**
     *  Reads an ADC channel.
     *
     *  @param  chan   The channel number (0 - 7)
     *  @param  units  The units enumeration: conv or raw
     *  @throws  DriverException
     */
    @Command(name = "adin", description = "Read an ADC channel")
    public void adIn(@Argument(description = "The channel to read")
                     int chan,
                     @Argument(description = "The units for the value")
                     AdUnits units) throws DriverException
    {
        out.println("Value = " + mcc.adcIn(chan, units.getValue()));
    }


    /**
     *  Reads an ADC channel.
     *
     *  @param  chan   The channel number (0 - 7)
     *  @throws  DriverException
     */
    @Command(name = "adin", description = "Read an ADC channel")
    public void adIn(@Argument(description = "The channel to read")
                     int chan) throws DriverException
    {
        adIn(chan, AdUnits.CONV);
    }


    /**
     *  Scans ADC channels.
     *
     *  @param  fchan  The first channel number (0 - 7)
     *  @param  lchan  The last channel number (0 - 7)
     *  @param  units  The units enumeration: conv or raw
     *  @throws  DriverException
     */
    @Command(name = "adscan", description = "Scan ADC channels")
    public void adScan(@Argument(description = "The first channel to read")
                       int fchan,
                       @Argument(description = "The last channel to read")
                       int lchan,
                       @Argument(description = "The units for the values")
                       AdUnits units) throws DriverException
    {
        float[] value = new float[8];
        int count = mcc.adcScan(fchan, lchan, units.getValue(), value);
        if (count <= 0) return;
        out.print("Values =");
        int posn = 8, line = 0;
        for (int j = 0; j < count; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8;
            }
            out.print(sValue);
        }
        out.println();
    }


    /**
     *  Scans ADC channels.
     *
     *  @param  fchan  The first channel number (0 - 7)
     *  @param  lchan  The last channel number (0 - 7)
     *  @throws  DriverException
     */
    @Command(name = "adscan", description = "Scan ADC channels")
    public void adScan(@Argument(description = "The first channel to read")
                       int fchan,
                       @Argument(description = "The last channel to read")
                       int lchan) throws DriverException
    {
        adScan(fchan, lchan, AdUnits.CONV);
    }


    /**
     *  Initializes the counter.
     *
     *  @throws  DriverException
     */
    @Command(name = "cntrinit", description = "Initialize the counter")
    public void cntrinit() throws DriverException
    {
        mcc.cntrInit();
    }


    /**
     *  Reads the counter.
     *
     *  @throws  DriverException
     */
    @Command(name = "cntrread", description = "Read the counter")
    public void cntrread() throws DriverException
    {
        out.println("Value = " + mcc.cntrRead());
    }


    /**
     *  Blinks the LED.
     *
     *  @throws  DriverException
     */
    @Command(name = "blink", description = "Blink the LED")
    public void blink() throws DriverException
    {
        mcc.blink();
    }


    /**
     *  Resets the device.
     *
     *  @throws  DriverException
     */
    @Command(name = "reset", description = "Reset the device")
    public void reset() throws DriverException
    {
        mcc.reset();
    }


    /**
     *  Shows the calibration status.
     *
     *  @throws  DriverException
     */
    @Command(name = "status", description = "Show the calibration status")
    public void status() throws DriverException
    {
        out.format("Value = 0x%02x\n", mcc.getStatus());
    }


    /**
     *  Shows help for setting a configuration item.
     *
     *  @param  itemE     The item enumeration
     *  @param  subitemE  The subitem enumeration
     *  @throws  DriverException
     */
    @Command(name = "itemset", description = "Display set configuration help")
    public void itemSet(@Argument(name = "item",
                                  description = "The item name")
                        ItemName itemE,
                        @Argument(name = "subitem",
                                  description = "The subitem name")
                        SubiName subitemE) throws DriverException
    {
        int item = itemE.ordinal(), subItem = subitemE.getValue();
        if (subItem == MccUsb.CSI_SENSOR_TYPE)
            showNames("sensor type", stypNames);
        else if (subItem == MccUsb.CSI_CONN_TYPE)
            showNames("thermal connection type", tconsNames);
        else if (subItem == MccUsb.CSI_FILTER_RATE)
            showNames("filter rate", fratsNames);
        else if (subItem == MccUsb.CSI_EXCITATION)
            showNames("excitation", excitNames);
        else if (subItem == MccUsb.CSI_CH_0_TC
                   || subItem ==MccUsb.CSI_CH_1_TC)
            showNames("thermocouple type", tctypNames);
        else if (subItem == MccUsb.CSI_CH_0_GAIN
                   || subItem ==MccUsb.CSI_CH_1_GAIN)
            showNames("gain", gainNames);
        else if (subItem == MccUsb.CSI_CH_0_VCONN
                   || subItem ==MccUsb.CSI_CH_1_VCONN)
            showNames("voltage connection type", vconnNames);
        else
            out.println("Missing value argument");
    }


    /**
     *  Sets a configuration item.
     *
     *  @param  itemE     The item enumeration
     *  @param  subitemE  The subitem enumeration
     *  @param  sValue    The value to set
     *  @throws  DriverException
     */
    @Command(name = "itemset", description = "Set a configuration item")
    public void itemSet(@Argument(name = "item",
                                  description = "The item name")
                        ItemName itemE,
                        @Argument(name = "subitem",
                                  description = "The subitem name")
                        SubiName subitemE,
                        @Argument(name = "value",
                                  description = "The value to set")
                        String sValue) throws DriverException
    {
        int item = itemE.ordinal(), subItem = subitemE.getValue();
        if ((subItem >= MccUsb.CSI_VREF
             && subItem <= MccUsb.CSI_V_VALUE_2)
                || (subItem >= MccUsb.CSI_CH_0_COEF_0
                    && subItem <= MccUsb.CSI_CH_1_COEF_3)) {
            try {
                mcc.setItem(item, subItem, Float.valueOf(sValue));
            }
            catch(NumberFormatException e) {
                out.println("Invalid floating point number");
            }
        }
        else {
            try {
                int iValue;
                if (subItem == MccUsb.CSI_SENSOR_TYPE)
                    iValue = encode(stypNames, sValue);
                else if (subItem == MccUsb.CSI_CONN_TYPE)
                    iValue = encode(tconsNames, sValue);
                else if (subItem == MccUsb.CSI_FILTER_RATE)
                    iValue = encode(fratsNames, sValue);
                else if (subItem == MccUsb.CSI_EXCITATION)
                    iValue = encode(excitNames, sValue);
                else if (subItem == MccUsb.CSI_CH_0_TC
                           || subItem == MccUsb.CSI_CH_1_TC)
                    iValue = encode(tctypNames, sValue);
                else if (subItem == MccUsb.CSI_CH_0_GAIN
                           || subItem == MccUsb.CSI_CH_1_GAIN)
                    iValue = encode(gainNames, sValue);
                else if (subItem == MccUsb.CSI_CH_0_VCONN
                           || subItem == MccUsb.CSI_CH_1_VCONN)
                    iValue = encode(vconnNames, sValue);
                else
                    iValue = Integer.decode(sValue);
                mcc.setItem(item, subItem, (float)iValue);
            }
            catch(NumberFormatException e) {
                out.println("Invalid keyword or integer");
            }
        }
    }


    /**
     *  Shows a configuration item.
     *
     *  @param  itemE     The item enumeration
     *  @param  subitemE  The subitem enumeration
     *  @throws  DriverException
     */
    @Command(name = "itemshow", description = "Show a configuration item")
    public void itemShow(@Argument(name = "item",
                                   description = "The item name")
                         ItemName itemE,
                         @Argument(name = "subitem",
                                   description = "The subitem name")
                         SubiName subitemE) throws DriverException
    {
        int item = itemE.ordinal(), subItem = subitemE.getValue();
        float value = mcc.getItem(item, subItem);
        if ((subItem >= MccUsb.CSI_VREF
               && subItem <= MccUsb.CSI_V_VALUE_2)
              || (subItem >= MccUsb.CSI_CH_0_COEF_0
                    && subItem <= MccUsb.CSI_CH_1_COEF_3))
            out.println("Value = " + value);
        else {
            int iValue = (int)value;
            if (subItem == MccUsb.CSI_SENSOR_TYPE)
                out.format("Value = %s (%s)\n", iValue,
                           stypNames.decode(iValue));
            else if (subItem == MccUsb.CSI_CONN_TYPE)
                out.format("Value = %s (%s)\n", iValue,
                           tconnNames.decode(iValue));
            else if (subItem == MccUsb.CSI_FILTER_RATE)
                out.format("Value = %s (%s)\n", iValue,
                           frateNames.decode(iValue));
            else if (subItem == MccUsb.CSI_EXCITATION)
                out.format("Value = %s (%s)\n", iValue,
                           excitNames.decode(iValue));
            else if (subItem == MccUsb.CSI_CH_0_TC
                       || subItem == MccUsb.CSI_CH_1_TC)
                out.format("Value = %s (%s)\n", iValue,
                           tctypNames.decode(iValue));
            else if (subItem == MccUsb.CSI_CH_0_GAIN
                       || subItem == MccUsb.CSI_CH_1_GAIN)
                out.format("Value = %s (%s)\n", iValue,
                           gainNames.decode(iValue));
            else if (subItem == MccUsb.CSI_CH_0_VCONN
                       || subItem == MccUsb.CSI_CH_1_VCONN)
                out.format("Value = %s (%s)\n", iValue,
                           vconnNames.decode(iValue));
            else
                out.println("Value = " + iValue);
        }
    }


    /**
     *  Sets an alarm configuration.
     *
     *  @param  alarm   The alarm number (0 - 7)
     *  @param  iOptns  The input options
     *  @param  oOptns  The output options
     *  @param  value1  The first alarm value
     *  @param  value2  The second alarm value
     *  @throws  DriverException
     */
    @Command(name = "alarmset", description = "Set an alarm configuration")
    public void alarmSet(@Argument(description = "The alarm number")
                         int alarm,
                         @Argument(name = "ioptns",
                                   description = "The input options")
                         int iOptns,
                         @Argument(name = "ooptns",
                                   description = "The output options")
                         int oOptns,
                         @Argument(description = "The first alarm value")
                         double value1,
                         @Argument(description = "The second alarm value")
                         double value2) throws DriverException
    {
        mcc.configAlarm(alarm, iOptns, oOptns, (float)value1, (float)value2);
    }


    /**
     *  Shows an alarm configuration.
     *
     *  @param  alarm   The alarm number (0 - 7)
     *  @throws  DriverException
     */
    @Command(name = "alarmshow", description = "Show an alarm configuration")
    public void alarmShow(@Argument(description = "The alarm number")
                          int alarm) throws DriverException
    {
        int[] optns = {0, 0};
        float[] values = {0f, 0f};
        mcc.getAlarmConfig(alarm, optns, values);
        out.format("InOptns = 0x%02x, OutOptns = 0x%02x, Value1 = %s, Value2 = %s\n",
                   optns[0], optns[1], values[0], values[1]);
    }


    /**
     *  Shows the name of the open device.
     *
     *  @throws  DriverException
     */
    @Command(name = "showdev", description = "Show the name of the open device")
    public void showDev() throws DriverException
    {
        int devId = mcc.getDevId();
        if (devId < 0)
            out.println("No device open");
        else
            out.format("Device = %s\n", devNames.get(devId));
    }


    /**
     *  Shows channel configuration.
     *
     *  @param  chan  The channel number (0 - 7)
     *  @param  type  The display type enumeration
     *  @throws  DriverException
     */
    @Command(name = "showchan", description = "Show channel configuration")
    public void showChan(@Argument(description = "The channel number")
                         int chan,
                         @Argument(description = "Display type (brief or full)")
                         DispType type) throws DriverException
    {
        showChannel(chan, type.getValue());
    }


    /**
     *  Shows configuration of all channels.
     *
     *  @param  type  The display type enumeration
     *  @throws  DriverException
     */
    @Command(name = "showchan", description = "Show channel configuration")
    public void showChan(@Argument(description = "Display type (brief or full)")
                         DispType type) throws DriverException
    {
        showChannel(-1, type.getValue());
    }


    /**
     *  Shows brief configuration of all channels.
     *
     *  @throws  DriverException
     */
    @Command(name = "showchan", description = "Show channel configuration")
    public void showChan() throws DriverException
    {
        showChannel(-1, false);
    }


    /**
     *  Shows and resets the burnout status.
     *
     *  @param  mask  The mask of channels not to reset
     *  @throws  DriverException
     */
    @Command(name = "showburn", description = "Show and reset the burnout status")
    public void showBurn(@Argument(description = "The mask of channels to not reset")
                         int mask) throws DriverException
    {
        out.format("Value = 0x%02x\n", mcc.getBurnout(mask));
    }


    /**
     *  Shows the number of calibration steps.
     *
     *  @throws  DriverException
     */
    @Command(name = "showcal", description = "Show the number of calibration steps")
    public void showCal() throws DriverException
    {
        int[] steps = new int[2];
        mcc.calSteps(steps);
        out.format("Calibration steps: temp = %s, voltage = %s\n", steps[0], steps[1]);
    }


    /**
     *  Configures a calibration.
     *
     *  @param  gain   The gain selector
     *  @param  polar  The polarity enumeration
     *  @param  path   The path enumeration
     *  @throws  DriverException
     */
    @Command(name = "calcfg", description = "Configure a calibration")
    public void calCfg(@Argument(description = "The gain selector")
                       int gain,
                       @Argument(description = "The polarity (positive or negative)")
                       CalPol polar,
                       @Argument(description = "The path (high or low)")
                       CalPath path) throws DriverException
    {
        mcc.calConfig(gain, polar.ordinal(), path.ordinal());
    }


    /**
     *  Performs a calibration.
     *
     *  @param  type   The type enumeration
     *  @throws  DriverException
     */
    @Command(name = "calibrate", description = "Perform a calibration")
    public void calibrate(@Argument(description = "The type (temp, voltage or abort)")
                          CalType type) throws DriverException
    {
        mcc.calibrate(type.ordinal());
    }


    /**
     *  Shows channel configuration information.
     */
    private final static String[] caption =
                {"Sensor type",  "Therm conn",  "Excitation",   "Filter rate",
                 "TC type",      "Gain",        "Voltage conn", "Refc voltage",
                 "I value 0",    "I value 1",   "I value 2",    "V value 0",
                 "V value 1",    "V value 2",   "Coeff 0",      "Coeff 1",
                 "Coeff 2",      "Coeff 3"};
    private final static boolean[] isValue =
                {false, false, false, false, false, false, false, true,
                 true,  true,  true,  true,  true,  true,  true,  true,
                 true,  true};
    private final static boolean[] isExtra =
                {false, true,  true,  false, false, false, false, false,
                 true,  true,  true,  true,  true,  true,  true,  true,
                 true,  true};
    private final static int[] loSubitem =
                {MccUsb.CSI_SENSOR_TYPE, MccUsb.CSI_CONN_TYPE,
                 MccUsb.CSI_EXCITATION,  MccUsb.CSI_FILTER_RATE,
                 MccUsb.CSI_CH_0_TC,     MccUsb.CSI_CH_0_GAIN,
                 MccUsb.CSI_CH_0_VCONN,  MccUsb.CSI_VREF,
                 MccUsb.CSI_I_VALUE_0,   MccUsb.CSI_I_VALUE_1,
                 MccUsb.CSI_I_VALUE_2,   MccUsb.CSI_V_VALUE_0,
                 MccUsb.CSI_V_VALUE_1,   MccUsb.CSI_V_VALUE_2,
                 MccUsb.CSI_CH_0_COEF_0, MccUsb.CSI_CH_0_COEF_1,
                 MccUsb.CSI_CH_0_COEF_2, MccUsb.CSI_CH_0_COEF_3};
    private final static int[] hiSubitem =
                {MccUsb.CSI_SENSOR_TYPE, MccUsb.CSI_CONN_TYPE,
                 MccUsb.CSI_EXCITATION,  MccUsb.CSI_FILTER_RATE,
                 MccUsb.CSI_CH_1_TC,     MccUsb.CSI_CH_1_GAIN,
                 MccUsb.CSI_CH_1_VCONN,  MccUsb.CSI_VREF,
                 MccUsb.CSI_I_VALUE_0,   MccUsb.CSI_I_VALUE_1,
                 MccUsb.CSI_I_VALUE_2,   MccUsb.CSI_V_VALUE_0,
                 MccUsb.CSI_V_VALUE_1,   MccUsb.CSI_V_VALUE_2,
                 MccUsb.CSI_CH_1_COEF_0, MccUsb.CSI_CH_1_COEF_1,
                 MccUsb.CSI_CH_1_COEF_2, MccUsb.CSI_CH_1_COEF_3};

    private void showChannel(int chan, boolean all) throws DriverException
    {
        int first = (chan < 0) ? 0 : chan & 7;
        int last = (chan < 0) ? 7 : chan & 7;

        out.print("              ");
        for (int j = first; j <= last; j++) {
            out.format("   Ch %s ", j);
        }
        out.println();
        out.print("              ");
        for (int j = first; j <= last; j++) {
            out.print(" -------");
        }
        out.println();

        for (int k = 0; k < caption.length; k++) {
            if (isExtra[k] && !all) continue;
            out.format("%-12s: ", caption[k]);
            int highSub = hiSubitem[k], lowSub = loSubitem[k];
            boolean isVal = isValue[k];
            for (int j = first; j <= last; j++) {
                float
                  value = mcc.getItem(j >> 1, (j & 1) != 0 ? highSub : lowSub);
                if (isVal)
                    out.format(" %7.7s", value);
                else {
                    String desc = "huh?";
                    if (lowSub == MccUsb.CSI_SENSOR_TYPE)
                        desc = stypsNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CONN_TYPE)
                        desc = tconsNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_EXCITATION)
                        desc = excitNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_FILTER_RATE)
                        desc = frateNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CH_0_TC)
                        desc = tctypNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CH_0_GAIN)
                        desc = gainNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CH_0_VCONN)
                        desc = vconsNames.decode((int)value);
                    out.format(" %7s", desc);
                }
            }
            out.println();
        }
    }


    /**
     *  Show all names in a lookup table along with their codes
     */
    private void showNames(String desc, CmndProcess.Lookup lookup)
    {
        out.println("Possible " + desc + " names (values):");
        for (int j = 0; j < lookup.count(); j++) {
            out.format("  %s (%s)\n", lookup.name(j), lookup.code(j));
        }
    }


    /**
     *  Encode string, checking first whether it's already encoded
     */
    private int encode(CmndProcess.Lookup lookup, String token)
        throws NumberFormatException
    {
        int value;

        try {
            value = Integer.valueOf(token);
        }
        catch (NumberFormatException e) {
            value = lookup.encode(token, false);
            if (value < 0) throw e;
        }

        return value;
    }

}
