package org.lsst.ccs.drivers.mcc;

import javax.usb.UsbException;
import org.lsst.ccs.drivers.usb.UsbComm;
import org.lsst.ccs.drivers.usb.UsbCommPipe;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Enables access to a Measurement Computing data acquisition device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class MccUsb {

   /**
    ***************************************************************************
    **
    **  Inner class for storing operation description
    **
    ***************************************************************************
    */
    private static class Operation {

        byte cmnd;      // Command code
        int  mask;      // Mask of valid devices

        Operation(byte cmnd, int... devs)
        {
            this.cmnd = cmnd;
            this.mask = 0;
            for (int j = 0; j < devs.length; j++)
                this.mask |= 1 << devs[j];
        }

    }

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    /*
    **  Device identifiers
    */
    /** The USB-TC-AI DAQ device */
    public final static int USB_TC_AI_DID     = 0;

    /** The USB-TC DAQ device */
    public final static int USB_TC_DID        = 1;

    /** The USB-TEMP DAQ device */
    public final static int USB_TEMP_DID      = 2;

    /** The number of known device identifiers */
    private final static int N_DIDS           = 3;

    /*
    **  Item codes
    */
    /** Configuration item - ADC 0 */
    public final static int CIT_ADC_0         = 0x00;

    /** Configuration item - ADC 1 */
    public final static int CIT_ADC_1         = 0x01;

    /** Configuration item - ADC 2 */
    public final static int CIT_ADC_2         = 0x02;

    /** Configuration item - ADC 3 */
    public final static int CIT_ADC_3         = 0x03;

    /*
    **  Subitem codes
    */
    /** Config subitem - Connected sensor type */
    public final static int CSI_SENSOR_TYPE   = 0x00;

    /** Config subitem - Connection type for RTD & Thermistor Sensors */
    public final static int CSI_CONN_TYPE     = 0x01;

    /** Config subitem - Filter update rate */
    public final static int CSI_FILTER_RATE   = 0x02;

    /** Config subitem - Current excitation */
    public final static int CSI_EXCITATION    = 0x03;

    /** Config subitem - Measured Vref value */
    public final static int CSI_VREF          = 0x04;

    /** Config subitem - Measured I value @ 10uA */
    public final static int CSI_I_VALUE_0     = 0x05;

    /** Config subitem - Measured I value @ 210uA */
    public final static int CSI_I_VALUE_1     = 0x06;

    /** Config subitem - Measured I value @ 10uA (3 wire conn) */
    public final static int CSI_I_VALUE_2     = 0x07;

    /** Config subitem - Measured V value @ 10uA */
    public final static int CSI_V_VALUE_0     = 0x08;

    /** Config subitem - Measured V value @ 210uA */
    public final static int CSI_V_VALUE_1     = 0x09;

    /** Config subitem - Measured V value @ 210uA (3 wire conn) */
    public final static int CSI_V_VALUE_2     = 0x0a;

    /** Config subitem - Thermocouple type for channel 0 */
    public final static int CSI_CH_0_TC       = 0x10;

    /** Config subitem - Thermocouple type for channel 1 */
    public final static int CSI_CH_1_TC       = 0x11;

    /** Config subitem - Channel 0 gain value */
    public final static int CSI_CH_0_GAIN     = 0x12;

    /** Config subitem - Channel 1 gain value */
    public final static int CSI_CH_1_GAIN     = 0x13;

    /** Config subitem - Channel 0 coefficient 0 */
    public final static int CSI_CH_0_COEF_0   = 0x14;

    /** Config subitem - Channel 1 coefficient 0 */
    public final static int CSI_CH_1_COEF_0   = 0x15;

    /** Config subitem - Channel 0 coefficient 1 */
    public final static int CSI_CH_0_COEF_1   = 0x16;

    /** Config subitem - Channel 1 coefficient 1 */
    public final static int CSI_CH_1_COEF_1   = 0x17;

    /** Config subitem - Channel 0 coefficient 2 */
    public final static int CSI_CH_0_COEF_2   = 0x18;

    /** Config subitem - Channel 1 coefficient 2 */
    public final static int CSI_CH_1_COEF_2   = 0x19;

    /** Config subitem - Channel 0 coefficient 3 */
    public final static int CSI_CH_0_COEF_3   = 0x1a;

    /** Config subitem - Channel 1 coefficient 3 */
    public final static int CSI_CH_1_COEF_3   = 0x1b;

    /** Config subitem - Input type for ADC 2&3 channel 0 */
    public final static int CSI_CH_0_VCONN    = 0x1c;

    /** Config subitem - Input type for ADC 2&3 channel 1 */
    public final static int CSI_CH_1_VCONN    = 0x1d;

    /*
    **  Sensor type codes
    */
    /** Sensor type - RTD */
    public final static int STP_RTD           = 0x00;

    /** Sensor type - Thermistor */
    public final static int STP_THERMISTOR    = 0x01;

    /** Sensor type - Thermocouple */
    public final static int STP_THERMOCOUPLE  = 0x02;

    /** Sensor type - Semiconductor */
    public final static int STP_SEMICONDUCTOR = 0x03;

    /** Sensor type - Disabled */
    public final static int STP_DISABLED      = 0x04;

    /** Sensor type - Voltage */
    public final static int STP_VOLTAGE       = 0x05;

    /*
    **  Filter rate codes
    */
    /** Filter rate - 500 Hz */
    public final static int FREQ_500_HZ       = 0x01;

    /** Filter rate - 250 Hz */
    public final static int FREQ_250_HZ       = 0x02;

    /** Filter rate - 125 Hz */
    public final static int FREQ_125_HZ       = 0x03;

    /** Filter rate - 62.5 Hz */
    public final static int FREQ_62_5_HZ      = 0x04;

    /** Filter rate - 50 Hz */
    public final static int FREQ_50_HZ        = 0x05;

    /** Filter rate - 39.2 Hz */
    public final static int FREQ_39_2_HZ      = 0x06;

    /** Filter rate - 33.3 Hz */
    public final static int FREQ_33_3_HZ      = 0x07;

    /** Filter rate - 19.6 Hz */
    public final static int FREQ_19_6_HZ      = 0x08;

    /** Filter rate - 16.7 Hz */
    public final static int FREQ_16_7_HZ      = 0x09;

    /** Filter rate - 12.5 Hz */
    public final static int FREQ_12_5_HZ      = 0x0b;

    /** Filter rate - 10 Hz */
    public final static int FREQ_10_HZ        = 0x0c;

    /** Filter rate - 8.33 Hz */
    public final static int FREQ_8_33_HZ      = 0x0d;

    /** Filter rate - 6.25 Hz */
    public final static int FREQ_6_25_HZ      = 0x0e;

    /** Filter rate - 4.17 Hz */
    public final static int FREQ_4_17_HZ      = 0x0f;

    /*
    **  Thermocouple type codes
    */
    /** Thermocouple type J */
    public final static int TC_TYPE_J         = 0x00;

    /** Thermocouple type K */
    public final static int TC_TYPE_K         = 0x01;

    /** Thermocouple type T */
    public final static int TC_TYPE_T         = 0x02;

    /** Thermocouple type E */
    public final static int TC_TYPE_E         = 0x03;

    /** Thermocouple type R */
    public final static int TC_TYPE_R         = 0x04;

    /** Thermocouple type S */
    public final static int TC_TYPE_S         = 0x05;

    /** Thermocouple type B */
    public final static int TC_TYPE_B         = 0x06;

    /** Thermocouple type N */
    public final static int TC_TYPE_N         = 0x07;

    /*
    **  Gain codes
    */
    /** Gain - 1X */
    public final static int GAIN_1X           = 0x00;

    /** Gain - 2X */
    public final static int GAIN_2X           = 0x01;

    /** Gain - 4X */
    public final static int GAIN_4X           = 0x02;

    /** Gain - 8X */
    public final static int GAIN_8X           = 0x03;

    /** Gain - 16X */
    public final static int GAIN_16X          = 0x04;

    /** Gain - 32X */
    public final static int GAIN_32X          = 0x05;

    /** Gain - 64X */
    public final static int GAIN_64X          = 0x06;

    /** Gain - 128X */
    public final static int GAIN_128X         = 0x07;

    /*
    **  Voltage range codes
    */
    /** Voltage range +/- 10V */
    public final static int RANGE_10V         = GAIN_4X;

    /** Voltage range +/- 5V */
    public final static int RANGE_5V          = GAIN_8X;

    /** Voltage range +/- 2.5V */
    public final static int RANGE_2_5V        = GAIN_16X;

    /** Voltage range +/- 1.25V */
    public final static int RANGE_1_25V       = GAIN_32X;

    /*
    **  Voltage connection type codes
    */
    /** Voltage connection - differential */
    public final static int VCT_DIFFERENTIAL  = 0x00;

    /** Voltage connection - single-ended */
    public final static int VCT_SINGLE_ENDED  = 0x01;

    /** Voltage connection - grounded */
    public final static int VCT_GROUNDED      = 0x02;

    /** Voltage connection - calibration */
    public final static int VCT_CALIBRATION   = 0x03;

    /*
    **  RTD and thermistor connection codes
    */
    /** Therm connection - two wire, one sensor */
    public final static int TCT_2WIRE_1SENSOR = 0x00;

    /** Therm connection - two wire, two sensors */
    public final static int TCT_2WIRE_2SENSOR = 0x01;

    /** Therm connection - three wire */
    public final static int TCT_3WIRE         = 0x02;

    /** Therm connection - four wire */
    public final static int TCT_4WIRE         = 0x03;

    /*
    **  Current excitation codes
    */
    /** Current excitation - off */
    public final static int CEX_OFF           = 0x00;

    /** Current excitation - 10 microamps */
    public final static int CEX_10UA          = 0x01;

    /** Current excitation - 210 microamps */
    public final static int CEX_210UA         = 0x02;


   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        MCC_VID           = 0x09db,
        USB_TC_PID        = 0x0090,
        USB_TEMP_PID      = 0x008d,
        USB_TC_AI_PID     = 0x00bb;

    private final static int
        OP_DIO_CFG        = 0,
        OP_DIO_CFG_BIT    = 1,
        OP_DIO_IN         = 2,
        OP_DIO_OUT        = 3,
        OP_DIO_IN_BIT     = 4,
        OP_DIO_OUT_BIT    = 5,
        OP_ADC_IN         = 6,
        OP_ADC_SCAN       = 7,
        OP_CNTR_INIT      = 8,
        OP_CNTR_READ      = 9,
        OP_MEM_READ       = 10,
        OP_MEM_WRITE      = 11,
        OP_LED_BLINK      = 12,
        OP_RESET          = 13,
        OP_GET_STATUS     = 14,
        OP_SET_ITEM       = 15,
        OP_GET_ITEM       = 16,
        OP_CALIBRATE      = 17,
        OP_GET_BURN_STS   = 18,
        OP_CAL_CONFIG     = 19,
        OP_CAL_STEPS      = 20,
        OP_PREP_DWNLD     = 21,
        OP_CODE_WRITE     = 22,
        OP_SET_SERIAL     = 23,
        OP_CODE_READ      = 24,
        OP_ALARM_CONFIG   = 25,
        OP_ALARM_GET      = 26,
        N_OPS             = 27;

    private final static byte
        CMND_DIO_CFG      = 0x01,
        CMND_DIO_CFG_BIT  = 0x02,
        CMND_DIO_IN       = 0x03,
        CMND_DIO_OUT      = 0x04,
        CMND_DIO_IN_BIT   = 0x05,
        CMND_DIO_OUT_BIT  = 0x06,

        CMND_ADC_IN       = 0x18,
        CMND_ADC_SCAN     = 0x19,

        CMND_CNTR_INIT    = 0x20,
        CMND_CNTR_READ    = 0x21,

        CMND_MEM_READ     = 0x30,
        CMND_MEM_WRITE    = 0x31,

        CMND_LED_BLINK    = 0x40,
        CMND_RESET        = 0x41,
        CMND_GET_STATUS   = 0x44,
        CMND_SET_ITEM     = 0x49,
        CMND_GET_ITEM     = 0x4a,
        CMND_CALIBRATE    = 0x4b,
        CMND_GET_BURN_STS = 0x4c,
        CMND_CAL_CONFIG   = 0x4d,
        CMND_CAL_STEPS    = 0x4e,

        CMND_PREP_DWNLD   = 0x50,
        CMND_CODE_WRITE   = 0x51,
        CMND_SET_SERIAL   = 0x53,
        CMND_CODE_READ    = 0x55,

        CMND_ALARM_CONFIG = 0x6b,
        CMND_ALARM_GET    = 0x6c,

        CHAN_HI_PATH      = 0,
        CHAN_LO_PATH      = 1;

    private final static int RCVE_TMO = 500;   // Receive timeout (ms)
    private final static int[] pids = new int[N_DIDS];
    static {
        pids[USB_TC_AI_DID]    = USB_TC_AI_PID;
        pids[USB_TC_DID]       = USB_TC_PID;
        pids[USB_TEMP_DID]     = USB_TEMP_PID;
    }
    private final static Operation[] opern = new Operation[N_OPS];
    static {
        opern[OP_DIO_CFG]      = new Operation(CMND_DIO_CFG,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_DIO_CFG_BIT]  = new Operation(CMND_DIO_CFG_BIT,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_DIO_IN]       = new Operation(CMND_DIO_IN,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_DIO_OUT]      = new Operation(CMND_DIO_OUT,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_DIO_IN_BIT]   = new Operation(CMND_DIO_IN_BIT,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_DIO_OUT_BIT]  = new Operation(CMND_DIO_OUT_BIT,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_ADC_IN]       = new Operation(CMND_ADC_IN,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_ADC_SCAN]     = new Operation(CMND_ADC_SCAN,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_CNTR_INIT]    = new Operation(CMND_CNTR_INIT,
                                               USB_TC_AI_DID);
        opern[OP_CNTR_READ]    = new Operation(CMND_CNTR_READ,
                                               USB_TC_AI_DID);
        opern[OP_MEM_READ]     = new Operation(CMND_MEM_READ,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_MEM_WRITE]    = new Operation(CMND_MEM_WRITE,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_LED_BLINK]    = new Operation(CMND_LED_BLINK,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_RESET]        = new Operation(CMND_RESET,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_GET_STATUS]   = new Operation(CMND_GET_STATUS,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_SET_ITEM]     = new Operation(CMND_SET_ITEM,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_GET_ITEM]     = new Operation(CMND_GET_ITEM,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_CALIBRATE]    = new Operation(CMND_CALIBRATE,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_GET_BURN_STS] = new Operation(CMND_GET_BURN_STS,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_CAL_CONFIG]   = new Operation(CMND_CAL_CONFIG,
                                               USB_TC_AI_DID);
        opern[OP_CAL_STEPS]    = new Operation(CMND_CAL_STEPS,
                                               USB_TC_AI_DID);
        opern[OP_PREP_DWNLD]   = new Operation(CMND_PREP_DWNLD,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_CODE_WRITE]   = new Operation(CMND_CODE_WRITE,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_SET_SERIAL]   = new Operation(CMND_SET_SERIAL,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_CODE_READ]    = new Operation(CMND_CODE_READ,
                                               USB_TC_AI_DID, USB_TC_DID,
                                               USB_TEMP_DID);
        opern[OP_ALARM_CONFIG] = new Operation(CMND_ALARM_CONFIG,
                                               USB_TC_AI_DID);
        opern[OP_ALARM_GET]    = new Operation(CMND_ALARM_GET,
                                               USB_TC_AI_DID);
    }

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private final UsbComm usb = new UsbComm();
    private UsbCommPipe out, in;
    private int devId = -1;


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public MccUsb() throws UsbException
    {
    }


   /**
    ***************************************************************************
    **
    **  Gets the device ID
    **
    **  @return  The ID of the current open device, or -1 if no device open.
    **
    ***************************************************************************
    */
    public int getDevId()
    {
        return devId;
    }


   /**
    ***************************************************************************
    **
    **  Gets the USB communication object
    **
    **  @return  The UsbComm object associated with this MCC object.
    **
    ***************************************************************************
    */
    public UsbComm getUsbComm()
    {
        return usb;
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the device
    **
    **  @param  did     The device identifier code.
    **
    **  @param  serial  The serial number of the device, or null if the first
    **                  found device of this type is to be used.
    **
    **  @param  force   If true, claiming the device is forced, allowing it
    **                  to be opened even if another driver has claimed it.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    public void open(int did, String serial, boolean force)
        throws MccUsbException, UsbException
    {
        if (devId >= 0)
            throw new MccUsbException("Device connection already open");

        if (did < 0 || did >= pids.length)
            throw new MccUsbException("Invalid device identifier");

        if (usb.findDevice(MCC_VID, pids[did], serial) == null)
            throw new MccUsbException("Device not found");

        usb.claim(0, force);
        out = usb.open(0, 0x01);
        in = usb.open(0, 0x81);
        devId = did;
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the device
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    public void close() throws MccUsbException, UsbException
    {
        if (devId < 0)
            throw new MccUsbException("No device connection open");

        devId = -1;
        out.close();
        in.close();
        usb.release(0);
    }


   /**
    ***************************************************************************
    **
    **  Configures the DIO port
    **
    **  @param  direction  The direction to set for all the DIO lines.  If
    **                     0, all lines are output; otherwise they are input.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void dioConfig(int direction)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_DIO_CFG), (byte)direction};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Configures one DIO line
    **
    **  @param  bitno      The number (0 - 7) of the DIO line to configure.
    **
    **  @param  direction  The direction to set for the DIO line.  If 0,
    **                     the line is output; otherwise it is input.
    **    
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void dioConfigBit(int bitno, int direction)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_DIO_CFG_BIT), (byte)bitno, (byte)direction};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the DIO lines
    **
    **  @return  The state of the 8 DIO lines represented as a bit mask.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int dioIn() throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_DIO_IN)}, resp = new byte[2];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);

        return (leng == resp.length) ? resp[1] & 0xff : -1;
    }


   /**
    ***************************************************************************
    **
    **  Reads one DIO line
    **
    **  @param  bitno  The number (0 - 7) of the DIO line to read.
    **
    **  @return  The state (0 or 1) of the DIO line.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int dioInBit(int bitno)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_DIO_IN_BIT), (byte)bitno}, resp = new byte[2];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);

        return (leng == resp.length) ? resp[1] & 0xff : -1;
    }


   /**
    ***************************************************************************
    **
    **  Writes all the DIO lines
    **
    **  @param  value  A bit mask of the states to set in the 8 DIO lines.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void dioOut(int value)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_DIO_OUT), (byte)value};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Writes one DIO line
    **
    **  @param  bitno  The number (0 - 7) of the DIO line to write.
    **
    **  @param  value  The value (0 or 1) to write.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void dioOutBit(int bitno, int value)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_DIO_OUT_BIT), (byte)bitno, (byte)value};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the event counter (sets it to zero)
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void cntrInit() throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_CNTR_INIT)};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Reads the event counter
    **
    **  @return  The counter value.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
        public synchronized int cntrRead() throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_CNTR_READ)}, resp = new byte[5];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);

        return (leng == resp.length) ? Convert.bytesToInt(resp, 1) : 0;
    }


   /**
    ***************************************************************************
    **
    **  Reads an ADC channel
    **
    **  @param  chan   The channel to read:  0 - 3 are temperatures; 4 - 7
    **                 are temperatures for the USB-TC-AI and voltages for
    **                 the USB-TC; 128 - 129 are the CJCs (only one for the
    **                 USB-TC-AI).
    **
    **  @param  units  The units of the result: for temperature channels
    **                 0 = temperature, 1 = raw units; for voltage channels
    **                 0 = voltage, 1 = ADC counts; CJC is always temperature.
    **
    **  @return  The read value.  Special values are -8888.0 (open
    **           thermocouple), -9000.0 (initialization error),
    **           -9999.0 (floating point error).
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized float adcIn(int chan, int units)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_ADC_IN), (byte)chan, (byte)units},
               resp = new byte[5];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);

        if (leng != resp.length) return 0f;
        if (units == 1 && devId != USB_TC_DID && chan >= 4 && chan < 128) {
            return (float)Convert.bytesToInt(resp, 1);
        }
        return Convert.bytesToFloat(resp, 1);
    }


   /**
    ***************************************************************************
    **
    **  Scans ADC channels
    **
    **  @param  start  The first channel to read (0 - 7).  
    **
    **  @param  end    The last channel to read (0 - 7 and >= start).
    **
    **  @param  units  The units of the result: for temperature channels
    **                 0 = temperature, 1 = raw units; for voltage channels
    **                 0 = voltage, 1 = ADC counts.
    **
    **  @param  value  An array of sufficient size to contain the read values.
    **
    **  @return  The number of channels read.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int adcScan(int start, int end, int units,
                                    float[] value)
        throws MccUsbException, UsbException
    {
        byte code = getCmnd(OP_ADC_SCAN);
        if (end > 7) end = 7;
        if (start < 0) start = 0;
        int nChan = end - start + 1;
        if (nChan <= 0) return 0;

        byte[] cmnd = {code, (byte)start, (byte)end, (byte)units},
               resp = new byte[33];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);
        if (leng >= 4 * nChan + 1) {
            for (int j = 0; j < nChan; j++) {
                if (units == 1 && (j + start) >= 4)
                    value[j] = (float)Convert.bytesToInt(resp, 4 * j + 1);
                else
                    value[j] = Convert.bytesToFloat(resp, 4 * j + 1);
            }
        }
        else
            nChan = 0;

        return nChan;
    }


   /**
    ***************************************************************************
    **
    **  Blinks the LED
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void blink() throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_LED_BLINK)};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Resets the device
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
        public synchronized void reset() throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_RESET)};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Gets a configuration item
    **
    **  @param  item     The item code (ADC number in the range 0 - 3) of the
    **                   configuration data.
    **
    **  @param  subItem  The subitem code of the configuration data.
    **
    **  @return  The value of the configuration item.  Note that most items
    **           are unsigned bytes, but are returned as floating point.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized float getItem(int item, int subItem)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_GET_ITEM), (byte)item, (byte)subItem},
               resp = new byte[5];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);
        if (leng == resp.length) {
            if ((subItem >= CSI_VREF && subItem <= CSI_V_VALUE_2)
                || (subItem >= CSI_CH_0_COEF_0 && subItem <= CSI_CH_1_COEF_3))
                return Convert.bytesToFloat(resp, 1);
            else
                return (float)(resp[1] & 0xff);
        }
        else
            return 0f;
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel pair's sensor type
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @return  The encoded sensor type
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getSensorType(int chan) throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1, CSI_SENSOR_TYPE);
    }


   /**
    ***************************************************************************
    **
    **  Gets a thermistor or RTD channel pair's connection type
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @return  The encoded connection type
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getThermConnType(int chan)
        throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1, CSI_CONN_TYPE);
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel pair's filter update rate
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @return  The encoded update rate
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getFilterRate(int chan) throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1, CSI_FILTER_RATE);
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel pair's current excitation
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @return  The encoded current excitation
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getExcitation(int chan) throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1, CSI_EXCITATION);
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel pair's reference voltage
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @return  The reference voltage
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public float getReferenceVoltage(int chan)
        throws MccUsbException, UsbException
    {
        return getItem(chan >> 1, CSI_VREF);
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel's thermocouple type
    **
    **  @param  chan   The channel number.
    **
    **  @return  The encoded thermocouple type
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getTCType(int chan) throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1,
                            (chan & 1) == 0 ? CSI_CH_0_TC : CSI_CH_1_TC);
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel's gain
    **
    **  @param  chan   The channel number.
    **
    **  @return  The encoded gain
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getGain(int chan) throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1,
                            (chan & 1) == 0 ? CSI_CH_0_GAIN : CSI_CH_1_GAIN);
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel's coefficient
    **
    **  @param  chan   The channel number.
    **
    **  @param  coeff  The coefficient number (0 - 3).
    **
    **  @return  The encoded gain
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public float getCoefficient(int chan, int coeff)
        throws MccUsbException, UsbException
    {
        int subItem = (chan & 1) == 0 ? CSI_CH_0_COEF_0 : CSI_CH_1_COEF_0;
        return getItem(chan >> 1, subItem + 2 * (coeff & 3));
    }


   /**
    ***************************************************************************
    **
    **  Gets a voltage channel's connection type
    **
    **  @param  chan   The channel number.
    **
    **  @return  The encoded connection type.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public int getVoltageConnType(int chan)
        throws MccUsbException, UsbException
    {
        return (int)getItem(chan >> 1,
                            (chan & 1) == 0 ? CSI_CH_0_VCONN : CSI_CH_1_VCONN);
    }


   /**
    ***************************************************************************
    **
    **  Sets a configuration item
    **
    **  @param  item     The item code (ADC number in the range 0 - 3) of the
    **                   configuration data.
    **
    **  @param  subItem  The subitem code of the configuration data.
    **
    **  @param  value    The value to set.  Note that most items are unsigned
    **                   bytes, but are specified as floating point.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void setItem(int item, int subItem, float value)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_SET_ITEM), (byte)item, (byte)subItem,
                       0, 0, 0, 0};
        if ((subItem >= CSI_VREF && subItem <= CSI_V_VALUE_2)
              || (subItem >= CSI_CH_0_COEF_0 && subItem <= CSI_CH_1_COEF_3)) {
            Convert.floatToBytes(value, cmnd, 3);
            out.write(cmnd);
        }
        else {
            cmnd[3] = (byte)value;
            out.write(cmnd, 0, 4);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets a channel pair's sensor type
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @param  type   The sensor type to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setSensorType(int chan, int type)
        throws MccUsbException, UsbException
    {
        setItem(chan >> 1, CSI_SENSOR_TYPE, (float)type);
    }


   /**
    ***************************************************************************
    **
    **  Sets a thermistor or RTD channel pair's connection type
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @param  type   The connection type to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setThermConnType(int chan, int type)
        throws MccUsbException, UsbException
    {
        setItem(chan >> 1, CSI_CONN_TYPE, (float)type);
    }


   /**
    ***************************************************************************
    **
    **  Sets a channel pair's current excitation
    **
    **  @param  chan   The channel number of one of the pair.
    **
    **  @param  value  The encoded excitation value to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setExcitation(int chan, int value)
        throws MccUsbException, UsbException
    {
        setItem(chan >> 1, CSI_EXCITATION, (float)value);
    }


   /**
    ***************************************************************************
    **
    **  Sets a channel's thermocouple type
    **
    **  @param  chan   The channel number.
    **
    **  @param  type   The thermocouple type to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setTCType(int chan, int type)
        throws MccUsbException, UsbException
    {
        setItem(chan >> 1, (chan & 1) == 0 ? CSI_CH_0_TC : CSI_CH_1_TC,
                (float)type);
    }


   /**
    ***************************************************************************
    **
    **  Sets a channel's gain
    **
    **  @param  chan   The channel number.
    **
    **  @param  gain   The encoded gain value to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setGain(int chan, int gain)
        throws MccUsbException, UsbException
    {
        setItem(chan >> 1, (chan & 1) == 0 ? CSI_CH_0_GAIN : CSI_CH_1_GAIN,
                (float)gain);
    }


   /**
    ***************************************************************************
    **
    **  Sets a channel's coefficient
    **
    **  @param  chan   The channel number.
    **
    **  @param  coeff  The coefficient number (0 - 3).
    **
    **  @param  value  The value to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setCoefficient(int chan, int coeff, float value)
        throws MccUsbException, UsbException
    {
        int subItem = (chan & 1) == 0 ? CSI_CH_0_COEF_0 : CSI_CH_1_COEF_0;
        setItem(chan >> 1, subItem + 2 * (coeff & 3), value);
    }


   /**
    ***************************************************************************
    **
    **  Sets a voltage channel's connection type
    **
    **  @param  chan   The channel number.
    **
    **  @param  type   The connection type to set.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public void setVoltageConnType(int chan, int type)
        throws MccUsbException, UsbException
    {
        setItem(chan >> 1, (chan & 1) == 0 ? CSI_CH_0_VCONN : CSI_CH_1_VCONN,
                (float)type);
    }


   /**
    ***************************************************************************
    **
    **  Configures an alarm
    **
    **  <p>
    **  This routine configures the temperature and voltage alarms.  There are
    **  eight alarms available, corresponding to the eight available DIO bits.
    **  If an alarm is enabled, its associated DIO line will be configured as
    **  an output on power on and driven to its non-alarmed state.  The alarm
    **  conditions are evaluated every measurement cycle.
    **
    **  <p>
    **  <b>Input options:</b>
    **  <table cellspacing=1>
    **    <thead>
    **      <tr><th>Bit</th><th>Description</th></tr>
    **    </thead>
    **    <tbody>
    **      <tr><td>[0-2]:</td><td>input channel (0-7)</td></tr>
    **      <tr><td align=right>3:</td><td>units: 0 = temperature, 1 = raw</td></tr>
    **      <tr><td>[4-6]:</td><td>threshold type:</td></tr>
    **      <tr><td></td><td>000 - alarm when reading > value1</td></tr>
    **      <tr><td></td><td>001 - alarm when reading > value1,
    **                             reset when reading < value2</td></tr>
    **      <tr><td></td><td>010 - alarm when reading < value1</td></tr>
    **      <tr><td></td><td>011 - alarm when reading < value1,
    **                             reset when reading > value2</td></tr>
    **      <tr><td></td><td>100 - alarm when reading < value1
    **                                             or > value2</td></tr>
    **      <tr><td></td><td>101 - not used</td></tr>
    **      <tr><td></td><td>110 - not used</td></tr>
    **      <tr><td></td><td>111 - not used</td></tr>
    **      <tr><td align=right>7:</td><td>not used, must be 0</td></tr>
    **    </tbody>
    **  </table>
    **
    **  <p>
    **  <b>Output options:</b>
    **  <table cellspacing=1>
    **    <thead>
    **      <tr><th>Bit</th><th>Description</th></tr>
    **    </thead>
    **    <tbody>
    **      <tr><td align=right>0:</td><td>1 = enable alarm, 0 = disable alarm</td></tr>
    **      <tr><td align=right>1:</td><td>0 = active low alarm, 1 = active high alarm</td></tr>
    **      <tr><td>[2-7]:</td><td>not used</td></tr>
    **    </tbody>
    **  </table>
    **
    **  @param  alarm   The number (0 - 7) of the alarm to configure.
    **
    **  @param  iOptns  Input options.
    **
    **  @param  oOptns  Output options.
    **
    **  @param  value1  The first alarm condition value.
    **
    **  @param  value2  The second alarm condition value.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void configAlarm(int alarm, int iOptns, int oOptns,
                                         float value1, float value2)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = new byte[12];
        cmnd[0] = getCmnd(OP_ALARM_CONFIG);
        cmnd[1] = (byte)alarm;
        cmnd[2] = (byte)iOptns;
        cmnd[3] = (byte)oOptns;
        Convert.floatToBytes(value1, cmnd, 4);
        Convert.floatToBytes(value2, cmnd, 8);
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Gets an alarm configuration
    **
    **  @param  alarm   The number (0 - 7) of the alarm.
    **
    **  @param  optns   A two-element array to receive the alarm's input [0]
    **                  and output [1] options.
    **
    **  @param  values  A two-element array to receive the alarm's value1 [0]
    **                  and value2 [1] condition values.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void getAlarmConfig(int alarm, int[] optns,
                                            float[] values)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_ALARM_GET), (byte)alarm}, resp = new byte[11];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);
        if (leng == resp.length) {
            optns[0] = resp[1] & 0xff;
            optns[1] = resp[2] & 0xff;
            values[0] = Convert.bytesToFloat(resp, 3);
            values[1] = Convert.bytesToFloat(resp, 7);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of calibration steps
    **
    **  @param  steps  A two-element array to receive the number of steps
    **                 used in temperature [0] and voltage [1] calibrations.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void calSteps(int[] steps)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_CAL_STEPS)}, resp = new byte[3];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);
        if (leng == resp.length) {
            steps[0] = resp[1] & 0xff;
            steps[1] = resp[2] & 0xff;
        }
        else
            steps[0] = steps[1] = 0;
    }


   /**
    ***************************************************************************
    **
    **  Configures the calibration MUX
    **
    **  <p>
    **  This configures the calibration mux for voltage channels 4-7.  To use
    **  the calibration input, the channel must be configured for calibration
    **  using setItem().
    **
    **  @param  gain      The gain selector.
    **
    **  @param  polarity  The polarity: 0 = positive, 1 = negative.
    **
    **  @param  path      The channel path: 0 = channel high, 1 = channel low
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void calConfig(int gain, int polarity, int path)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_CAL_CONFIG), (byte)gain, (byte)polarity,
                       (byte)path};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the ADC channels
    **
    **  <p>
    **  This is used after reconfiguring the channel(s).  This may take up to
    **  several seconds, and the completion may be determined by polling the
    **  status with getStatus().  Temperature readings will not be updated
    **  while the calibration is ongoing, but DIO operations may be performed.
    **  The device will not accept setItem() or memWrite() commands while
    **  calibration is being performed.  Additionally, any calibrate()
    **  commands with type argument other than 255 (calibrate abort) will be
    **  ignored.  After a calibration is aborted, getStatus() will indicate a
    **  calibration error until a new calibration is started.  Once voltage
    **  calibration has been completed successfully, the calibration path
    **  location in the isolated microcontroller's EEPROM will be updatated to
    **  indicate which path was used for the most recent calibration.
    **
    **  @param  type  The type of calibration to perform: 0 = temperature,
    **                1 = voltage, 255 = abort.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void calibrate(int type)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_CALIBRATE), (byte)type, CHAN_HI_PATH};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Gets the calibration status
    **
    **  @return  The status byte, formatted as follows:
    **
    **             bits 0-5:  Calibration progress (0 = no calibration)
    **             bit 6:     Calibration type: 0 = temperature, 1 = voltage
    **             bit 7:     Calibration status: 0 = good, 1 = error
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int getStatus() throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_GET_STATUS)}, resp = new byte[2];
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO);

        return (leng == resp.length) ? resp[1] & 0xff : -1;
    }


   /**
    ***************************************************************************
    **
    **  Gets the burnout status and clears it under a mask
    **
    **  <p>
    **  This routine returns the status of burnout detection for thermocouple
    **  channels.  The return value is a bitmap indicating the burnout
    **  detection status for all 8 channels.  Individual bits will be set if
    **  an open circuit has been detected on that channel.  The bits will be
    **  cleared after the call using the mask parameter.  If a bit is set, the
    **  corresponding bit in the status will be left at its current value.
    **
    **  @param  mask  The mask of status bits to clear (0 = clear).
    **
    **  @return  The burnout status (before being cleared).
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int getBurnout(int mask)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_GET_BURN_STS), (byte)mask},
               resp = new byte[2];
        out.write(cmnd, 0, 1);
        int leng = in.read(resp, RCVE_TMO);
        if (leng == resp.length) {
            out.write(cmnd);
            return resp[1] & 0xff;
        }

        return 0;
    }


   /**
    ***************************************************************************
    **
    **  Reads device memory
    **
    **  @param  addr   The memory address.
    **
    **  @param  type   The memory type: 0 = main microcontroller,
    **                 1 = isolated microcontroller.
    **
    **  @param  count  The number of bytes to read (62 max for main, 60 max
    **                 for isolated).
    **
    **  @param  data   An array of sufficient size to contain the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int readMemory(int addr, int type, int count,
                                       byte[] data)
        throws MccUsbException, UsbException
    {
        if (count > 62 && type == 0) count = 62;
        else if (count > 60 && type != 0) count = 60;
        byte[] cmnd = {getCmnd(OP_MEM_READ), 0, 0, (byte)type, (byte)count},
               resp = new byte[count + 1];
        Convert.shortToBytes((short)addr, cmnd, 1);
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO) - 1;
        System.arraycopy(resp, 1, data, 0, leng);

        return leng;
    }


   /**
    ***************************************************************************
    **
    **  Writes device memory
    **
    **  <p>
    **  Locations 0 - 255 are available on the main microcontroller.
    **
    **  @param  addr   The memory address.
    **
    **  @param  type   The memory type: 0 = main microcontroller,
    **                 1 = isolated microcontroller.
    **
    **  @param  count  The number of bytes to write (59 max).
    **
    **  @param  data   The data to write.
    **
    **  @return  The number of bytes written (?).
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int writeMemory(int addr, int type, int count,
                                        byte[] data)
        throws MccUsbException, UsbException
    {
        byte code = getCmnd(OP_MEM_READ);
        if (count > 59) count = 59;
        if ((addr + count) > 0xff) return 0;

        byte[] cmnd = new byte[count + 5];
        cmnd[0] = code;
        Convert.shortToBytes((short)addr, cmnd, 1);
        cmnd[3] = (byte)type;
        cmnd[4] = (byte)count;
        System.arraycopy(data, 0, cmnd, 5, count);
        return out.write(cmnd) - 5;
    }


   /**
    ***************************************************************************
    **
    **  Writes the serial number
    **
    **  <p>
    **  Note: The new serial number will be programmed but not used until
    **  hardware reset.
    **
    **  @param  serial  The serial number to write.  At most 8 bytes are
    **                  written, and shorter strings are padded with blanks.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void writeSerial(String serial)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = new byte[9];
        cmnd[0] = getCmnd(OP_SET_SERIAL);
        byte[] bSerial = serial.getBytes();
        for (int j = 1; j < cmnd.length; j++) {
            if (j <= bSerial.length) cmnd[j] = bSerial[j - 1];
            else cmnd[j] = 0x20;
        }
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Prepares for a download
    **
    **  <p>
    **  This routine puts the device into code update mode.  The unlock code
    **  must be correct as a further safety device.  Call this once before
    **  sending code with writeCode().  If not in code update mode, any call
    **  to writeCode() will be ignored.  A reset() call must be issued after
    **  the code download in order to return the device to operation with
    **  the new code.
    **
    **  @param  code   The unlock code (must be 0xad).
    **
    **  @param  micro  The microcontroller to load (0 = main, 1 = isolated).
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void prepareDownload(int code, int micro)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_PREP_DWNLD), (byte)code, (byte)micro};
        out.write(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Writes firmware code
    **
    **  <p>
    **  This routine writes to the program memory in the device, and is not
    **  accepted unless the device is in update mode.  It will normally be
    **  used when downloading a new hex file, so it supports memory ranges
    **  that may be found in the hex file.  The microcontroller that is being
    **  written to is selected with the prepareDownload() routine.
    **
    **  <p>
    **    The address ranges are:
    **
    **  <p>
    **    0x000000 - 0x0075FF:  Microcontroller FLASH program memory
    **  <br>
    **    0x200000 - 0x200007:  ID memory (serial number is stored here on
    **                          main micro)
    **  <br>
    **    0x300000 - 0x30000F:  CONFIG memory (processor configuration data)
    **  <br>
    **    0xF00000 - 0xF03FFF:  EEPROM memory
    **
    **  <p>
    **    FLASH program memory: The device must receive data in 64-byte
    **    segments that begin on a 64-byte boundary.  The data is sent in
    **    messages containing 32 bytes.  count must always equal 32.
    **
    **  <p>
    **    Other memory: Any number of bytes up to the maximum (32) may be sent.
    **
    **  @param  addr   The start address to be written to.
    **
    **  @param  count  The number of bytes to write.
    **
    **  @param  data   The data to be written.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized void writeCode(int addr, int count, byte[] data)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = new byte[37];
        cmnd[0] = getCmnd(OP_CODE_WRITE);
        Convert.intToBytes(addr, cmnd, 1);
        count &= 0xff;
        if (count > 32) count = 32;
        cmnd[4] = (byte)count;
        System.arraycopy(data, 0, cmnd, 5, count);
        out.write(cmnd, 0, count + 5);
    }


   /**
    ***************************************************************************
    **
    **  Reads firmware code
    **
    **  @param  addr   The start address to be read from.
    **
    **  @param  count  The number of bytes to read.
    **
    **  @param  data   An array large enough to hold the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  MccUsbException
    **  @throws  UsbException
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    public synchronized int readCode(int addr, int count, byte[] data)
        throws MccUsbException, UsbException
    {
        byte[] cmnd = {getCmnd(OP_CODE_READ), 0, 0, 0, 0}, resp = new byte[63];
        Convert.intToBytes(addr, cmnd, 1);
        count &= 0xff;
        if (count > 62) count = 62;
        cmnd[4] = (byte)count;
        out.write(cmnd);
        int leng = in.read(resp, RCVE_TMO) - 1;
        System.arraycopy(resp, 1, data, 0, count);

        return leng;
    }


   /**
    ***************************************************************************
    **
    **  Checks operation validity and gets command code
    **
    **  First it is checked that a device is open.  If so, it is then checked
    **  that the operation is supported on that device.
    **
    **  @param  opcode  The operation code.  If negative, only the open check
    **                  is performed.
    **
    **  @return  The command code for the operation.
    **
    **  @throws  MccUsbException if no device is open
    **  @throws  UnsupportedOperationException
    **
    ***************************************************************************
    */
    private byte getCmnd(int opcode) throws MccUsbException
    {
        if (devId < 0)
            throw new MccUsbException("No device connection open");
        Operation oper = opern[opcode];
        if (((oper.mask >> devId) & 1) == 0)
            throw new UnsupportedOperationException();

        return oper.cmnd;
    }

}
