package org.lsst.ccs.drivers.mcc;

import java.io.IOException;
import java.util.Scanner;
import javax.usb.UsbException;
import jline.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test a Measurement Computing DAQ device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestMccUsb implements CmndProc.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_OPEN        = 0,
        CMD_CLOSE       = 1,
        CMD_DIOCFG      = 2,
        CMD_DIOCFGBIT   = 3,
        CMD_DIOIN       = 4,
        CMD_DIOINBIT    = 5,
        CMD_DIOOUT      = 6,
        CMD_DIOOUTBIT   = 7,
        CMD_ADIN        = 8,
        CMD_ADSCAN      = 9,
        CMD_CNTRINIT    = 10,
        CMD_CNTRREAD    = 11,
        CMD_BLINK       = 12,
        CMD_RESET       = 13,
        CMD_STATUS      = 14,
        CMD_ITEMSET     = 15,
        CMD_ITEMGET     = 16,
        CMD_ALARMSET    = 17,
        CMD_ALARMGET    = 18,
        CMD_SHOWCHAN    = 19,
        CMD_SHOWBURN    = 20,
        CMD_SHOWCAL     = 21,
        CMD_CALCFG      = 22,
        CMD_CALIBRATE   = 23,
        CMD_SHOWDEV     = 24,
        NUM_CMDS        = 25;

    /*
    **  Command help text
    */
    private final static String[] helpOpen = {
        "Open device connection",
        "open [<device>] [<force>] [<serial>]",
        "device  The name of the MCC device to connect to",
        "force   If present and non-zero, force the interface claim",
        "serial  The serial number of the MCC device (default any)",
    };

    private final static String[] helpClose = {
        "Close device connection",
        "close",
    };

    private final static String[] helpDiocfg = {
        "Configure the DIO port",
        "diocfg <value>",
        "value  The mask of lines to set as input",
    };

    private final static String[] helpDiocfgbit = {
        "Configure a DIO line",
        "diocfgbit <line> <value>",
        "line   The line to configure",
        "value  The value to set (0 = out, 1 = in)",
    };

    private final static String[] helpDioin = {
        "Read the DIO port and display the value",
        "dioin",
    };

    private final static String[] helpDioinbit = {
        "Read a DIO line and display its value",
        "dioinbit <line>",
        "line  The line to read",
    };

    private final static String[] helpDioout = {
        "Write to the DIO port",
        "dioout <value>",
        "value  The value to write",
    };

    private final static String[] helpDiooutbit = {
        "Write to a DIO line",
        "diooutbit <line> <value>",
        "line   The line to write to",
        "value  The value to write",
    };

    private final static String[] helpAdin = {
        "Read an ADC channel and display its value",
        "adin <chan> <units>",
        "chan   The ADC channel to read",
        "units  The units to use (0 = temperature, 1 = voltage",
    };

    private final static String[] helpAdscan = {
        "Scan ADC channels and display their values",
        "adscan <first> <last> <units>",
        "first  The first ADC channel to read",
        "last   The last ADC channel to read",
        "units  The units to use (0 = temperature, 1 = voltage",
    };

    private final static String[] helpCntrinit = {
        "Initialize the counter",
        "cntrinit",
    };

    private final static String[] helpCntrread = {
        "Read and display the counter value",
        "cntrread",
    };

    private final static String[] helpBlink = {
        "Blink the LED",
        "blink",
    };

    private final static String[] helpReset = {
        "Reset the device",
        "reset",
    };

    private final static String[] helpStatus = {
        "Read and display the status byte",
        "status",
    };

    private final static String[] helpItemset = {
        "Set a configuration item",
        "itemset <item> <subitem> <value>",
        "item     The name or number of the item to be configured",
        "subitem  The name or number of the subitem",
        "value    The value to set",
    };

    private final static String[] helpItemget = {
        "Get and display a configuration item",
        "itemget <item> <subitem>",
        "item     The name or number of the configured item",
        "subitem  The name or number of the subitem",
    };

    private final static String[] helpAlarmset = {
        "Set the configuration for an alarm",
        "alarmset <alarm> <ioptns> <ooptns> <value1> <value2>",
        "alarm   The alarm number",
        "ioptns  The input options",
        "ooptns  The output options",
        "value1  The first alarm value",
        "value2  The second alarm value",
    };

    private final static String[] helpAlarmget = {
        "Get and display an alarm configuration",
        "alarmget <alarm>",
        "alarm  The alarm number",
    };

    private final static String[] helpShowchan = {
        "Get and display channel configuration",
        "showchan [<chan>] [<all>]",
        "chan  The channel number (0 - 7), or * for all channels",
        "all   If present and non-zero, display all configuration data",
    };

    private final static String[] helpShowdev = {
        "Display open device name",
        "showdev",
    };

    private final static String[] helpShowburn = {
        "Get, display and reset the burnout status",
        "showburn <mask>",
        "mask  The mask of channels to not reset (default 0x0f)",
    };

    private final static String[] helpShowcal = {
        "Get and display the number of calibration steps",
        "showcal",
    };

    private final static String[] helpCalcfg = {
        "Configure a calibration",
        "calcfg <gain> <polar> <path>",
        "gain   The gain selector",
        "polar  The polarity:  0 = positive, 1 = negative",
        "path   The path to use:  0 = channel high, 1 = channel low",
    };

    private final static String[] helpCalibrate = {
        "Perform a calibration",
        "calibrate <type>",
        "type  The type of calibration:  0 = temp, 1 = voltage, 2 = abort",
    };

    /*
    **  Command table
    */
    private final static CmndProc.Command cmnd;
    static {
        cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("open",        CMD_OPEN,        helpOpen);
        cmnd.add("close",       CMD_CLOSE,       helpClose);
        cmnd.add("diocfg",      CMD_DIOCFG,      helpDiocfg);
        cmnd.add("diocfgbit",   CMD_DIOCFGBIT,   helpDiocfgbit);
        cmnd.add("dioin",       CMD_DIOIN,       helpDioin);
        cmnd.add("dioinbit",    CMD_DIOINBIT,    helpDioinbit);
        cmnd.add("dioout",      CMD_DIOOUT,      helpDioout);
        cmnd.add("diooutbit",   CMD_DIOOUTBIT,   helpDiooutbit);
        cmnd.add("adin",        CMD_ADIN,        helpAdin);
        cmnd.add("adscan",      CMD_ADSCAN,      helpAdscan);
        cmnd.add("cntrinit",    CMD_CNTRINIT,    helpCntrinit);
        cmnd.add("cntrread",    CMD_CNTRREAD,    helpCntrread);
        cmnd.add("blink",       CMD_BLINK,       helpBlink);
        cmnd.add("reset",       CMD_RESET,       helpReset);
        cmnd.add("itemset",     CMD_ITEMSET,     helpItemset);
        cmnd.add("itemget",     CMD_ITEMGET,     helpItemget);
        cmnd.add("status",      CMD_STATUS,      helpStatus);
        cmnd.add("alarmset",    CMD_ALARMSET,    helpAlarmset);
        cmnd.add("alarmget",    CMD_ALARMGET,    helpAlarmget);
        cmnd.add("showdev",     CMD_SHOWDEV,     helpShowdev);
        cmnd.add("showchan",    CMD_SHOWCHAN,    helpShowchan);
        cmnd.add("showburn",    CMD_SHOWBURN,    helpShowburn);
        cmnd.add("showcal",     CMD_SHOWCAL,     helpShowcal);
        cmnd.add("calcfg",      CMD_CALCFG,      helpCalcfg);
        cmnd.add("calibrate",   CMD_CALIBRATE,   helpCalibrate);
    }

    /*
    **  Lookup tables
    */
    private final static CmndProc.Lookup devNames;
    static {
        devNames = new CmndProc.Lookup(2);
        devNames.add("tc-ai", MccUsb.USB_TC_AI_DID);
        devNames.add("tc",    MccUsb.USB_TC_DID);
    }
    private final static CmndProc.Lookup itemNames;
    static {
        itemNames = new CmndProc.Lookup(4);
        itemNames.add("adc0", 0);
        itemNames.add("adc1", 1);
        itemNames.add("adc2", 2);
        itemNames.add("adc3", 3);
    }
    private final static CmndProc.Lookup subiNames;
    static {
        subiNames = new CmndProc.Lookup(25);
        subiNames.add("senstype",   MccUsb.CSI_SENSOR_TYPE);
        subiNames.add("conntype",   MccUsb.CSI_CONN_TYPE);
        subiNames.add("filtrate",   MccUsb.CSI_FILTER_RATE);
        subiNames.add("excitation", MccUsb.CSI_EXCITATION);
        subiNames.add("vref",       MccUsb.CSI_VREF);
        subiNames.add("ival0",      MccUsb.CSI_I_VALUE_0);
        subiNames.add("ival1",      MccUsb.CSI_I_VALUE_1);
        subiNames.add("ival2",      MccUsb.CSI_I_VALUE_2);
        subiNames.add("vval0",      MccUsb.CSI_V_VALUE_0);
        subiNames.add("vval1",      MccUsb.CSI_V_VALUE_1);
        subiNames.add("vval2",      MccUsb.CSI_V_VALUE_2);
        subiNames.add("tc0",        MccUsb.CSI_CH_0_TC);
        subiNames.add("tc1",        MccUsb.CSI_CH_1_TC);
        subiNames.add("gain0",      MccUsb.CSI_CH_0_GAIN);
        subiNames.add("gain1",      MccUsb.CSI_CH_1_GAIN);
        subiNames.add("coef00",     MccUsb.CSI_CH_0_COEF_0);
        subiNames.add("coef01",     MccUsb.CSI_CH_1_COEF_0);
        subiNames.add("coef10",     MccUsb.CSI_CH_0_COEF_1);
        subiNames.add("coef11",     MccUsb.CSI_CH_1_COEF_1);
        subiNames.add("coef20",     MccUsb.CSI_CH_0_COEF_2);
        subiNames.add("coef21",     MccUsb.CSI_CH_1_COEF_2);
        subiNames.add("coef30",     MccUsb.CSI_CH_0_COEF_3);
        subiNames.add("coef31",     MccUsb.CSI_CH_1_COEF_3);
        subiNames.add("vconn0",     MccUsb.CSI_CH_0_VCONN);
        subiNames.add("vconn1",     MccUsb.CSI_CH_1_VCONN);
    }
    private final static CmndProc.Lookup stypNames;
    static {
        stypNames = new CmndProc.Lookup(6);
        stypNames.add("rtd",           MccUsb.STP_RTD);
        stypNames.add("thermistor",    MccUsb.STP_THERMISTOR);
        stypNames.add("thermocouple",  MccUsb.STP_THERMOCOUPLE);
        stypNames.add("semiconductor", MccUsb.STP_SEMICONDUCTOR);
        stypNames.add("disabled",      MccUsb.STP_DISABLED);
        stypNames.add("voltage",       MccUsb.STP_VOLTAGE);
    }
    private final static CmndProc.Lookup stypsNames;
    static {
        stypsNames = new CmndProc.Lookup(6);
        stypsNames.add("rtd",     MccUsb.STP_RTD);
        stypsNames.add("thermis", MccUsb.STP_THERMISTOR);
        stypsNames.add("thermoc", MccUsb.STP_THERMOCOUPLE);
        stypsNames.add("semicon", MccUsb.STP_SEMICONDUCTOR);
        stypsNames.add("disabld", MccUsb.STP_DISABLED);
        stypsNames.add("voltage", MccUsb.STP_VOLTAGE);
    }
    private final static CmndProc.Lookup frateNames;
    static {
        frateNames = new CmndProc.Lookup(14);
        frateNames.add("500 Hz",  MccUsb.FREQ_500_HZ);
        frateNames.add("250 Hz",  MccUsb.FREQ_250_HZ);
        frateNames.add("125 Hz",  MccUsb.FREQ_125_HZ);
        frateNames.add("62.5 Hz", MccUsb.FREQ_62_5_HZ);
        frateNames.add("50 Hz",   MccUsb.FREQ_50_HZ);
        frateNames.add("39.2 Hz", MccUsb.FREQ_39_2_HZ);
        frateNames.add("33.3 Hz", MccUsb.FREQ_33_3_HZ);
        frateNames.add("19.6 Hz", MccUsb.FREQ_19_6_HZ);
        frateNames.add("16.7 Hz", MccUsb.FREQ_16_7_HZ);
        frateNames.add("12.5 Hz", MccUsb.FREQ_12_5_HZ);
        frateNames.add("10 Hz",   MccUsb.FREQ_10_HZ);
        frateNames.add("8.33 Hz", MccUsb.FREQ_8_33_HZ);
        frateNames.add("6.25 Hz", MccUsb.FREQ_6_25_HZ);
        frateNames.add("4.17 Hz", MccUsb.FREQ_4_17_HZ);
    }
    private final static CmndProc.Lookup tctypNames;
    static {
        tctypNames = new CmndProc.Lookup(8);
        tctypNames.add("J", MccUsb.TC_TYPE_J);
        tctypNames.add("K", MccUsb.TC_TYPE_K);
        tctypNames.add("T", MccUsb.TC_TYPE_T);
        tctypNames.add("E", MccUsb.TC_TYPE_E);
        tctypNames.add("R", MccUsb.TC_TYPE_R);
        tctypNames.add("S", MccUsb.TC_TYPE_S);
        tctypNames.add("B", MccUsb.TC_TYPE_B);
        tctypNames.add("N", MccUsb.TC_TYPE_N);
    }
    private final static CmndProc.Lookup gainNames;
    static {
        gainNames = new CmndProc.Lookup(8);
        gainNames.add("1X",   MccUsb.GAIN_1X);
        gainNames.add("2X",   MccUsb.GAIN_2X);
        gainNames.add("4X",   MccUsb.GAIN_4X);
        gainNames.add("8X",   MccUsb.GAIN_8X);
        gainNames.add("16X",  MccUsb.GAIN_16X);
        gainNames.add("32X",  MccUsb.GAIN_32X);
        gainNames.add("64X",  MccUsb.GAIN_64X);
        gainNames.add("128X", MccUsb.GAIN_128X);
    }
    private final static CmndProc.Lookup rangeNames;
    static {
        rangeNames = new CmndProc.Lookup(4);
        rangeNames.add("10V",   MccUsb.RANGE_10V);
        rangeNames.add("5V",    MccUsb.RANGE_5V);
        rangeNames.add("2.5V",  MccUsb.RANGE_2_5V);
        rangeNames.add("1.25V", MccUsb.RANGE_1_25V);
    }
    private final static CmndProc.Lookup vconnNames;
    static {
        vconnNames = new CmndProc.Lookup(4);
        vconnNames.add("differential", MccUsb.VCT_DIFFERENTIAL);
        vconnNames.add("single-ended", MccUsb.VCT_SINGLE_ENDED);
        vconnNames.add("grounded",     MccUsb.VCT_GROUNDED);
        vconnNames.add("calibration",  MccUsb.VCT_CALIBRATION);
    }
    private final static CmndProc.Lookup vconsNames;
    static {
        vconsNames = new CmndProc.Lookup(4);
        vconsNames.add("differ",  MccUsb.VCT_DIFFERENTIAL);
        vconsNames.add("snglend", MccUsb.VCT_SINGLE_ENDED);
        vconsNames.add("ground",  MccUsb.VCT_GROUNDED);
        vconsNames.add("calib",   MccUsb.VCT_CALIBRATION);
    }
    private final static CmndProc.Lookup excitNames;
    static {
        excitNames = new CmndProc.Lookup(3);
        excitNames.add("off",    MccUsb.CEX_OFF);
        excitNames.add("10 uA",  MccUsb.CEX_10UA);
        excitNames.add("210 uA", MccUsb.CEX_210UA);
    }
    private final static CmndProc.Lookup tconnNames;
    static {
        tconnNames = new CmndProc.Lookup(4);
        tconnNames.add("2-wire, 1-sensor", MccUsb.TCT_2WIRE_1SENSOR);
        tconnNames.add("2-wire, 2-sensor", MccUsb.TCT_2WIRE_2SENSOR);
        tconnNames.add("3-wire",           MccUsb.TCT_3WIRE);
        tconnNames.add("4-wire",           MccUsb.TCT_4WIRE);
    }
    private final static CmndProc.Lookup tconsNames;
    static {
        tconsNames = new CmndProc.Lookup(4);
        tconsNames.add("2-wire1", MccUsb.TCT_2WIRE_1SENSOR);
        tconsNames.add("2-wire2", MccUsb.TCT_2WIRE_2SENSOR);
        tconsNames.add("3-wire",  MccUsb.TCT_3WIRE);
        tconsNames.add("4-wire",  MccUsb.TCT_4WIRE);
    }

    /*
    **  Private fields
    */
    private final Output out = new ConsOut();
    private final ConsoleReader reader = new ConsoleReader();
    private final CmndProc proc = new CmndProc();
    private final MccUsb mcc = new MccUsb();
    private final boolean debug;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestMccUsb(boolean dbg) throws UsbException, IOException
    {
        debug = dbg;
        proc.add(this, cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        try {
            (new TestMccUsb(args.length > 0)).run();
        }
        catch (UsbException e) {
            System.out.println(e);
        }
        catch (IOException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run() throws IOException
    {
        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            try {
                if (!proc.process(line)) break;
            }
            catch (RuntimeException e) {
                if (debug) e.printStackTrace(System.out);
                else out.println(e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing - exception-free version
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, Scanner scan)
    {
        try {
            return dispatchCmnd(code, scan);
        }
        catch (UsbException e) {
            out.println(e.toString());
            return true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing
    **
    ***************************************************************************
    */
    private boolean dispatchCmnd(int code, Scanner scan) throws UsbException
    {
        int found;
        Object[] args = new Object[16];

        switch (code) {

        case CMD_OPEN:
            if ((found = CmndProc.scanArgs(scan, "sis", args)) >= 0) {
                if ((found & 0x01) == 0) {
                    showNames("device", devNames);
                    break;
                }
                int devId = devNames.encode((String)args[0], true);
                if (devId < 0) break;
                String serialNo = null;
                if ((found & 0x04) != 0)
                    serialNo = (String)args[2];
                mcc.open(devId, serialNo,
                         ((found & 0x02) != 0 ? (Integer)args[1] : 0) != 0);
            }
            break;
        
        case CMD_CLOSE:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                mcc.close();
            }
            break;

        case CMD_DIOCFG:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                mcc.dioConfig((Integer)args[0]);
            }
            break;
        
        case CMD_DIOCFGBIT:
            if (CmndProc.scanArgs(scan, "II", args) >= 0) {
                mcc.dioConfigBit((Integer)args[0], (Integer)args[1]);
            }
            break;
        
        case CMD_DIOIN:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                out.format("Value = 0x%02x\n", mcc.dioIn());
            }
            break;
        
        case CMD_DIOINBIT:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                out.println("Value = " + mcc.dioInBit((Integer)args[0]));
            }
            break;
        
        case CMD_DIOOUT:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                mcc.dioOut((Integer)args[0]);
            }
            break;
        
        case CMD_DIOOUTBIT:
            if (CmndProc.scanArgs(scan, "II", args) >= 0) {
                mcc.dioOutBit((Integer)args[0], (Integer)args[1]);
            }
            break;
        
        case CMD_ADIN:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0) {
                int chan = (Integer)args[0];
                int units = (found & 0x02) != 0 ? (Integer)args[1] : 0;
                out.println("Value = " + mcc.adcIn(chan, units));
            }
            break;
        
        case CMD_ADSCAN:
            if ((found = CmndProc.scanArgs(scan, "IIi", args)) >= 0) {
                float[] value = new float[8];
                int units = (found & 0x04) != 0 ? (Integer)args[2] : 0;
                int fChan = (Integer)args[0];
                int count = mcc.adcScan(fChan, (Integer)args[1], units, value);
                if (count <= 0) break;
                out.print("Values =");
                int posn = 8, line = 0;
                for (int j = 0; j < count; j++) {
                    String sValue = " " + value[j];
                    posn += sValue.length();
                    if (posn > 80) {
                        out.print("\n        ");
                        posn = 8;
                    }
                    out.print(sValue);
                }
                out.println();
            }
            break;

        case CMD_CNTRINIT:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                mcc.cntrInit();
            }
            break;

        case CMD_CNTRREAD:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                out.println("Value = " + mcc.cntrRead());
            }
            break;
        
        case CMD_BLINK:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                mcc.blink();
            }
            break;
        
        case CMD_RESET:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                mcc.reset();
            }
            break;
        
        case CMD_STATUS:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                out.format("Value = 0x%02x\n", mcc.getStatus());
            }
            break;

        case CMD_ITEMSET:
            if ((found = CmndProc.scanArgs(scan, "ssf", args)) >= 0) {
                if ((found & 0x01) == 0) {
                    showNames("item", itemNames);
                    break;
                }
                int item = encode(itemNames, (String)args[0]);
                if (item < 0) break;
                if ((found & 0x02) == 0) {
                    showNames("subitem", subiNames);
                    break;
                }
                int subItem = encode(subiNames, (String)args[1]);
                if (subItem < 0) break;
                if ((found & 0x04) == 0) {
                    if (subItem == MccUsb.CSI_FILTER_RATE)
                        showNames("filter rate", frateNames);
                    else if (subItem == MccUsb.CSI_CH_0_TC
                               || subItem ==MccUsb.CSI_CH_1_TC)
                        showNames("thermocouple type", tctypNames);
                    else if (subItem == MccUsb.CSI_CH_0_GAIN
                               || subItem ==MccUsb.CSI_CH_1_GAIN)
                        showNames("gain", gainNames);
                    else if (subItem == MccUsb.CSI_CH_0_VCONN
                               || subItem ==MccUsb.CSI_CH_1_VCONN)
                        showNames("connection type", vconnNames);
                    else
                        out.println("Missing value argument");
                    break;
                }
                mcc.setItem(item, subItem, (Float)args[2]);
            }
            break;

        case CMD_ITEMGET:
            if ((found = CmndProc.scanArgs(scan, "ss", args)) >= 0) {
                if ((found & 0x01) == 0)
                    showNames("item", itemNames);
                else if ((found & 0x02) == 0)
                    showNames("subitem", subiNames);
                else {
                    int item = encode(itemNames, (String)args[0]);
                    int subItem = encode(subiNames, (String)args[1]);
                    if (item < 0 || subItem < 0) break;
                    float value = mcc.getItem(item, subItem);
                    if ((subItem >= MccUsb.CSI_VREF
                           && subItem <= MccUsb.CSI_V_VALUE_2)
                          || (subItem >= MccUsb.CSI_CH_0_COEF_0
                                && subItem <= MccUsb.CSI_CH_1_COEF_3))
                        out.println("Value = " + value);
                    else {
                        int iValue = (int)value;
                        if (subItem == MccUsb.CSI_SENSOR_TYPE)
                            out.format("Value = %s (%s)\n", iValue,
                                       stypNames.decode(iValue));
                        else if (subItem == MccUsb.CSI_FILTER_RATE)
                            out.format("Value = %s (%s)\n", iValue,
                                       frateNames.decode(iValue));
                        else if (subItem == MccUsb.CSI_EXCITATION)
                            out.format("Value = %s (%s)\n", iValue,
                                       excitNames.decode(iValue));
                        else if (subItem == MccUsb.CSI_CH_0_TC
                                   || subItem == MccUsb.CSI_CH_1_TC)
                            out.format("Value = %s (%s)\n", iValue,
                                       tctypNames.decode(iValue));
                        else if (subItem == MccUsb.CSI_CH_0_GAIN
                                   || subItem == MccUsb.CSI_CH_1_GAIN)
                            out.format("Value = %s (%s)\n", iValue,
                                       gainNames.decode(iValue));
                        else if (subItem == MccUsb.CSI_CH_0_VCONN
                                   || subItem == MccUsb.CSI_CH_1_VCONN)
                            out.format("Value = %s (%s)\n", iValue,
                                       vconnNames.decode(iValue));
                        else
                            out.println("Value = " + iValue);
                    }
                }
            }
            break;

        case CMD_ALARMSET:
            if (CmndProc.scanArgs(scan, "IIIFF", args) >= 0) {
                mcc.configAlarm((Integer)args[0], (Integer)args[1],
                                (Integer)args[2], (Float)args[3],
                                (Float)args[4]);
            }
            break;

        case CMD_ALARMGET:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                int[] optns = {0, 0};
                float[] values = {0f, 0f};
                mcc.getAlarmConfig((Integer)args[0], optns, values);
                out.format("InOptns = 0x%02x, OutOptns = 0x%02x, "
                             + "Value1 = %s, Value2 = %s\n",
                           optns[0], optns[1], values[0], values[1]);
            }
            break;

        case CMD_SHOWDEV:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                int devId = mcc.getDevId();
                if (devId < 0)
                    out.println("No device open");
                else {
                    out.format("Device = %s\n", devNames.decode(devId));
                }
            }
            break;

        case CMD_SHOWCHAN:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                showChannel((found & 0x01) != 0 ? (Integer)args[0] : -1,
                            (found & 0x02) != 0 && (Integer)args[1] != 0);
            }
            break;

        case CMD_SHOWBURN:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                int mask = (found & 0x01) != 0 ? (Integer)args[0] : 0xff;
                out.format("Value = 0x%02x\n", mcc.getBurnout(mask));
            }
            break;

        case CMD_SHOWCAL:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                int[] steps = new int[2];
                mcc.calSteps(steps);
                out.format("Calibration steps: temp = %s, voltage = %s\n",
                           steps[0], steps[1]);
            }
            break;

        case CMD_CALCFG:
            if (CmndProc.scanArgs(scan, "III", args) >= 0) {
                mcc.calConfig((Integer)args[0], (Integer)args[1],
                              (Integer)args[2]);
            }
            break;

        case CMD_CALIBRATE:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                mcc.calibrate((Integer)args[0]);
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Show channel configuration information
    **
    ***************************************************************************
    */
    private final static String[] caption =
                {"Sensor type",  "Therm conn",  "Excitation",   "Filter rate",
                 "TC type",      "Gain",        "Voltage conn", "Refc voltage",
                 "I value 0",    "I value 1",   "I value 2",    "V value 0",
                 "V value 1",    "V value 2",   "Coeff 0",      "Coeff 1",
                 "Coeff 2",      "Coeff 3"};
    private final static boolean[] isValue =
                {false, false, false, false, false, false, false, true,
                 true,  true,  true,  true,  true,  true,  true,  true,
                 true,  true};
    private final static boolean[] isExtra =
                {false, true,  true,  false, false, false, false, false,
                 true,  true,  true,  true,  true,  true,  true,  true,
                 true,  true};
    private final static int[] loSubitem =
                {MccUsb.CSI_SENSOR_TYPE, MccUsb.CSI_CONN_TYPE,
                 MccUsb.CSI_EXCITATION,  MccUsb.CSI_FILTER_RATE,
                 MccUsb.CSI_CH_0_TC,     MccUsb.CSI_CH_0_GAIN,
                 MccUsb.CSI_CH_0_VCONN,  MccUsb.CSI_VREF,
                 MccUsb.CSI_I_VALUE_0,   MccUsb.CSI_I_VALUE_1,
                 MccUsb.CSI_I_VALUE_2,   MccUsb.CSI_V_VALUE_0,
                 MccUsb.CSI_V_VALUE_1,   MccUsb.CSI_V_VALUE_2,
                 MccUsb.CSI_CH_0_COEF_0, MccUsb.CSI_CH_0_COEF_1,
                 MccUsb.CSI_CH_0_COEF_2, MccUsb.CSI_CH_0_COEF_3};
    private final static int[] hiSubitem =
                {MccUsb.CSI_SENSOR_TYPE, MccUsb.CSI_CONN_TYPE,
                 MccUsb.CSI_EXCITATION,  MccUsb.CSI_FILTER_RATE,
                 MccUsb.CSI_CH_1_TC,     MccUsb.CSI_CH_1_GAIN,
                 MccUsb.CSI_CH_1_VCONN,  MccUsb.CSI_VREF,
                 MccUsb.CSI_I_VALUE_0,   MccUsb.CSI_I_VALUE_1,
                 MccUsb.CSI_I_VALUE_2,   MccUsb.CSI_V_VALUE_0,
                 MccUsb.CSI_V_VALUE_1,   MccUsb.CSI_V_VALUE_2,
                 MccUsb.CSI_CH_1_COEF_0, MccUsb.CSI_CH_1_COEF_1,
                 MccUsb.CSI_CH_1_COEF_2, MccUsb.CSI_CH_1_COEF_3};

    private void showChannel(int chan, boolean all) throws UsbException
    {
        int first = (chan < 0) ? 0 : chan & 7;
        int last = (chan < 0) ? 7 : chan & 7;

        out.print("              ");
        for (int j = first; j <= last; j++) out.format("   Ch %s ", j);
        out.println();
        out.print("              ");
        for (int j = first; j <= last; j++) out.print(" -------");
        out.println();

        for (int k = 0; k < caption.length; k++) {
            if (isExtra[k] && !all) continue;
            out.format("%-12s: ", caption[k]);
            int highSub = hiSubitem[k], lowSub = loSubitem[k];
            boolean isVal = isValue[k];
            for (int j = first; j <= last; j++) {
                float
                  value = mcc.getItem(j >> 1, (j & 1) != 0 ? highSub : lowSub);
                if (isVal)
                    out.format(" %7.7s", value);
                else {
                    String desc = "huh?";
                    if (lowSub == MccUsb.CSI_SENSOR_TYPE)
                        desc = stypsNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CONN_TYPE)
                        desc = tconsNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_EXCITATION)
                        desc = excitNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_FILTER_RATE)
                        desc = frateNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CH_0_TC)
                        desc = tctypNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CH_0_GAIN)
                        desc = gainNames.decode((int)value);
                    else if (lowSub == MccUsb.CSI_CH_0_VCONN)
                        desc = vconsNames.decode((int)value);
                    out.format(" %7s", desc);
                }
            }
            out.println();
        }
    }


   /**
    ***************************************************************************
    **
    **  Show all names in a lookup table along with their codes
    **
    ***************************************************************************
    */
    private void showNames(String desc, CmndProc.Lookup lookup)
    {
        out.println("Possible " + desc + " names:");
        for (int j = 0; j < lookup.count(); j++)
            out.format("  %s (%s)\n", lookup.name(j), lookup.code(j));
    }


   /**
    ***************************************************************************
    **
    **  Encode string, checking first whether it's already encoded
    **
    ***************************************************************************
    */
    private int encode(CmndProc.Lookup lookup, String token)
    {
        int value;

        try {
            value = Integer.valueOf(token);
        }
        catch (NumberFormatException e) {
            value = lookup.encode(token, true);
        }

        return value;
    }

}
