package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

import java.time.Duration;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * Represents a communications link over which commands are sent to the device
 * and replies received from it.
 * @author tether
 */
public interface Transport {
    
    /**
     * The baud rate to use for serial-port or FTDI connections.
     */
    public final int SERIAL_BAUD = 19200;
    
    /**
     * The data characteristics to use for serial-port or FTDI connections.
     */
    public final int SERIAL_CHARACTERISTICS = Ascii.makeDataCharacteristics(
        Ascii.DataBits.EIGHT,
        Ascii.StopBits.ONE,
        Ascii.Parity.NONE,
        Ascii.FlowCtrl.NONE
    );
    
    /**
     * All ASCII Modbus commands frames and response frames must end with this. Transport
     * objects will append this to outgoing command frames. They will stop collecting
     * the response frame when this sequence arrives and strip it from the returned string.
     */
    public final Ascii.Terminator MODBUS_FRAME_TERMINATOR = Ascii.Terminator.CRLF;
    
    /**
     * Sends a command to the device.
     * @param command The command string, w/o the final CRLF, to send.
     * @throws DriverException for any other problem.
     */
    void send(String command) throws DriverException;
    
    /**
     * Receives a response from the device.
     * @param timeout {@code timeout.getSeconds()} is how long to wait for the start of a response.
     * @return The response string stripped of the final CRLF.
     * @throws DriverTimeoutException if no response bytes arrive before the deadline.
     * @throws DriverException for any other problem.
     */
    String receive(Duration timeout) throws DriverException;
    
    /**
     * Closes the comm link to the device.
     */
    void close();
}
