package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.ascii.Ascii;

import org.lsst.ccs.drivers.commons.DriverException;

import java.time.Duration;

/**
 * Sends and receives bytes using a serial port on the local host.
 * @author tether
 */
public class SerialTransport implements Transport {
    
    private final Ascii port;
    
    /**
     * Opens the serial port.
     * @param portName The name of the port, for example /dev/ttyUSB0.
     * @throws DriverException if the open fails.
     */
    public SerialTransport(String portName) throws DriverException {
        Ascii newPort = null;
        try {
            newPort = new Ascii();
            newPort.open(
                Ascii.ConnType.SERIAL,
                portName,
                SERIAL_BAUD,
                SERIAL_CHARACTERISTICS
            );
            newPort.setTerminator(MODBUS_FRAME_TERMINATOR);
            newPort.flush(); // Discard any prior data in the receive buffer.
            port = newPort;
        }
        catch (Exception exc) {
            if (newPort != null) {
                newPort.closeSilent(); // Make sure the port is no longer busy.
            }
            throw exc;
        }

    }
    
    /** {@inheritDoc } */
    @Override
    public void send(String command) throws DriverException {
        port.write(command);
    }
    
    /** {@inheritDoc } */
    @Override
    public String receive(Duration timeout) throws DriverException {
        port.setTimeout(timeout.getSeconds());
        return port.read();
    }
    
    /** {@inheritDoc } */
    @Override
    public void close() {
        port.closeSilent();
    }
}
