package org.lsst.ccs.drivers.lighthouse;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * The device status flags that can be set for a Lighthouse data record.
 * @author tether
 */
public enum RecordFlag {
    /**
     * No problems.
     */
    OK("ok", 0),
    
    /**
     * Laser problem.
     */
    LASER("laser!", 1),
    
    /**
     * Couldn't get a normal air flow rate.
     */
    FLOW_RATE("flowRate!", 2),
    
    /**
     * Counter overflow.
     */
    OVERFLOW("overflow!", 4),
    
    /**
     * Instrument malfunction.
     */
    MALFUNCTION("malfunction!", 8),
    
    /** The reading is above the upper threshold. */
    HIGH("high!", 16),
    
    /**
     * The reading is below the lower threshold.
     */
    LOW("low!", 32),
    
    /**
     * Sampler problem.
     */
    SAMPLER("sampler!", 64);
    
    private final String text;
    private final int mask;
    
    RecordFlag(String text, int mask) {this.text = text; this.mask = mask;}
    
    /**
     * Gets the text form of the flag.
     * @return The text.
     */
    @Override
    public String toString() {return text;}
    
    /**
     * Gets the hardware mask value for the flag.
     * @return The mask.
     */
    public int getMask() {return mask;}
    
    /**
     * Calculates the set of flags whose bits are set in a given mask.
     * @param mask The channel status bits from the counter.
     * @return The read-only set of flags corresponding to the mask. Contains OK if nothing
     * else matches.
     */
    public static Set<RecordFlag> flagsFromMask(int mask) {
        final EnumSet<RecordFlag> flags =
            Stream.of(RecordFlag.values())
            .filter(flag -> (flag.getMask() & mask) != 0)
            .collect(Collectors.toCollection(()->EnumSet.noneOf(RecordFlag.class)));
        if (flags.isEmpty()) flags.add(OK);
        return Collections.unmodifiableSet(flags);
    }
}
