package org.lsst.ccs.drivers.lighthouse;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * The status flags that can be set for a Lighthouse counter device.
 * @author tether
 */
public enum DeviceFlag {
    /**
     * The counter is running, that is, is not stopped.
     */
    RUNNING("running!", 1),
    
    /**
     * The counter is running and taking a sample right now.
     */
    SAMPLING("sampling!", 2),
    
    /**
     * New data has been collected since the last time the data registers were read.
     */
    NEW_DATA("data!", 4),
    
    /**
     * There's a problem.
     */
    ERROR("error!", 8);
    
    private final String text;
    private final int mask;
    
    DeviceFlag(String text, int mask) {this.text = text; this.mask = mask;}
    
    /**
     * Gets the text form of the flag.
     * @return The text.
     */
    @Override
    public String toString() {return text;}
    
    /**
     * Gets the hardware mask value for the flag.
     * @return The mask.
     */
    public int getMask() {return mask;}
    
    /**
     * Calculates the set of flags whose bits are set in a given mask.
     * @param mask The channel status bits from the counter.
     * @return The read-only set of flags corresponding to the mask. C
     */
    public static Set<DeviceFlag> flagsFromMask(int mask) {
        final EnumSet<DeviceFlag> flags =
            Stream.of(DeviceFlag.values())
            .filter(flag -> (flag.getMask() & mask) != 0)
            .collect(Collectors.toCollection(()->EnumSet.noneOf(DeviceFlag.class)));
        return Collections.unmodifiableSet(flags);
    }
}
