package org.lsst.ccs.drivers.lighthouse;

import java.util.Collections;
import java.util.Formatter;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;
import java.time.Instant;
import java.time.Duration;

/**
 * Represents a single data record from a Lighthouse particle counter. There's some header
 * information for the record as a whole, as well as a series of channel data.
 * @author tether
 */
public final class DataRecord  implements java.io.Serializable {

    private static final long serialVersionUID = 1L;

    final Instant startTime;
    final Duration duration;
    final String location;
    final Set<RecordFlag> flags;
    final List<ChannelDatum> channels;

    /**
     * Creates a new instance from the field values.
     * @param startTime The start time. See {@link #getStartTime()}.
     * @param duration The duration. See {@link #getDuration()}.
     * @param location The location. See {@link #getLocation() }.
     * @param flags    The record flags. See {@link #getFlags() }.
     * @param channels The channel data items. See {@link #getChannels() }.
     */
    public DataRecord(
            Instant startTime,
            Duration duration,
            String location,
            Set<RecordFlag> flags,
            List<ChannelDatum> channels)
    {
        this.startTime = startTime;
        this.duration = duration;
        this.location = location;
        this.flags    = flags;
        this.channels = channels;
    }

    /**
     * Gets the time at which data collection was started for this record.
     * @return The start time.
     */
    public Instant getStartTime() {return startTime;}

    /**
     * Gets the length of time for which data was collected for this record.
     * @return The duration.
     */
    public Duration getDuration() {return duration;}

    /**
     * Gets the location at which the record's data was taken.
     * @return The location string.
     */
    public String getLocation() {return location;}

    /**
     * The device status flags for the record.
     * @return The set of flags that were on when the data were taken.
     */
    public Set<RecordFlag> getFlags() {return Collections.unmodifiableSet(flags);}

    /**
     * Gets the channel data associated with this record.
     * @return A stream of the channel data items.
     */
    public Stream<ChannelDatum> getChannels() {return channels.stream();}

    /**
     * Returns a string representation of the data record.
     *<p>
     * <pre><samp>
     * begin record
     *     start     yyyy-mm-ddThh24:mm:ssZ
     *     duration  xxxxx seconds
     *     location  B84_123
     *     status    laser! sampler!
     *     channel 10.0   counter      12230 particles
     *     channel  2.5   counter      14 particles
     *     channel temp   temperature  39.1 Celsius
     *     channel hum    relhumidity  1.0 percent
     * end record
     * </samp></pre>
     * @return The string.
     */
    @Override
    public String toString() {
        final Formatter fmt = new Formatter();
        fmt.format("begin record%n");
        fmt.format("    start %s%n", getStartTime().toString());
        fmt.format("    duration %d seconds%n", getDuration().getSeconds());
        fmt.format("    location %s%n", getLocation());
        fmt.format("    status");
        getFlags().stream().forEach(
                flag -> fmt.format(" %s", flag)
        );
        fmt.format("%n");
        getChannels().forEachOrdered(datum -> fmt.format("    %s%n", datum));
        fmt.format("end record");
        return fmt.toString();
    }
}
