package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.FileNotFoundException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;

import java.time.Duration;
import java.time.Instant;

import java.util.Optional;

import java.util.stream.Stream;


/**
 * Implements a set of shell commands for manipulating a single Lighthouse device.
 * @author tether
 */
public class LighthouseDriver {

    private static final Duration LH_COMM_TIMEOUT = Duration.ofSeconds(10);

    private static final Logger log = Logger.getLogger("org.lsst.ccs.drivers.lighthouse");

    private volatile Optional<LighthouseClient> lhClient = Optional.empty();

    /** Constructor. */
    public LighthouseDriver() {}

    /**
     * Opens the device on a serial port. Creates suitable implementations of
     * the interfaces {@code Transport} and {@code Hardware}.
     * @see LighthouseClient#serialOpen(java.lang.String, java.time.Duration, boolean)
     * @param portName The serial port name, for example {@code /dev/ttyUSB0}.
     * @param debug If true enables debug-level logging of device command traffic.
     * @throws DriverException for all errors.
     */
    @Command(description = "Opens the device on a serial port.")
    public void serialOpen(
            @Argument(description="The serial port name, for example /dev/ttyUSB0.")
            String portName,
            @Argument(description="When true, activates debug logging.", defaultValue="false")
            boolean debug
    ) throws DriverException
    {
        if (lhClient.isPresent()) lhClient.get().close();
        lhClient = Optional.of(LighthouseClient.serialOpen(portName, LH_COMM_TIMEOUT, debug));
        checkDevice();
    }


    /**
     * Opens the device on an "raw" FTDI USB-serial cable. Creates suitable implementations of
     * the interfaces {@code Transport} and {@code Hardware}.
     * @see LighthouseClient#ftdiOpen(java.lang.String, java.time.Duration, boolean)
     * @param ftdiID The serial no. of the FTDI device to use, preceded by a hostname
     * and a colon if remote. Examples: XXXXX or foo:XXXXX.
     * @param debug If true enables debug-level logging of device command traffic.
     * @throws DriverException for all errors.
     */
    @Command(description = "Opens the device on an FTDI USB-serial cable.")
    public void ftdiOpen(
            @Argument(description=
                    "The serial no. of the cable, preceded by a hostname and a colon if it's a remote cable.")
            String ftdiID,
            @Argument(description="When true, activates debug logging.", defaultValue="false")
            boolean debug
    ) throws DriverException
    {
        if (lhClient.isPresent()) lhClient.get().close();
        lhClient = Optional.of(LighthouseClient.ftdiOpen(ftdiID, LH_COMM_TIMEOUT, debug));
        checkDevice();
    }


    /**
     * Opens the device on a network socket. Creates suitable implementations of
     * the interfaces {@code Transport} and {@code Hardware}.
     * @see LighthouseClient#netOpen(java.lang.String, int, java.time.Duration, boolean)
     * @param hostname The remote host's name or IP address.
     * @param portNo The number of the port to connect to.
     * @param debug If true enables debug-level logging of device command traffic.
     * @throws DriverException for all errors.
     */
    @Command(description = "Opens the device on a network socket.")
    public void netOpen(
            @Argument(description="The remote host's name or IP address.")
            String hostname,
            @Argument(description="The number of the port to connect to.")
            int portNo,
            @Argument(description="When true, activates debug logging.", defaultValue="false")
            boolean debug
    ) throws DriverException
    {
        if (lhClient.isPresent()) lhClient.get().close();
        lhClient = Optional.of(LighthouseClient.netOpen(hostname, portNo, LH_COMM_TIMEOUT, debug));
        checkDevice();
    }

    private void checkDevice() throws DriverException {
        final String model = lhClient.get().getModel();
        log.info("Device model: " + model);
        if (!model.matches(".*(2016|3016|5016).*")) {
            throw new DriverException("Unknown device model");
        }
        final int mapver = lhClient.get().getRegisterMapVersion();
        log.info(String.format("Modbus register map v%d.%02d", mapver/100, mapver%100));
    }

    /**
     * Closes the device.
     *  @throws DriverException for all errors.
     */
    @Command(name = "close", description = "Closes link used to communicate with the counter.")
    public void close() throws DriverException {
       if (lhClient.isPresent()) lhClient.get().close();
       lhClient = Optional.empty();
    }

    /**
     * Stops the device so that it no longer updates its internal data set.
     * Has no effect if recording is already stopped.
     * @throws DriverException for all errors.
     */
    @Command(
        name = "stopRecording",
        description = "Stops the counter so that it no longer updates its internal data set.")
    public void stopRecording() throws DriverException {
        checkForClient();
        lhClient.get().stopRecording();
    }

    /**
     * Starts the device, appending all future records to its internal data set.
     * Has no effect if the device is already active.
     * @throws DriverException
     */
    @Command(
        name = "startRecording",
        description = "Starts the counter, appending all future records to its internal data set.")
    public void startRecording() throws DriverException {
        checkForClient();
        lhClient.get().startRecording();
    }

    /**
     * Clears all records from the device's internal data set. Recording must first be stopped.
     * @throws DriverException
     */
    @Command(
        name = "clearRecords",
        description = "Clears all readings from the counter's internal data set.")
    public void clearRecords() throws DriverException {
        checkForClient();
        lhClient.get().clearAllRecords();
    }

    /**
     * Prints in ASCII, earliest first, the device's buffered data records.
     * The internal data set remains unchanged. Recording must first be stopped.
     * @param nlimit The maximum number of records to get (0 = no limit).
     * @param nskip The number of records to skip at the beginning.
     * -n means all but the last n.
     * @param filename Where to print (System.out if empty).
     * @throws DriverException for all checked exceptions.
     * @see DataRecord#toString()
     */
    @Command(
        name = "getRecords",
        description = "Prints in ASCII, earliest first, the counter's buffered data records.")
    public void getRecords(
        @Argument(description="The maximum number of records to get (0 = no limit).",
            defaultValue="0")
        int nlimit,
        @Argument(description="The number of records to skip at the start. -n means all but the last n.",
            defaultValue="0")
        int nskip,
        @Argument(description="The name of the file in which to write (System.out if omitted).",
            defaultValue="")
        String filename
    ) throws DriverException
    {
        checkForClient();
        try (final PrintWriter p = makeWriter(filename)) {
            printRecords(p, lhClient.get().getRecordsByIndex(nskip, nlimit).stream());
        }
    }

    /**
     * Prints in ASCII, latest first, those data records more recent than a given date and time.
     * @param instantString All the records printed will be newer than this. Format yyyy-mm-ddThh:mm:ssZ.
     * @param filename Where to print (System.out if empty).
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the first argument is rejected by Instant.parse().
     * @see DataRecord#toString()
     */
    @Command(description="Prints, latest first, data records more recent than a given date and time.")
    public void getRecentRecords(
        @Argument(name="after", description="The records written will all be newer than this.")
        String instantString,
        @Argument(description="The name of the file (System.out if omitted).", defaultValue="")
        String filename
    )
    throws DriverException
    {
        checkForClient();
        Instant after;
        try {
            after = Instant.parse(instantString);
        }
        catch (java.time.format.DateTimeParseException exc) {
            throw new IllegalArgumentException("Invalid Java Instant: "+instantString, exc);
        }

        try (final PrintWriter p = makeWriter(filename)) {
            printRecords(p, lhClient.get().getRecentRecords(after).stream());
        }
    }

    /**
     * Sets the device clock from the computer's.
     * @throws DriverException if I/O error or the device clock can't be set to within 30 seconds
     * of the computer's clock.
     */
    @Command(description="Set the device clock from the computer's.")
    public void setClock() throws DriverException {
        checkForClient();
        lhClient.get().setClock(Duration.ofSeconds(30));
    }

    /**
     * Displays the device clock value.
     * @return The clock value as an Instant.
     * @throws DriverException for I/O errors.
     */
    @Command(description="Displays the device clock value.")
    public Instant getClock()  throws DriverException {
        checkForClient();
        return lhClient.get().getClock();
    }

    /**
     * Returns the setting of the alarm enable for a particle channel.
     * @param chanName The name of the channel.
     * @return The alarm enable flag, true or false.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if {@code chanName} is for an analog channel
     * or is invalid.
     */
    @Command(description="Returns the setting of the alarm enable for a particle channel.")
    public boolean alarmIsEnabled(
        @Argument(description="The name of the channel.")
        String chanName
    ) throws DriverException
    {
        checkForClient();
        final DataChannel chan = DataChannel.parse(chanName);
        return lhClient.get().alarmIsEnabled(chan);
    }

    /**
     * Sets the alarm enable state for a particle channel.
     * @param chanName The name of the channel.
     * @param on true for on, false for off.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if {@code chanName} is for an analog channel
     * or is invalid.
     */
    @Command(description="Sets the alarm enable state for a particle channel.")
    public void setAlarmEnable(
        @Argument(description="The name of the channel.")
        String chanName,
        @Argument(description="true for on, false for off.")
        boolean on
    ) throws DriverException
    {
        checkForClient();
        final DataChannel chan = DataChannel.parse(chanName);
        lhClient.get().setAlarmEnable(chan, on);
    }

    /**
     * Sets the alarm threshold for a particle channel.
     * @param chanName The name of the channel.
     * @param threshold The threshold value in raw counts per sample.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if {@code chanName} is for an analog channel
     * or is invalid.
     */
    @Command(description="Sets the alarm threshold for a particle channel.")
    public void setAlarmThreshold(
        @Argument(description="The name of the channel.")
        String chanName,
        @Argument(description="The threshold value in raw counts per sample.")
        long threshold
    ) throws DriverException
    {
        checkForClient();
        final DataChannel chan = DataChannel.parse(chanName);
        lhClient.get().setAlarmThreshold(chan, threshold);
    }

    /**
     * Gets the alarm threshold for a particle channel.
     * @param chanName The name of the channel.
     * @return The threshold value in raw counts per sample.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if {@code chanName} is for an analog channel
     * or is invalid.
     */
    @Command(description="Sets the alarm threshold for a particle channel.")
    public long getAlarmThreshold(
        @Argument(description="The name of the channel.")
        String chanName
    ) throws DriverException
    {
        checkForClient();
        final DataChannel chan = DataChannel.parse(chanName);
        return lhClient.get().getAlarmThreshold(chan);
    }


    /**
     * Generates a safely closable ASCII PrintWriter wrapping either System.out or a file stream.
     * Closing a normal PrintWriter wrapping System.out would close System.out which
     * would be bad news.
     * @param filename An empty string for System.out or the file name.
     * @return The PrintWriter.
     * @throws DriverException on I/O or encoding errors.
     */
    private PrintWriter makeWriter(String filename) throws DriverException {
        PrintWriter p;
        try {
            p = filename.equals("")
                ? new PrintWriter(new OutputStreamWriter(System.out, "US-ASCII"), true) {
                    @Override
                    public void close() {this.flush();}
                }
                : new PrintWriter(filename, "US-ASCII");
        }
        catch (UnsupportedEncodingException exc) {
            throw new DriverException("Can't write in US-ASCII", exc);
        }
        catch(FileNotFoundException exc) {
            throw new DriverException("Can't open " + filename, exc);
        }
        return p;
    }

    private void printRecords(PrintWriter out, Stream<DataRecord> recs) throws DriverException {
        recs.forEachOrdered(rec -> out.println(rec));
    }

    private void checkForClient() throws DriverException {
        lhClient.orElseThrow(() -> {return new DriverException("No open Lighthouse device");});
    }

}
