package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.commons.DriverException;

import java.time.Duration;
import java.time.Instant;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import java.util.Spliterator;
import java.util.Spliterators;


/**
 * Exports high-level operations on a Lighthouse particle counter. Only operations and data
 * are visible, not registers or communication protocols. Implements a command set common
 * to the 2016, 3016 and 5016 family of hand-held and wall-mounted units.
 * <p>
 * Only a single thread should manipulate an instance of this class. Only a single stream
 * of data records should be in use at any time since advancing to the next record involves updating
 * the record index register on the instrument before reading the record data registers. This lazy
 * retrieval allows you to skip needless I/O if you decide you don't want all the stored records.
 * @author tether
 */
public class LighthouseClient {

    // Lambda expressions can't throw checked exceptions, so we'll have to
    // wrap them in this unchecked exception.
    private static class WrapperException extends RuntimeException {
        WrapperException(DriverException exc) {super(exc);}
    }

    /**
     * Opens the device on a generic serial port. Creates suitable implementations of
     * the interfaces {@code Transport} and {@code Hardware}.
     * @param portName The serial port name, for example {@code /dev/ttyUSB0}.
     * @param timeout How long to wait for a response after sending a command.
     * @param debug If true enables debug-level logging of device command traffic.
     * @return The new Lighthouse client object.
     * @throws DriverException for all errors.
     */
    public static LighthouseClient serialOpen(String portName, Duration timeout, boolean debug)
            throws DriverException
    {
        return new LighthouseClient(new ASCIIModbus(new SerialTransport(portName), timeout, debug));
    }

    /**
     * Opens the device on a raw FTDI serial port. Creates suitable implementations of
     * the interfaces {@code Transport} and {@code Hardware}. It's possible to connect to a device
     * on another host if that host is running an FTDI server.
     * @param ident The FTDI chip's serial number, possibly preceded by a host name/IP and a colon.
     * @param timeout How long to wait for a response after sending a command.
     * @param debug If true enables debug-level logging of device command traffic.
     * @return The new Lighthouse client object.
     * @throws DriverException for all errors.
     */
    public static LighthouseClient ftdiOpen(String ident, Duration timeout, boolean debug)
            throws DriverException
    {
        return new LighthouseClient(new ASCIIModbus(new FTDITransport(ident), timeout, debug));
    }

    /**
     * Opens a device owned by a TCP server. Creates suitable implementations of
     * the interfaces {@code Transport} and {@code Hardware}.
     * @param hostname The server's hostname or IP address.
     * @param portNo The server's port number.
     * @param timeout How long to wait for a response after sending a command.
     * @param debug If true enables debug-level logging of device command traffic.
     * @return The new Lighthouse client object.
     * @throws DriverException for all errors.
     */
    public static LighthouseClient netOpen(String hostname, int portNo, Duration timeout, boolean debug)
            throws DriverException
    {
        return new LighthouseClient(new ASCIIModbus(new NetTransport(hostname, portNo), timeout, debug));
    }

    private final Hardware hw;
    private final String model;
    private final int registerMapVersion;

    /**
     * Stores a reference to a {@code Hardware} object used to communicate with the instrument and
     * reads the device model and the register map version.
     * @param hw The communications object.
     * @throws DriverException if communication problems arise.
     */
    public LighthouseClient(Hardware hw) throws DriverException {
        this.hw = hw;
        this.model = hw.readDeviceModel();
        this.registerMapVersion = hw.readDeviceMapVersion();
        // Check that the channels are set up as expected.
        for (DataChannel chan : DataChannel.values()) {
            hw.readChannelType(chan);
            hw.readChannelUnits(chan);
            if (!hw.channelIsEnabled(chan)) {
                throw new DriverException("Channel "+chan.getName()+" is not enabled");
            }
        }
    }

    /**
     * Closes the underlying connection to the device.
     * @throws DriverException for I/O errors.
     */
    public void close() throws DriverException {
        hw.close();
    }

    /**
     * Gets the model of Lighthouse counter hardware (read by constructor).
     * @return The model designation string.
     */
    public String getModel() {return model;}

    /**
     * Gets the version of the Modbus register map used by the hardware (read by constructor).
     * @return An integer with decimal digits xxyy, where xx is the major version
     * and yy the minor version.
     */
    public int getRegisterMapVersion() {return registerMapVersion;}

    /**
     * Stops data collection. Aborts any new record being created. Has no effect if
     * data collection has already been stopped.
     * @throws DriverException if communication problems arise.
     */
    public void stopRecording() throws DriverException  {
        hw.writeDeviceCommand(DeviceCommand.INSTRUMENT_STOP);
    }

    /**
     * Starts automatic data collection. Has no effect if data
     * collection is already started.
     * @throws DriverException if communication problems arise.
     */
    public void startRecording() throws DriverException  {
        hw.writeDeviceCommand(DeviceCommand.INSTRUMENT_START);
    }

    /**
     * Gets the data records specified by {@code start} and {@code limit}.
     * @param start Specifies the index of the earliest record to fetch.
     * A non-negative value means index {@code 0 + start} while a negative value
     * means index {@code N + start}, where {@code N} is the total number of records
     * available.
     * @param limit The number of records to fetch, with zero meaning all after the starting
     * point.
     * @return The unmodifiable record list in ASCENDING order by record index (and timestamp).
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public List<DataRecord> getRecordsByIndex(int start, int limit) throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't read records when the device is running.");
        }
        final int nrec = hw.readRecordCount();
        final int first = start >= 0 ? start : Math.max(0, nrec + start);
        final int last = limit <= 0 ? nrec - 1 : Math.min(nrec - 1, first + limit - 1);
        final List<DataRecord> records = new ArrayList<>();
        for (int i = first; i <= last; ++i) {
            records.add(fetchRecord(i));
        }
        return Collections.unmodifiableList(records);
    }

    /**
     * Gets the data records more recent than a given time.
     * @param threshold All records returned will be more recent than this.
     * @return The unmodifiable record list, most recent record first.
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public List<DataRecord> getRecentRecords(Instant threshold) throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't read records when the device is running.");
        }
        final int nrec = hw.readRecordCount();
        final List<DataRecord> records = new ArrayList<>();
        for (int i = nrec - 1; i >= 0; --i) {
            final DataRecord nextrec = fetchRecord(i);
            if (!nextrec.getStartTime().isAfter(threshold)) break;
            records.add(nextrec);
        }
        return Collections.unmodifiableList(records);
    }

    /**
     * Delete all records currently stored in the instrument.
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public void clearAllRecords() throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't clear records when the device is running.");
        }
        hw.writeDeviceCommand(DeviceCommand.CLEAR_DATA_BUFFER);
    }

    /**
     * Sets the particle count alarm threshold for a particle channel but doesn't
     * change the alarm enable status.
     * @param chan The channel to be affected.
     * @param threshold The limiting particle count.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog or if the threshold
     * is not in the interval {@code [0, 2^32-1]}.
     */
    public void setAlarmThreshold(DataChannel chan, long threshold) throws DriverException {
        hw.setAlarmThreshold(chan, threshold);
    }

    /**
     * Gets the current alarm threshold for a particle channel.
     * @param chan The channel to query.
     * @return The current alarm threshold.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog.
     */
    public long getAlarmThreshold(DataChannel chan) throws DriverException {
        return hw.getAlarmThreshold(chan);
    }

    /**
     * Enables or disables the alarm for a particle channel.
     * @param chan The channel to affect.
     * @param on true to enable alarms, false to disable.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog.
     */
    public void setAlarmEnable(DataChannel chan, boolean on) throws DriverException {
        hw.setAlarmEnable(chan, on);
    }

    /**
     * Gets the alarm enable status for a particle channel.
     * @param chan The channel to query.
     * @return true if enabled, false if disabled.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog.
     */
    public boolean alarmIsEnabled(DataChannel chan) throws DriverException {
        return hw.alarmIsEnabled(chan);
    }

    /**
     * Sets the device's time-of-day clock to the current system time.
     * @param tolerance When read back the device clock must be no further than
     * this from the current system time just after the read.
     * @throws DriverException for I/O errors or if the clock value read back is bad.
     */
    public void setClock(Duration tolerance) throws DriverException {
        hw.setClock(Instant.now().getEpochSecond());
        final Instant deviceNow = getClock();
        final Instant sysLow = Instant.now().minus(tolerance);
        final Instant sysHigh = sysLow.plus(tolerance.multipliedBy(2));
        if (deviceNow.isBefore(sysLow) | deviceNow.isAfter(sysHigh)) {
            throw new DriverException("Can't get device clock within "+tolerance+" of system time.");
        }
    }

    /**
     * Gets the device's time-of-day clock.
     * @return The clock reading.
     * @throws DriverException for I/O errors.
     */
    public Instant getClock() throws DriverException {
        return Instant.ofEpochSecond(hw.getClock());
    }
    
    /**
     * Sets the register that contains the index number of the instrument's
       location. 
     * @param index The index (1 = lowest index).
     * @throws DriverException if the operation failed.
     * @throws IllegalArgumentException if the index is invalid.
     */
    public void setLocationIndex(int index) throws DriverException {
        hw.setDeviceLocationIndex(index);
    }
    
    /**
     * Gets the label for the current location set for the instrument.
     * @return The location label string.
     * @throws DriverException if the operation fails.
     */
    public String getLocationName() throws DriverException {
        return hw.getDeviceLocationName();
    }

    private DataRecord fetchRecord(int irec) throws DriverException {
        // Tell the instrument which record we want, then read it.
        hw.writeRecordIndex(irec);
        return hw.readRecord();
     }

}
