package org.lsst.ccs.drivers.lighthouse;

import java.util.Collections;
import java.util.Map;
import java.util.function.Function;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * The measurement units for Lighthouse data channels. Not an exhaustive list by any means.
 * @author tether
 */
public enum ChannelUnits {

    /**
     * A count of particles of a certain size.
     */
    COUNTS("counts", "#"),

    /**
     * A percentage, for example relative humidity.
     */
    PERCENTAGE("percent", "%"),

    /**
     * A temperature on the Celsius scale.
     */
    CELSIUS("Celsius", "C"),

    /**
     * A temperature on the Fahrenheit scale.
     */
    FAHRENHEIT("Fahrenheit", "F");

    private final String fileText, deviceText;

    ChannelUnits(String fileText, String deviceText) {
        this.fileText = fileText;
        this.deviceText = deviceText;
    }

    /**
     * Gets the string used to represent these units in files or other external storage.
     * @return The string.
     */
    @Override
    public String toString() {return fileText;}

    /**
     * Gets the string used by the device to represent these units.
     * @return The string.
     */
    public String getDeviceText() {return deviceText;}


    private static final Map<String, ChannelUnits> labelMap =
            Collections.unmodifiableMap(
                Stream.of(ChannelUnits.values())
                .collect(
                        Collectors.toMap(
                                ChannelUnits::getDeviceText,
                                Function.identity()
                        )
                )
            );


    /**
     * Find the measurement type corresponding to a device units-label.
     * @param label The data units label used by the counter.
     * @return The corresponding ChannelUnits enumerator.
     */
    public static ChannelUnits fromUnitsLabel(String label) {return labelMap.get(label);}
}
