/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.drivers.lighthouse;

import java.io.FileNotFoundException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.Optional;
import java.util.stream.Stream;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.lighthouse.DataChannel;
import org.lsst.ccs.drivers.lighthouse.DataRecord;
import org.lsst.ccs.drivers.lighthouse.LighthouseClient;
import org.lsst.ccs.utilities.logging.Logger;

public class LighthouseDriver {
    private static final Duration LH_COMM_TIMEOUT = Duration.ofSeconds(10L);
    private static final Logger log = Logger.getLogger((String)"org.lsst.ccs.drivers.lighthouse");
    private volatile Optional<LighthouseClient> lhClient = Optional.empty();

    @Command(description="Opens the device on a serial port.")
    public void serialOpen(@Argument(description="The serial port name, for example /dev/ttyUSB0.") String portName, @Argument(description="When true, activates debug logging.", defaultValue="false") boolean debug) throws DriverException {
        if (this.lhClient.isPresent()) {
            this.lhClient.get().close();
        }
        this.lhClient = Optional.of(LighthouseClient.serialOpen(portName, LH_COMM_TIMEOUT, debug));
        this.checkDevice();
    }

    @Command(description="Opens the device on an FTDI USB-serial cable.")
    public void ftdiOpen(@Argument(description="The serial no. of the cable, preceded by a hostname and a colon if it's a remote cable.") String ftdiID, @Argument(description="When true, activates debug logging.", defaultValue="false") boolean debug) throws DriverException {
        if (this.lhClient.isPresent()) {
            this.lhClient.get().close();
        }
        this.lhClient = Optional.of(LighthouseClient.ftdiOpen(ftdiID, LH_COMM_TIMEOUT, debug));
        this.checkDevice();
    }

    @Command(description="Opens the device on a network socket.")
    public void netOpen(@Argument(description="The remote host's name or IP address.") String hostname, @Argument(description="The number of the port to connect to.") int portNo, @Argument(description="When true, activates debug logging.", defaultValue="false") boolean debug) throws DriverException {
        if (this.lhClient.isPresent()) {
            this.lhClient.get().close();
        }
        this.lhClient = Optional.of(LighthouseClient.netOpen(hostname, portNo, LH_COMM_TIMEOUT, debug));
        this.checkDevice();
    }

    private void checkDevice() throws DriverException {
        String model = this.lhClient.get().getModel();
        log.info((Object)("Device model: " + model));
        if (!model.matches(".*(2016|3016|5016).*")) {
            throw new DriverException("Unknown device model");
        }
        int mapver = this.lhClient.get().getRegisterMapVersion();
        log.info((Object)String.format("Modbus register map v%d.%02d", mapver / 100, mapver % 100));
    }

    @Command(name="close", description="Closes link used to communicate with the counter.")
    public void close() throws DriverException {
        if (this.lhClient.isPresent()) {
            this.lhClient.get().close();
        }
        this.lhClient = Optional.empty();
    }

    @Command(name="stopRecording", description="Stops the counter so that it no longer updates its internal data set.")
    public void stopRecording() throws DriverException {
        this.checkForClient();
        this.lhClient.get().stopRecording();
    }

    @Command(name="startRecording", description="Starts the counter, appending all future records to its internal data set.")
    public void startRecording() throws DriverException {
        this.checkForClient();
        this.lhClient.get().startRecording();
    }

    @Command(name="clearRecords", description="Clears all readings from the counter's internal data set.")
    public void clearRecords() throws DriverException {
        this.checkForClient();
        this.lhClient.get().clearAllRecords();
    }

    @Command(name="getRecords", description="Prints in ASCII, earliest first, the counter's buffered data records.")
    public void getRecords(@Argument(description="The maximum number of records to get (0 = no limit).", defaultValue="0") int nlimit, @Argument(description="The number of records to skip at the start. -n means all but the last n.", defaultValue="0") int nskip, @Argument(description="The name of the file in which to write (System.out if omitted).", defaultValue="") String filename) throws DriverException {
        this.checkForClient();
        try (PrintWriter p = this.makeWriter(filename);){
            this.printRecords(p, this.lhClient.get().getRecordsByIndex(nskip, nlimit).stream());
        }
    }

    @Command(description="Prints, latest first, data records more recent than a given date and time.")
    public void getRecentRecords(@Argument(name="after", description="The records written will all be newer than this.") String instantString, @Argument(description="The name of the file (System.out if omitted).", defaultValue="") String filename) throws DriverException {
        Instant after;
        this.checkForClient();
        try {
            after = Instant.parse(instantString);
        }
        catch (DateTimeParseException exc) {
            throw new IllegalArgumentException("Invalid Java Instant: " + instantString, exc);
        }
        try (PrintWriter p = this.makeWriter(filename);){
            this.printRecords(p, this.lhClient.get().getRecentRecords(after).stream());
        }
    }

    @Command(description="Set the device clock from the computer's.")
    public void setClock() throws DriverException {
        this.checkForClient();
        this.lhClient.get().setClock(Duration.ofSeconds(30L));
    }

    @Command(description="Displays the device clock value.")
    public Instant getClock() throws DriverException {
        this.checkForClient();
        return this.lhClient.get().getClock();
    }

    @Command(description="Returns the setting of the alarm enable for a particle channel.")
    public boolean alarmIsEnabled(@Argument(description="The name of the channel.") String chanName) throws DriverException {
        this.checkForClient();
        DataChannel chan = DataChannel.parse(chanName);
        return this.lhClient.get().alarmIsEnabled(chan);
    }

    @Command(description="Sets the alarm enable state for a particle channel.")
    public void setAlarmEnable(@Argument(description="The name of the channel.") String chanName, @Argument(description="true for on, false for off.") boolean on) throws DriverException {
        this.checkForClient();
        DataChannel chan = DataChannel.parse(chanName);
        this.lhClient.get().setAlarmEnable(chan, on);
    }

    @Command(description="Sets the alarm threshold for a particle channel.")
    public void setAlarmThreshold(@Argument(description="The name of the channel.") String chanName, @Argument(description="The threshold value in raw counts per sample.") long threshold) throws DriverException {
        this.checkForClient();
        DataChannel chan = DataChannel.parse(chanName);
        this.lhClient.get().setAlarmThreshold(chan, threshold);
    }

    @Command(description="Sets the alarm threshold for a particle channel.")
    public long getAlarmThreshold(@Argument(description="The name of the channel.") String chanName) throws DriverException {
        this.checkForClient();
        DataChannel chan = DataChannel.parse(chanName);
        return this.lhClient.get().getAlarmThreshold(chan);
    }

    private PrintWriter makeWriter(String filename) throws DriverException {
        PrintWriter p;
        try {
            p = filename.equals("") ? new PrintWriter(new OutputStreamWriter((OutputStream)System.out, "US-ASCII"), true){

                @Override
                public void close() {
                    this.flush();
                }
            } : new PrintWriter(filename, "US-ASCII");
        }
        catch (UnsupportedEncodingException exc) {
            throw new DriverException("Can't write in US-ASCII", exc);
        }
        catch (FileNotFoundException exc) {
            throw new DriverException("Can't open " + filename, exc);
        }
        return p;
    }

    private void printRecords(PrintWriter out, Stream<DataRecord> recs) throws DriverException {
        recs.forEachOrdered(rec -> out.println(rec));
    }

    private void checkForClient() throws DriverException {
        this.lhClient.orElseThrow(() -> new DriverException("No open Lighthouse device"));
    }
}

