package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.ascii.Ascii;

import org.lsst.ccs.drivers.commons.DriverException;

import java.time.Duration;

/**
 * Sends and receives bytes using an FTDI USB-to-serial cable. You have to unload (or 
 * prevent from being loaded) the kernel modules ftdi_sio and usbserial in order to see the raw FTDI chip.
 * @author tether
 */
public class FTDITransport implements Transport {
    
    private final Ascii port;
    
    /**
     * Opens the FTDI chip.
     * @param ident The serial number string of the interface, optionally
    **                  prepended with the node name of a remote FTDI server
    **                  and separated from it by a ":"
     * @throws DriverException if the open fails.
     */
    public FTDITransport(String ident) throws DriverException {
        port = new Ascii();
        port.open(
            Ascii.ConnType.FTDI,
            ident,
            SERIAL_BAUD,
            SERIAL_CHARACTERISTICS
        );
        port.setTerminator(MODBUS_FRAME_TERMINATOR);
    }
    
    /** {@inheritDoc } */
    @Override
    public void send(String command) throws DriverException {
        port.write(command);
    }
    
    /** {@inheritDoc } */
    @Override
    public String receive(Duration timeout) throws DriverException {
        port.setTimeout(timeout.getSeconds());
        return port.read();
    }
    
    /** {@inheritDoc } */
    @Override
    public void close() {port.closeSilent();}
}
