package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.commons.DriverException;

import java.time.Duration;
import java.time.Instant;

import java.util.function.Consumer;
import java.util.function.Predicate;

import java.util.Spliterator;
import java.util.Spliterators;

import java.util.stream.IntStream;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Exports high-level requests made of a Lighthouse particle counter. Only operations and data
 * are visible, not registers or communication protocols. Implements a command set common
 * to the 2016, 3016 and 5016 family of hand-held and wall-mounted units.
 * <p>
 * Only a single thread should manipulate an instance of this class. Only a single stream
 * of data records should be in use at any time since advancing to the next record involves updating
 * the record index register on the instrument before reading the record data registers. This lazy
 * retrieval allows you to skip needless I/O if you decide you don't want all the stored records.
 * @author tether
 */
public class Client {
    private final Hardware hw;
    private final String model;
    private final int registerMapVersion;

    /**
     * Stores a reference to a Hardware object used to communicate with the instrument and
     * reads the device model and the register map version.
     * @param hw The communications object.
     * @throws DriverException if communication problems arise.
     */
    public Client(Hardware hw) throws DriverException {
        this.hw = hw;
        this.model = hw.readDeviceModel();
        this.registerMapVersion = hw.readDeviceMapVersion();
        // Check that the channels are set up as expected.
        for (DataChannel chan : DataChannel.values()) {
            hw.readChannelType(chan);
            hw.readChannelUnits(chan);
            if (!hw.channelIsEnabled(chan)) {
                throw new DriverException("Channel "+chan.getName()+" is not enabled");
            }
        }
    }

    /**
     * Gets the model of Lighthouse counter hardware (read by constructor).
     * @return The model designation string.
     */
    public String getModel() {return model;}

    /**
     * Gets the version of the Modbus register map used by the hardware (read by constructor).
     * @return An integer with decimal digits xxyy, where xx is the major version
     * and yy the minor version.
     */
    public int getRegisterMapVersion() {return registerMapVersion;}

    /**
     * Stops data collection. Aborts any new record being created. Has no effect if
     * data collection has already been stopped.
     * @throws DriverException if communication problems arise.
     */
    public void stopRecording() throws DriverException  {
        hw.writeDeviceCommand(DeviceCommand.INSTRUMENT_STOP);
    }

    /**
     * Starts automatic data collection. Has no effect if data
     * collection is already started.
     * @throws DriverException if communication problems arise.
     */
    public void startRecording() throws DriverException  {
        hw.writeDeviceCommand(DeviceCommand.INSTRUMENT_START);
    }

    /**
     * Gets the data records specified by {@code start} and {@code limit}.
     * @param start Specifies the index of the earliest record to fetch.
     * A non-negative value means index {@ 0 + start} while a negative value
     * means index {@code N + start}, where {@code N} is the total number of records
     * available.
     * @param limit The number of records to fetch, with zero meaning all after the starting
     * point.
     * @return The record stream in ASCENDING order by record index (and timestamp).
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public Stream<DataRecord> getRecordsByIndex(int start, int limit) throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't read records when the device is running.");
        }
        final int nrec = hw.readRecordCount();
        final int nskip = start >= 0 ? start : Math.max(0, nrec + start);
        final int nlimit = limit > 0 ? limit : nrec;
        try {
            return IntStream.range(0, nrec)
                .skip(nskip)
                .limit(nlimit)
                .mapToObj(this::fetchRecord);
        }
        catch (WrapperException exc) {throw (DriverException)exc.getCause();}
    }

    /**
     * Gets the data records more recent than a given time.
     * @param threshold All records returned will be more recent than this.
     * @return The record stream, most recent record first.
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public Stream<DataRecord> getRecentRecords(Instant threshold) throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't read records when the device is running.");
        }
        final int nrec = hw.readRecordCount();
        try {
            return 
                takeWhile(
                    IntStream
                        .iterate(nrec - 1, (i) -> i - 1) // Work backward from most recent.
                        .limit(nrec)                     // Only so many are stored.
                        .mapToObj(this::fetchRecord),
                    rec -> rec.getStartTime().isAfter(threshold)
                );
        }
        catch (WrapperException exc) {throw (DriverException)exc.getCause();}
    }

    /**
     * Delete all records currently stored in the instrument.
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public void clearAllRecords() throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't clear records when the device is running.");
        }
        hw.writeDeviceCommand(DeviceCommand.CLEAR_DATA_BUFFER);
    }
       
    /**
     * Sets the particle count alarm threshold for a particle channel but doesn't
     * change the alarm enable status.
     * @param chan The channel to be affected.
     * @param threshold The limiting particle count.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog or if the threshold
     * is not in the interval {@code [0, 2^32-1]}.
     */
    public void setAlarmThreshold(DataChannel chan, long threshold) throws DriverException {
        hw.setAlarmThreshold(chan, threshold);
    }
    
    /**
     * Gets the current alarm threshold for a particle channel.
     * @param chan The channel to query.
     * @return The current alarm threshold.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog.
     */
    public long getAlarmThreshold(DataChannel chan) throws DriverException {
        return hw.getAlarmThreshold(chan);
    }
    
    /**
     * Enables or disables the alarm for a particle channel.
     * @param chan The channel to affect.
     * @param on true to enable alarms, false to disable.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog.
     */
    public void setAlarmEnable(DataChannel chan, boolean on) throws DriverException {
        hw.setAlarmEnable(chan, on);
    }

    /**
     * Gets the alarm enable status for a particle channel.
     * @param chan The channel to query.
     * @return true if enabled, false if disabled.
     * @throws DriverException for I/O errors.
     * @throws IllegalArgumentException if the channel is analog.
     */
    public boolean alarmIsEnabled(DataChannel chan) throws DriverException {
        return hw.alarmIsEnabled(chan);
    }
    
    /**
     * Sets the device's time-of-day clock to the current system time.
     * @param tolerance When read back the device clock must be no further than
     * this from the current system time just after the read.
     * @throws DriverException for I/O errors or if the clock value read back is bad.
     */
    public void setClock(Duration tolerance) throws DriverException {
        hw.setClock(Instant.now().getEpochSecond());
        final Instant deviceNow = getClock();
        final Instant sysLow = Instant.now().minus(tolerance);
        final Instant sysHigh = sysLow.plus(tolerance.multipliedBy(2));
        if (deviceNow.isBefore(sysLow) | deviceNow.isAfter(sysHigh)) {
            throw new DriverException("Can't get device clock within "+tolerance+" of system time.");
        }
    }
    
    /**
     * Gets the device's time-of-day clock.
     * @return The clock reading.
     * @throws DriverException for I/O errors.
     */
    public Instant getClock() throws DriverException {
        return Instant.ofEpochSecond(hw.getClock());
    }
    
    

    // Lambda expressions can't throw checked exceptions, so we'll have to
    // wrap them in this unchecked exception.
    private static class WrapperException extends RuntimeException {
        WrapperException(DriverException exc) {super(exc);}
    }

    private DataRecord fetchRecord(int irec) {
        try {
            // Tell the instrument which record we want then read it.
            hw.writeRecordIndex(irec);
            return hw.readRecord();
        }
        catch (DriverException exc) {throw new WrapperException(exc);}
     }

    // Custom Stream operator, since Stream.takeWhile() won't be implemented until Java 9.
    // Like the standard filter() except that the stream is terminated by the
    // first non-conforming object (which doesn't appear in the final stream).
    // Stolen from http://stackoverflow.com/questions/20746429/limit-a-stream-by-a-predicate
    private static <T> Stream<T> takeWhile(Stream<T> stream, Predicate<? super T> predicate) {
        return StreamSupport.stream(takeWhile(stream.spliterator(), predicate), false);
    }
    
    private static <T> Spliterator<T> takeWhile
    (Spliterator<T> splitr, Predicate<? super T> predicate)
    {
        return new Spliterators.AbstractSpliterator<T>(splitr.estimateSize(), 0) {
            boolean stillGoing = true;
            @Override
            public boolean tryAdvance(Consumer<? super T> consumer) {
                if (stillGoing) {
                    boolean hadNext = splitr.tryAdvance(elem -> {
                        if (predicate.test(elem)) {
                            consumer.accept(elem);
                        }
                        else {
                            stillGoing = false;
                        }
                    });
                    return hadNext && stillGoing;
                }
                return false;
            }
    };
}

}
