package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.utilities.logging.Logger;

import java.time.Duration;
import java.time.Instant;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import java.util.stream.Collectors;
import java.util.stream.DoubleStream;
import java.util.stream.IntStream;
import java.util.stream.LongStream;
import java.util.stream.Stream;

/**
 * Very simple ASCII Modbus communications, just good enough for the Lighthouse particle counter.
 * <p>
 * Built atop the CCS ASCII device driver.
 * <p>
 * The internal form of a Modbus command or response used by this class is an array of int where each int
 * contains a byte value 0-255. The last byte is reserved for the Linear Redundancy Check
 * to be calculated prior to encoding as a frame. Big-endian byte order is used for multi-byte quantities
 * such as register addresses. The non-LRC bytes for a register reading command are:
 * <ul>
 * <li>Modbus slave number (1 to 247)</li>
 * <li>Function code (0 to 127)</li>
 * <li>Register address (high)</li>
 * <li>Register address (low)</li>
 * <li>Number of registers (high)</li>
 * <li>Number of registers (low)</li>
 * </ul>
 * <p>
 * The register writing command format is similar except that the number of registers to read
 * is replaced by the value to write to the register. Note that all Modbus registers are 16 bits wide.
 * <p>
 * The internal form of a normal response to a read-registers command echos the slave number and function code
 * of the command, then follows with:
 * <ul>
 * <li>Count N of data bytes to follow. Yes, BYTES not registers, and only one byte for the count itself.</li>
 * <li>Data byte 0 (first register high)</li>
 * <li>Data byte 1 (first register low)</li>
 * <li>...</li>
 * <li>Data byte N - 2 (last register high)</li>
 * <li>Data byte N - 1 (last register low).</li>
 * </ul>
 * I don't know what would happen if you tried to read more than 128 registers using a single command.
 * You might get an exception or multiple responses.
 * <p>
 * The normal response to a write-register command is to echo the command without change.
 * <p>
 * An exception (abnormal) response looks like this:
 * <ul>
 * <li>Slave number</li>
 * <li>Original function code + 128</li>
 * <li>Error code.</li>
 * </ul>
 * <p>
 * The transmitted (framed) representation of the internal binary command or response conforms to
 * the older 7-bit ASCII variant of the Modbus protocol. The frame starts with a colon, is followed
 * by two hex characters for each data byte and ends with CRLF. "Data byte" includes the LRC
 * byte which is calculated before framing and set as the last data byte. The LRC is simply the
 * lowest-order eight bits of the twos-complement of the sum of the non-LRC data bytes (before
 * framing).
 * <p>
 * The Lighthouse counter implements just two classes of Modbus registers: holding (16-bit read/write) and
 * input (16-bit read-only). The register addresses used in commands are not those that appear
 * in the documentation. There, input register addresses start at 30001 and holding register
 * addresses start at 40001. Each command deals with only a single class of register and so the
 * address in the command is the documented address minus the starting address of the class.
 * <p>
 * This software treats all Modbus registers as 16-bit unsigned values. In order to use streams and
 * avoid problems with sign extension, ints are used for eight-bit and 16-bit quantities while
 * longs are used for 32-bit quantities. A register value is represented
 * as first the high-order byte, then the low. A 32-bit unsigned int is represented using
 * a pair of consecutive registers with the high 16 bits in the lower-addressed register.
 * An IEEE single-precision floating-point value also uses two consecutive registers but in this
 * case the high portion containing the exponent is in the second register. A string is represented
 * as a series of eight-bit bytes occupying a series of consecutive registers, read from
 * lowest to highest address. The first character in each register is in the high-order byte. The
 * number of registers depends on the maximum length of the string. Strings shorter than the
 * maximum are padded on the right with NULs, but no NUL appears for a string that uses all
 * the bytes in the assigned registers.
 * @author tether
 */
public class ASCIIModbus implements Hardware {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.drivers.lighthouse");
    private final Transport transport;
    private final Duration timeout;
    private final boolean debug;
    private final Map<Integer, String> locationMap = new HashMap<>();

    /**
     * Saves a reference to the transport used for the device and saves the given
     * parameters.
     * @param transport The open transport link.
     * @param timeout How long to wait for a response to a command.
     * @param debug Only if true, all frames sent and received will be logged with
     * {@code <package logger>.debug()}.
     */
    public ASCIIModbus(Transport transport, Duration timeout, boolean debug) {
        this.transport = transport;
        this.timeout = timeout;
        this.debug = debug;
    }

    /** The slave ID number used by the device. */
    private static final int MODBUS_SLAVENO = 1;
    
    /** The Modbus command code for reading a series of holding (r/w) registers. */
    private static final int MODBUS_READ_HOLDING = 3;
    
    /** The Modbus command code for reading a series of input (r/o) registers. */
    private static final int MODBUS_READ_INPUT = 4;
    
    /** The Modbus command code for writing a holding register. */
    private static final int MODBUS_WRITE_HOLDING = 6;

    
    /** {@inheritDoc } */
    @Override
    public String readDeviceModel() throws DriverException {
        if (debug) log.debug("Reading model.");
        return getString(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, 14, 8));
    }

    /** {@inheritDoc } */
    @Override
    public int readDeviceMapVersion() throws DriverException {
        if (debug) log.debug("Reading map version.");
        return getShorts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, 0, 1))[0];
    }

    /** {@inheritDoc } */
    @Override
    public Set<DeviceFlag> readDeviceFlags() throws DriverException {
        if (debug) log.debug("Reading device flags.");
        return DeviceFlag.flagsFromMask(
                getShorts(
                        readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, 2, 1)
                )[0]
        );
    }

    /** {@inheritDoc } */
    @Override
    public void writeDeviceCommand(DeviceCommand cmd) throws DriverException {
        if (debug) log.debug("Writing command " + cmd);
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, 1, cmd.getCommandNum());
    }

    /** {@inheritDoc } */
    @Override
    public int readRecordCount() throws DriverException {
        if (debug) log.debug("Reading record count.");
        return getShorts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, 23, 1))[0];
    }

    /** {@inheritDoc } */
    @Override
    public void writeRecordIndex(int index) throws DriverException {
        if (debug) log.debug("Writing " + index + " to record index.");
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, 24, index);
    }

    /** {@inheritDoc } */
    @Override
    public Instant readRecordStartTime() throws DriverException {
        if (debug) log.debug("Reading record start time.");
        return Instant.ofEpochSecond(
                getInts(
                        readRegisters(MODBUS_SLAVENO, MODBUS_READ_INPUT, 0, 2)
                )[0]
        );
    }

    /** {@inheritDoc } */
    @Override
    public Duration readRecordDuration() throws DriverException {
        if (debug) log.debug("Reading record duration.");
        return Duration.ofSeconds(
                getInts(
                        readRegisters(MODBUS_SLAVENO, MODBUS_READ_INPUT, 2, 2)
                )[0]
        );
    }

    private static final int LOCATION_STRINGS_ADDRESS = 199;
    
    /** {@inheritDoc } */
    @Override
    public String readRecordLocation() throws DriverException {
        if (debug) log.debug("Reading record location.");
        final int locno = (int)getInts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_INPUT, 4, 2))[0];
        // Read from the location name array. Location numbers start at 1.
        return readLocationString(locno);
    }
    
    /**
     * Reads the location string corresponding to a location number.
     * @param location The location number, {@literal >=} 1.
     * @return The location string.
     * @throws DriverException for I/O errors.
     */
    public String readLocationString(int location) throws DriverException {
        return getString(
            readRegisters(
                MODBUS_SLAVENO,
                MODBUS_READ_HOLDING,
                LOCATION_STRINGS_ADDRESS + 4*(location - 1), 4));
    }

    /** {@inheritDoc } */
    @Override
    public Set<RecordFlag> readRecordFlags() throws DriverException {
        if (debug) log.debug("Reading record flags.");
        return RecordFlag.flagsFromMask(
                (int)getInts(
                        readRegisters(MODBUS_SLAVENO, MODBUS_READ_INPUT, 6, 2)
                )[0]
        );
    }

    /** {@inheritDoc } */
    @Override
    public ChannelType readChannelType(DataChannel chan) throws DriverException {
        if (debug) log.debug(String.format("Reading type for channel '%s'.", chan.getName()));
        final int addr = channelTypeAddress(chan.getIndex());
        final String devType = getString(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, addr, 2));
        final ChannelType ctype = ChannelType.fromTypeLabel(devType);
        if (ctype == null)
            throw new DriverException("Unknown channel data type: " + devType);
        if (!ctype.equals(chan.getType()))
            throw new DriverException(
                    String.format(
                            "Wrong data type of '%s' (%s) for channel '%s' which should be a %s channel",
                            devType, ctype, chan.getName(), chan.getType()
                    )
            );
        return ctype;
    }

    /** {@inheritDoc } */
    @Override
    public ChannelUnits readChannelUnits(DataChannel chan) throws DriverException {
        if (debug) log.debug(String.format("Reading units for channel '%s'.", chan.getName()));
        final int addr = channelUnitsAddress(chan.getIndex());
        final String devUnits = getString(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, addr, 2));
        final ChannelUnits cunits = ChannelUnits.fromUnitsLabel(devUnits);
        if (cunits == null)
            throw new DriverException("Unknown channel data units: " + devUnits);
        if (!cunits.equals(chan.getUnits()))
            throw new DriverException(
                    String.format("Read units of '%s' (%s) for channel '%s' instead of '%s' (%s)",
                            devUnits, cunits, chan.getName(), chan.getUnits().getDeviceText(), chan.getUnits()
                    )
            );
        return cunits;
    }

    /** {@inheritDoc } */
    @Override
    public double readChannelValue(DataChannel chan) throws DriverException {
        if (debug) log.debug(String.format("Reading value for channel '%s'.", chan.getName()));
        if (!channelIsEnabled(chan))
            throw new DriverException(String.format("Channel '%s' is disabled.", chan.getName()));
        final int addr = channelValueAddress(chan.getIndex());
        final int[] response = readRegisters(MODBUS_SLAVENO, MODBUS_READ_INPUT, addr, 2);
        return chan.isAnalog() ? getFloats(response)[0] : getInts(response)[0];
    }

    /** {@inheritDoc } */
    @Override
    public boolean channelIsEnabled(DataChannel chan) throws DriverException {
        final int addr = channelEnablesAddress(chan.getIndex());
        final long enables = getInts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, addr, 2))[0];
        return (enables & 1) != 0;
    }
    
        
    /** The address (less 3001) of the first input register in the record header. */
    public static final int RECORD_ADDRESS = 0;
    
    /** The offset from {@code RECORD_ADDRESS} of the timestamp-high register. */
    public static final int TIMESTAMP_OFFSET = 0;
    
    /** The offset from {@code RECORD_ADDRESS} of the sample-time-high register. */
    public static final int SAMPLE_TIME_OFFSET = 2;
    
    /** The offset from {@code RECORD_ADDRESS} of the location-number-high register. */
    public static final int LOCATION_NUMBER_OFFSET = 4;
    
    /** offset from {@code RECORD_ADDRESS} of the device-status-high register. */
    public static final int DEVICE_STATUS_OFFSET = 6;
    
    /** The offset from {@code RECORD_ADDRESS} of the first input register with
     *  particle channel data. */
    public static final int PARTICLE_CHANNEL_OFFSET = 8;
    
    /** The offset from {@code RECORD_ADDRESS} of the first input register with
     * analog channel data. */
    public static final int ANALOG_CHANNEL_OFFSET = 40;
    
    /** The offset from {@code RECORD_ADDRESS} of the first input register containing
     *  channel alarm flags.
    */
    public static final int ALARM_FLAGS_OFFSET = 74;
    
    /** The number of registers in the record header. Particle channels abut the record header. */
    public static final int RECORD_HEADER_SIZE = PARTICLE_CHANNEL_OFFSET;

    /** {@inheritDoc } */
    @Override
    public DataRecord readRecord() throws DriverException {
        // This implementation is much faster than either reading each register
        // individually or reading all the data record registers in a single
        // operation. It also has the advantage of not mixing floats and ints
        // in the same read.
        final long[] headerAndPChannels =
            getInts(
                readRegisters(MODBUS_SLAVENO,
                    MODBUS_READ_INPUT,
                    RECORD_ADDRESS,
                    RECORD_HEADER_SIZE + 2 * DataChannel.NUM_PARTICLE_CHANNELS)
            );
        final double[] AChannels =
            getFloats(
                readRegisters(MODBUS_SLAVENO,
                    MODBUS_READ_INPUT,
                    RECORD_ADDRESS + ANALOG_CHANNEL_OFFSET,
                    2 * DataChannel.NUM_ANALOG_CHANNELS)
            );
        
        // Collect the channel data. First the particle channels, then the analog channels.
        List<ChannelDatum> data = 
            Stream.of(DataChannel.values())
            .filter(chan -> !chan.isAnalog())
            .map(chan ->
                new ChannelDatum(
                    chan,
                    (double)headerAndPChannels[PARTICLE_CHANNEL_OFFSET/2 + chan.getIndex()]
                )
            )
            .collect(Collectors.toList());
        data
            .addAll(
                Stream.of(DataChannel.values())
                .filter(chan -> chan.isAnalog())
                .map(chan ->
                    new ChannelDatum(
                        chan,
                        AChannels[chan.getIndex() - DataChannel.FIRST_ANALOG_CHANNEL.getIndex()]
                    )
                )
                .collect(Collectors.toList())
            );
        // Extract the record header info. First look up the location string
        // corresponding to the location number, then make the data record.
        final Integer locno = (int)headerAndPChannels[2];
        String locStr = locationMap.get(locno);
        if (locStr == null) {
            locStr = readLocationString(locno);
            locationMap.put(locno, locStr);
        }
        return new DataRecord(
            Instant.ofEpochSecond(headerAndPChannels[0]), // Start time.
            Duration.ofSeconds(headerAndPChannels[1]),    // Sample duration.
            locStr,
            RecordFlag.flagsFromMask((int)headerAndPChannels[3]), // Record flags.
            data
        );
    }
       
    /** {@inheritDoc } */
    @Override
    public void setAlarmThreshold(DataChannel chan, long threshold) throws DriverException {
        if (chan.isAnalog()) throw new IllegalArgumentException("Alarms don't apply to analog channels.");
        final int addr = channelThresholdAddress(chan.getIndex());
        if ((threshold & 0xffffffffL) != threshold) {
            throw new IllegalArgumentException("Threshold values can't exceed 32 bits.");
        }
        final int thr = (int)threshold;
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, addr  , (thr >> 16) & 0xffff);
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, addr+1, thr         & 0xffff);
    }
    
    /** {@inheritDoc } */
    @Override
    public long getAlarmThreshold(DataChannel chan) throws DriverException {
        if (chan.isAnalog()) throw new IllegalArgumentException("Alarms don't apply to analog channels.");
        final int addr = channelThresholdAddress(chan.getIndex());
        return getInts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, addr, 2))[0];
    }
    
    /** {@inheritDoc } */
    @Override
    public void setAlarmEnable(DataChannel chan, boolean on) throws DriverException {
        if (chan.isAnalog()) throw new IllegalArgumentException("Alarms don't apply to analog channels.");
        final int addr = channelEnablesAddress(chan.getIndex());
        int enb = (int)getInts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, addr, 2))[0];
        if (on) enb |= 2; else enb &= ~2;
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, addr  , (enb >> 16) & 0xffff);
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, addr+1, enb         & 0xffff);
    }

    /** {@inheritDoc } */
    @Override
    public boolean alarmIsEnabled(DataChannel chan) throws DriverException {
        if (chan.isAnalog()) throw new IllegalArgumentException("Alarms don't apply to analog channels.");
        final int addr = channelEnablesAddress(chan.getIndex());
        final long enables = getInts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, addr, 2))[0];
        return (enables & 2) != 0;
    }
    
    private final int CLOCK_ADDRESS = 26;
    private final int DATA_SET_ADDRESS = 34;
    
    /** {@inheritDoc } */
    @Override
    public void setClock(long epochSecond) throws DriverException {
        if ((epochSecond & 0xffffffffL) != epochSecond) {
            throw new IllegalArgumentException("Clock values can't exceed 32 bits.");
        }
        final long high = (epochSecond >> 16) & 0xffffL;
        final long low  = epochSecond         & 0xffffL;
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, DATA_SET_ADDRESS  , (int)high);
        writeRegister(MODBUS_SLAVENO, MODBUS_WRITE_HOLDING, DATA_SET_ADDRESS+1, (int)low);
        this.writeDeviceCommand(DeviceCommand.SET_CLOCK);
    }
    
    /** {@inheritDoc } */
    @Override
    public long getClock() throws DriverException {
        return getInts(readRegisters(MODBUS_SLAVENO, MODBUS_READ_HOLDING, CLOCK_ADDRESS, 2))[0];
    }
    
    /**
     * Gives the holding register address to use in Modbus commands for the data type of a channel.
     * @param index The zero-based channel number.
     * @return The address.
     */
    private int channelTypeAddress(int index) {
        return 1008 + 2 * index;
    }

    /**
     * Gives the holding register address to use in Modbus commands for the units of a channel.
     * @param index The zero-based channel number.
     * @return The address.
     */
    private int channelUnitsAddress(int index) {
        return 2008 + 2 * index;
    }

    /**
     * Gives the holding register address to use in Modbus commands for the enables word of a channel.
     * @param index The zero-based channel number.
     * @return The address.
     */
    private int channelEnablesAddress(int index) {
        return 3008 + 2 * index;
    }

    /**
     * Gives the holding register address to use in Modbus commands for the
     * alarm threshold of a channel.
     * @param index The zero-based channel number.
     * @return The address.
     */
    private int channelThresholdAddress(int index) {
        return 5008 + 2 * index;
    }

    /**
     * Gives the input register address to use in Modbus commands for the value of a channel.
     * @param index The zero-based channel number.
     * @return The address.
     */
    private int channelValueAddress(int index) {
        return 8 + 2 * index;
    }

    /**
     * Sends a read-registers command then waits for and decodes the response.
     * @param slaveno The eight-bit Modbus slave number.
     * @param function The seven-bit function code.
     * @param address  The 16-bit register address.
     * @param nreg The 16-bit number of registers to read.
     * @return The decoded response frame, see {@link #decodeFrame(java.lang.String) }.
     * @throws DriverException if the Modbus device reports an exception or an I/O error occurs.
     */
    private int[] readRegisters(int slaveno, int function, int address, int nreg)
            throws DriverException
    {
        transport.send(encodeFrame(makeReadCommand(slaveno, function, address, nreg)));
        return checkResponse(slaveno, function);
    }

    /**
     * Sends a write-register command then waits for and decodes the response.
     * @param slaveno The eight-bit Modbus slave number.
     * @param function The seven-bit function code.
     * @param value The 16-bit value to write.
     * @throws DriverException if the Modbus device reports an exception or an I/O error occurs.
     */
    private void writeRegister(int slaveno, int function, int address, int value)
            throws DriverException
    {
        transport.send(encodeFrame(makeWriteCommand(slaveno, function, address, value)));
        checkResponse(slaveno, function);
    }

    /**
     * Wait for a response and check it for correct function code, correct slave number and
     * whether it's an exception response.
     * @param slaveno The Modbus slave number.
     * @param function The function code.
     * @return The decoded response frame, see {@link #decodeFrame(java.lang.String) }.
     * @throws DriverException for I/O errors, exception responses, incorrect slave or function
     * numbers.
     */
    private int[] checkResponse(int slaveno, int function) throws DriverException {
        final int[] response = decodeFrame(transport.receive(this.timeout));
        if (response[0] != slaveno)
            throw new DriverException("Wrong slave no. in Modbus response");
        if ((response[1] & 0x7f) != function) // Mask off the exception bit.
            throw new DriverException("Wrong function code in Modbus response");
        if (isException(response))
            throw new DriverException("Modbus exception " + getExceptionCode(response));
        return response;
    }

    /**
     * Create, as binary byte values without framing, a Modbus register read commmand. A trailing
     * zero byte will be added as a placeholder for the LRC.
     * @param slaveno The eight-bit Modbus slave number.
     * @param function The seven-bit function code.
     * @param address  The 16-bit register address.
     * @param nreg The 16-bit number of registers to read.
     * @return The int array each containing a command byte, in transmission order.
     */
    private static int[] makeReadCommand(int slaveno, int function, int address, int nreg) {
        return new int[]{
            slaveno & 0xff,
            function & 0xff,
            (address >> 8) & 0xff,
            address & 0xff,
            (nreg >> 8) & 0xff,
            nreg & 0xff,
            0
        };
    }

    /**
     * Create, as binary bytes values without framing, a Modbus command to write a single register. A
     * trailing zero byte will be added as a placeholder for the LRC.
     * @param slaveno The eight-bit Modbus slave nmber.
     * @param function The seven-bit function code.
     * @param address The 16-bit register address.
     * @param value The 16-bit register value.
     * @return The array of command bytes in transmission order.
     */
    private static int[] makeWriteCommand(int slaveno, int function, int address, int value) {
        return new int[]{
            slaveno & 0xff,
            function & 0xff,
            (address >> 8) & 0xff,
            address & 0xff,
            (value >> 8) & 0xff,
            value & 0xff,
            0
        };
    }

    /**
     * Converts a byte-value-array Modbus command to an ASCII Modbus frame. Frame format: (1) ":",
     * (2) The command bytes converted to two hex digits each (big-endian, uppercase),
     * (3) An eight-bit Linear Redundancy Check converted to ASCII hex form (big-endian, uppercase).
     * The trailing CRLF is left off and must be added during transmission.
     * <p>
     * The LRC is calculated from the command bytes before framing.
     * @param command The command bytes to be encoded.
     * @return The frame described above, using US-ASCII encoding.
     */
    private String encodeFrame(int[] command) {
        // Add the LRC.
        final int sum = IntStream.of(command)
                .limit(command.length - 1)
                .sum();
        command[command.length - 1] = (-sum) & 0xff;
        // Convert to ASCII.
        final String frame = Stream.concat(
                Stream.of(":"),
                IntStream.of(command).mapToObj(v -> String.format("%02X", v))
        ).collect(Collectors.joining());
        if (debug) log.debug("Sending " + frame);
        return frame;
    }

    /**
     * Regex used to check response frames. A frame must start with a colon followed only by
     * pairs of hex digits, at least four such pairs. This assumes that the trailing CRLF
     * has already been stripped off. Four pairs is the minimum in order to allow
     * for exception responses.
     */
    private static final Pattern frameRegex = Pattern.compile(":(?:\\p{XDigit}{2}){4,}");

    /**
     * Converts an ASCII response back into a binary byte-value array. The leading colon is
     * removed, the LRC is checked and if it's valid the data bytes are
     * converted from their ASCII form into binary bytes.
     * @param frame The received frame with the trailing CRLF removed.
     * @return The byte array.
     */
    private int[] decodeFrame(final String frame) throws DriverException {
        if (debug) log.debug("Received " + frame);
        // Check the frame's form.
        final Matcher mat = frameRegex.matcher(frame);
        if (!mat.matches())
            throw new DriverException("Malformed Modbus response frame");
        // Convert to byte-values.
        final int[] result = IntStream.iterate(1, i -> i + 2)
                .limit((frame.length() - 1) / 2)
                .map(i -> Integer.parseInt(frame.substring(i, i+2), 16))
                .toArray();
        // Check the LRC.
        final int sum = IntStream.of(result).sum() & 0xff;
        if (sum != 0)
            throw new DriverException("Modbus LRC is incorrect");
        return result;
    }

    /**
     * Checks whether a response decoded from a frame represents an exception. Exceptions
     * have the high bit set in the function code.
     * @param response The decoded response.
     * @return true if the response is an exception, else false.
     */
    private static boolean isException(int[] response) {return (response[1] & 0x80) != 0;}

    /**
     * Gets the eight-bit exception code from a decoded exception response.
     * @param response The decoded response, presumed to be an exception.
     * @return The exception code.
     */
    private static int getExceptionCode(int[] response) {return response[2];}

    /**
     * Interprets the data portion of a decoded non-exception response as a series of
     * 16-bit unsigned integers. Each value is composed of two eight-bit unsigneds
     * A then B such that the value is {@code (A<<8) | B}.
     * @param response The decoded response.
     * @return The values grouped into an array of int.
     */
    private int[] getShorts(int[] response) {
        final int nreg = response[2] / 2;
        final int[] shorts = new int[nreg];
        for (int i = 0; i < nreg; ++i) {
            shorts[i] = (response[3+2*i] << 8) | response[4+2*i];
        }
        if (debug) {
            log.debug(
                IntStream.of(shorts)
                    .mapToObj(i -> String.format("%04x", i))
                    .collect(Collectors.joining(" ", "Shorts (hex): ", ""))
            );
        }
        return shorts;
    }

    /**
     * Interprets the data portion of a decoded non-exception response as a series of
     * 32-bit unsigned integers. Each value is composed of two 16-bit unsigneds
     * A then B such that the value is {@code (A<<16) | B}.
     * @param response The decoded response.
     * @return The values grouped into an array of long.
     */
    private long[] getInts(int[] response) {
        final int[] shorts = getShorts(response);
        final long[] ints = new long[shorts.length / 2];
        for (int i = 0; i < ints.length; ++i) {
            ints[i] = ((long)shorts[2*i] << 16) | (long)shorts[2*i+1];
        }
        if (debug) {
            log.debug(
                LongStream.of(ints)
                    .mapToObj(i -> String.format("%08x", i))
                    .collect(Collectors.joining(" ", "Ints (hex): ", ""))
            );
        }
        return ints;
    }

    /**
     * Interprets the data portion of a decoded non-exception response as a series of
     * IEEE single floats. Each value is composed of two 16-bit unsigneds A then B
     * such that the float is {@code Float.intBitsToFloat((B << 16) | A)}.
     * @param response The decoded response.
     * @return The values grouped into an array of double.
     */
    private double[] getFloats(int[] response) {
        final double[] floats = LongStream.of(getInts(response))
                .map(l -> ((l & 0xffffL)<< 16) | ((l >> 16) & 0xffffL)) // Swap halves.
                .mapToDouble(l -> Float.intBitsToFloat((int)l))
                .toArray();
        if (debug) {
            log.debug(
                DoubleStream.of(floats)
                    .mapToObj(i -> String.format("%e", i))
                    .collect(Collectors.joining(" ", "Floats: ", ""))
            );
        }
        return floats;
    }

    /**
     * Interprets the data portion of a decoded non-exception response as a
     * US-ASCII string. It's assumed that the string has no embedded NULs
     * but may have trailing NULs which are to be dropped.
     * @param response The decoded response.
     * @return The string, with trailing NUL characters stripped.
     */
    private String getString(int[] response) {
        final String s = IntStream.of(response)
                .skip(3)             // Skip slave no., function and byte count.
                .limit(response[2])  // Omit LRC.
                .filter(i -> i != 0) // No NULs.
                .mapToObj(i -> String.valueOf((char)i))
                .collect(Collectors.joining());
        if (debug) log.debug("String: " + s);
        return s;
    }



}
