package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;

import java.time.Duration;

import java.util.Optional;

/**
 **************************************************************************
 **
 ** Shell command driver.
 **
 ** Driver methods are annotated for use by CCS command dictionary and shell
 ** system. Exceptions are passed through to next level where either subsystem
 ** or shell code handles them.
 **
 ** @author tether
 **
 ***************************************************************************
 */


public class LighthouseDriver {

    /**
     * Contains the objects that describe an open device.
     */
    private final static class DriverClient extends Client {
        private final Transport trans;

        /**
         * Saves the transport object and creates new Client and ASCIIModbus objects.
         * @param trans An open data transport channel to the device.
         * @param debug The debug flag used to create an ASCIIModbus instance.
         * @throws DriverException
         */
        public DriverClient(Transport trans, boolean debug) throws DriverException {
            super(new ASCIIModbus(trans, LH_COMM_TIMEOUT, debug));
            this.trans = trans;
        }

        /**
         * Closes the link to the device.
         */
        public void close() {trans.close();}
    }

    /**
     **************************************************************************
     **
     ** Public constants
     **
     ***************************************************************************
     */

    /**
     **************************************************************************
     **
     ** Private constants
     **
     ***************************************************************************
     */
    private final static Duration LH_COMM_TIMEOUT = Duration.ofSeconds(10);

    /**
     **************************************************************************
     **
     ** Private fields
     **
     ***************************************************************************
     */

    private static final Logger log = Logger.getLogger("org.lsst.ccs.drivers.lighthouse");

    private volatile Optional<DriverClient> lhClient = Optional.empty();

    /**
     **************************************************************************
     **
     ** Constructor.
     **
     ***************************************************************************
     */
    public LighthouseDriver() throws DriverException {}

    /**
     **************************************************************************
     **
     ** Opens the device on a serial port. Creates suitable implementations of
     ** the interfaces Transport and Hardware.
     ** @param portName The serial port name, for example /dev/ttyUSB0.
     ** @param debug If true enables debug-level logging of device command traffic.
     ** @throws DriverException for all errors.
     **
     **
     ***************************************************************************
     */
    @Command(description = "Opens the device on a serial port.")
    public void serialOpen(
            @Argument(description="The serial port name, for example /dev/ttyUSB0.")
            String portName,
            @Argument(description="When true, activates debug logging.", defaultValue="false")
            boolean debug
    ) throws DriverException
    {
        lhClient.ifPresent(x -> x.close());
        lhClient = Optional.of(new DriverClient(new SerialTransport(portName), debug));
        checkDevice();
    }


    /**
     **************************************************************************
     **
     ** Opens the device on an FTDI USB-serial cable. Creates suitable implementations of
     ** the interfaces Transport and Hardware.
     ** @param ftdiID The serial no. of the FTDI device to use, preceded by a hostname
     ** and a colon if remote. Examples: XXXXX or foo:XXXXX.
     ** @param debug If true enables debug-level logging of device command traffic.
     ** @throws DriverException for all errors.
     **
     **
     ***************************************************************************
     */
    @Command(description = "Opens the device on an FTDI USB-serial cable.")
    public void ftdiOpen(
            @Argument(description=
                    "The serial no. of the cable, preceded by a hostname and a colon if it's a remote cable.")
            String ftdiID,
            @Argument(description="When true, activates debug logging.", defaultValue="false")
            boolean debug
    ) throws DriverException
    {
        lhClient.ifPresent(x -> x.close());
        lhClient = Optional.of(new DriverClient(new FTDITransport(ftdiID), debug));
        checkDevice();
    }


    /**
     **************************************************************************
     **
     ** Opens the device on a network socket. Creates suitable implementations of
     ** the interfaces Transport and Hardware.
     ** @param hostname The remote host's name or IP address.
     ** @param portNo The number of the port to connect to.
     ** @param debug If true enables debug-level logging of device command traffic.
     ** @throws DriverException for all errors.
     **
     **
     ***************************************************************************
     */
    @Command(description = "Opens the device on a network socket.")
    public void netOpen(
            @Argument(description="The remote host's name or IP address.")
            String hostname,
            @Argument(description="The number of the port to connect to.")
            int portNo,
            @Argument(description="When true, activates debug logging.", defaultValue="false")
            boolean debug
    ) throws DriverException
    {
        lhClient.ifPresent(x -> x.close());
        lhClient = Optional.of(new DriverClient(new NetTransport(hostname, portNo), debug));
        checkDevice();
    }

    private void checkDevice() throws DriverException {
        final String model = lhClient.get().getModel();
        log.info("Device model: " + model);
        if (!model.matches(".*(2016|3016|5016).*")) {
            throw new DriverException("Unknown device model");
        }
        final int mapver = lhClient.get().getRegisterMapVersion();
        log.info(String.format("Modbus register map v%d.%02d", mapver/100, mapver%100));
    }

    /**
     **************************************************************************
     **
     ** Closes the device.
     ** @throws DriverException for all errors.
     **
     ***************************************************************************
     */
    @Command(name = "close", description = "Closes link used to communicate with the counter.")
    public void close() throws DriverException {
       lhClient.ifPresent(x -> x.close());
       lhClient = Optional.empty();
    }

    /**
     **************************************************************************
     **
     ** Stops the device so that it no longer updates its internal data set.
     ** Has no effect if recording is already stopped.
     ** @throws DriverException for all errors.
     **
     ***************************************************************************
     */
    @Command(
        name = "stopRecording",
        description = "Stops the counter so that it no longer updates its internal data set.")
    public void stopRecording() throws DriverException {
        lhClient.get().stopRecording();
    }

    /**
     **************************************************************************
     **
     ** Starts the device, appending all future records to its internal data set.
     ** Has no effect if the device is already active.
     ** @throws DriverException
     **
     ***************************************************************************
     */
    @Command(
        name = "startRecording",
        description = "Starts the counter, appending all future records to its internal data set.")
    public void startRecording() throws DriverException {
        lhClient.get().startRecording();
    }

    /**
     **************************************************************************
     **
     ** Clears all records from the device's internal data set. Recording must first be stopped.
     ** @throws DriverException
     **
     ***************************************************************************
     */
    @Command(
        name = "clearRecords",
        description = "Clears all readings from the counter's internal data set.")
    public void clearRecords() throws DriverException {
        lhClient.get().clearRecords();
    }

    /**
     **************************************************************************
     **
     ** Saves the device's internal data set as ASCII text. The internal data
     ** set remains unchanged. Recording must first be stopped.
     ** @param filename The name of a file to receive the data.
     ** @throws DriverException for all checked exceptions.
     **
     ** <pre><samp>
     ** begin record
     **     start     yyyy-mm-ddThh24:mm:ssZ
     **     duration  xxxxx seconds
     **     location  B84_123
     **     status    laser! sampler!
     **     channel 10.0   counter      12230 particles
     **     channel  2.5   counter      14 particles
     **     channel temp   temperature  39.1 Celsius
     **     channel hum    relhumidity  1.0 percent
     ** end record
     ** </samp></pre>
     **
     ***************************************************************************
     */
    @Command(
        name = "getRecords",
        description = "Saves the counter's buffered data set as ASCII text.")
    public void getRecords(
        @Argument(description="The name of the file in which to save")
        String filename
    ) throws DriverException {
        try (PrintWriter p = new PrintWriter(filename, "US-ASCII")) {
            printRecords(p);
        }
        catch(FileNotFoundException | UnsupportedEncodingException exc) {
            throw new DriverException("Can't open " + filename, exc);
        }
    }

    /**
     **************************************************************************
     **
     ** Prints to stdout the device's internal data set using the default encoding.
     ** The internal data
     ** set remains unchanged. Recording must first be stopped.
     ** @throws DriverException for all checked exceptions.
     ** @see #getRecords(java.lang.String)
     **
     ***************************************************************************
     */
    @Command(
        name = "getRecords",
        description = "Prints to stdout the counter's buffered data set using the default encoding.")
    public void getRecords() throws DriverException {
        printRecords(new PrintWriter(System.out));
    }

    private void printRecords(PrintWriter out) throws DriverException {
        lhClient.get().getRecords().forEach(
                rec -> {
                    out.printf("begin record%n");
                    out.printf("    start %s%n", rec.getStartTime().toString());
                    out.printf("    duration %d seconds%n", rec.getDuration().getSeconds());
                    out.printf("    location %s%n", rec.getLocation());
                    out.printf("    status", rec);
                    rec.getFlags().stream().forEach(
                            flag -> out.printf(" %s", flag)
                    );
                    out.println();
                    rec.getChannels().forEach(
                            datum -> {
                                final DataChannel chan = datum.getChannel();
                                out.printf("    channel");
                                out.printf(" %s", chan.getName());
                                out.printf(" %s", chan.getType());
                                if (chan.isAnalog()) {
                                    out.printf(" %5.2f", datum.getValue());
                                }
                                else {
                                    out.printf(" %d", (long)datum.getValue());
                                }
                                out.printf(" %s%n", chan.getUnits());
                            }
                    );
                    out.printf("end record%n");
                }
        );
        out.flush();
    }

}
