package org.lsst.ccs.drivers.lighthouse;

/**
 * The data channels we can read from a Lighthouse counter.
 *
 * For each channel the instrument keeps several fields such as value, units, flags, etc. For each such field
 * it has an array we can read with one element per channel. Each element consists of N 16-bit registers
 * where N depends on the type of the field. The channel index, I, assigned to each member of this enum
 * is used to index these arrays: regAddress = I * N + arrayBaseAddress.
 * <p>
 * The device stores analog values as IEEE single floats and digital values as ints, hence we need
 * {@link #isAnalog() }.
 * <p> The device allows for 16 possible counter channels followed by 16 possible analog channels.
 * @author tether
 */
public enum DataChannel {
    /** Counter of 0.3-micron and larger particles. */
    COUNTER_0_3 (0,  "0.3", false, ChannelType.COUNTER, ChannelUnits.COUNTS),
    /** Counter of 0.5-micron and larger particles. */
    COUNTER_0_5 (1,  "0.5", false, ChannelType.COUNTER, ChannelUnits.COUNTS),
    /** Counter of 1-micron and larger particles. */
    COUNTER_1_0 (2,  "1.0", false, ChannelType.COUNTER, ChannelUnits.COUNTS),
    /** Counter of 2.5-micron and larger particles. */
    COUNTER_2_5 (3,  "2.5", false, ChannelType.COUNTER, ChannelUnits.COUNTS),
    /** Counter of 5-micron and larger particles. */
    COUNTER_5_0 (4,  "5.0", false, ChannelType.COUNTER, ChannelUnits.COUNTS),
    /** Counter of 10-micron and larger particles. */
    COUNTER_10_0(5, "10.0", false, ChannelType.COUNTER, ChannelUnits.COUNTS),
    /** Ambient temperature probe. */
    TEMP       (16, "temp", true, ChannelType.TEMPERATURE, ChannelUnits.FAHRENHEIT),
    /** Ambient humidity probe. */
    HUMID       (17, "humid", true, ChannelType.RELATIVE_HUMIDITY, ChannelUnits.PERCENTAGE);

    private final int index;
    private final String name;
    private final boolean analog;
    private final ChannelType type;
    private final ChannelUnits units;

    DataChannel(int index, String name, boolean analog, ChannelType type, ChannelUnits units) {
        this.index = index;
        this.name = name;
        this.analog = analog;
        this.type = type;
        this.units = units;
    }

    /**
     * Gets the index of the channel.
     * @return The index.
     */
    public int getIndex() {return index;}

    /**
     * Gets the type of quantity measured.
     * @return The type.
     */
    public ChannelType getType() {return type;};

    /**
     * Gets the name of the channel.
     * @return The name.
     */
    public String getName() {return name;};

    /**
     * Is this an analog channel?
     * @return The flag.
     */
    public boolean isAnalog() {return analog;}

    /**
     * Gets the units of the channel value.
     * @return The units.
     */
    ChannelUnits getUnits() {return units;}

}
