/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.drivers.lighthouse;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.time.Duration;
import java.util.Optional;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.lighthouse.ASCIIModbus;
import org.lsst.ccs.drivers.lighthouse.Client;
import org.lsst.ccs.drivers.lighthouse.DataChannel;
import org.lsst.ccs.drivers.lighthouse.FTDITransport;
import org.lsst.ccs.drivers.lighthouse.NetTransport;
import org.lsst.ccs.drivers.lighthouse.SerialTransport;
import org.lsst.ccs.drivers.lighthouse.Transport;
import org.lsst.ccs.utilities.logging.Logger;

public class LighthouseDriver {
    private static final Duration LH_COMM_TIMEOUT = Duration.ofSeconds(10L);
    private static final Logger log = Logger.getLogger((String)"org.lsst.ccs.drivers.lighthouse");
    private volatile Optional<DriverClient> lhClient = Optional.empty();

    @Command(description="Opens the device on a serial port.")
    public void serialOpen(@Argument(description="The serial port name, for example /dev/ttyUSB0.") String portName, @Argument(description="When true, activates debug logging.", defaultValue="false") boolean debug) throws DriverException {
        this.lhClient.ifPresent(x -> x.close());
        this.lhClient = Optional.of(new DriverClient(new SerialTransport(portName), debug));
        this.checkDevice();
    }

    @Command(description="Opens the device on an FTDI USB-serial cable.")
    public void ftdiOpen(@Argument(description="The serial no. of the cable, preceded by a hostname and a colon if it's a remote cable.") String ftdiID, @Argument(description="When true, activates debug logging.", defaultValue="false") boolean debug) throws DriverException {
        this.lhClient.ifPresent(x -> x.close());
        this.lhClient = Optional.of(new DriverClient(new FTDITransport(ftdiID), debug));
        this.checkDevice();
    }

    @Command(description="Opens the device on a network socket.")
    public void netOpen(@Argument(description="The remote host's name or IP address.") String hostname, @Argument(description="The number of the port to connect to.") int portNo, @Argument(description="When true, activates debug logging.", defaultValue="false") boolean debug) throws DriverException {
        this.lhClient.ifPresent(x -> x.close());
        this.lhClient = Optional.of(new DriverClient(new NetTransport(hostname, portNo), debug));
        this.checkDevice();
    }

    private void checkDevice() throws DriverException {
        String model = this.lhClient.get().getModel();
        log.info((Object)("Device model: " + model));
        if (!model.matches(".*(2016|3016|5016).*")) {
            throw new DriverException("Unknown device model");
        }
        int mapver = this.lhClient.get().getRegisterMapVersion();
        log.info((Object)String.format("Modbus register map v%d.%02d", mapver / 100, mapver % 100));
    }

    @Command(name="close", description="Closes link used to communicate with the counter.")
    public void close() throws DriverException {
        this.lhClient.ifPresent(x -> x.close());
        this.lhClient = Optional.empty();
    }

    @Command(name="stopRecording", description="Stops the counter so that it no longer updates its internal data set.")
    public void stopRecording() throws DriverException {
        this.lhClient.get().stopRecording();
    }

    @Command(name="startRecording", description="Starts the counter, appending all future records to its internal data set.")
    public void startRecording() throws DriverException {
        this.lhClient.get().startRecording();
    }

    @Command(name="clearRecords", description="Clears all readings from the counter's internal data set.")
    public void clearRecords() throws DriverException {
        this.lhClient.get().clearRecords();
    }

    @Command(name="getRecords", description="Saves the counter's buffered data set as ASCII text.")
    public void getRecords(@Argument(description="The name of the file in which to save") String filename) throws DriverException {
        try (PrintWriter p = new PrintWriter(filename, "US-ASCII");){
            this.printRecords(p);
        }
        catch (FileNotFoundException | UnsupportedEncodingException exc) {
            throw new DriverException("Can't open " + filename, exc);
        }
    }

    @Command(name="getRecords", description="Prints to stdout the counter's buffered data set using the default encoding.")
    public void getRecords() throws DriverException {
        this.printRecords(new PrintWriter(System.out));
    }

    private void printRecords(PrintWriter out) throws DriverException {
        this.lhClient.get().getRecords().forEach(rec -> {
            out.printf("begin record%n", new Object[0]);
            out.printf("    start %s%n", rec.getStartTime().toString());
            out.printf("    duration %d seconds%n", rec.getDuration().getSeconds());
            out.printf("    location %s%n", rec.getLocation());
            out.printf("    status", rec);
            rec.getFlags().stream().forEach(flag -> out.printf(" %s", flag));
            out.println();
            rec.getChannels().forEach(datum -> {
                DataChannel chan = datum.getChannel();
                out.printf("    channel", new Object[0]);
                out.printf(" %s", chan.getName());
                out.printf(" %s", new Object[]{chan.getType()});
                if (chan.isAnalog()) {
                    out.printf(" %5.2f", datum.getValue());
                } else {
                    out.printf(" %d", (long)datum.getValue());
                }
                out.printf(" %s%n", new Object[]{chan.getUnits()});
            });
            out.printf("end record%n", new Object[0]);
        });
        out.flush();
    }

    private static final class DriverClient
    extends Client {
        private final Transport trans;

        public DriverClient(Transport trans, boolean debug) throws DriverException {
            super(new ASCIIModbus(trans, LH_COMM_TIMEOUT, debug));
            this.trans = trans;
        }

        public void close() {
            this.trans.close();
        }
    }
}

