package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.drivers.serial.SerialPort;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;

import java.time.Duration;

import java.util.Optional;

import java.util.logging.Level;

/**
 **************************************************************************
 **
 ** Shell command driver.
 **
 ** Driver methods are annotated for use by CCS command dictionary and shell
 ** system. Exceptions are passed through to next level where either subsystem
 ** or shell code handles them.
 **
 ** @author tether
 **
 ***************************************************************************
 */


public class LighthouseDriver {

    /**
     **************************************************************************
     **
     ** Public constants
     **
     ***************************************************************************
     */

    /**
     **************************************************************************
     **
     ** Private constants
     **
     ***************************************************************************
     */
    private final int LH_BAUD_19200 = 19200;
    private final int LH_NDATA_8 = 8;
    private final int LH_NSTOP_1 = 1;
    private final int LH_PARITY_NONE = 0;
    private final Duration LH_COMM_TIMEOUT = Duration.ofSeconds(10);

    /**
     **************************************************************************
     **
     ** Private fields
     **
     ***************************************************************************
     */
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.drivers.lighthouse");
    
    private Optional<Client> lhClient = Optional.empty();
    private Optional<Hardware> lhHardware = Optional.empty();
    private Optional<SerialPort> lhPort = Optional.empty();

    /**
     **************************************************************************
     **
     ** Constructor.
     **
     ***************************************************************************
     */
    public LighthouseDriver() throws DriverException {
    }

    /**
     **************************************************************************
     **
     ** Opens the device in normal mode.
     ** @param portName The serial port name, for instance "/dev/ttyUSB0".
     ** @throws DriverException for all errors.
     **
     **
     ***************************************************************************
     */
    @Command(name = "open", description = "Opens the device in normal mode")
    public void open(
            @Argument(description = "The serial port name, for example /dev/ttyUSB0.")
            String portName
    ) throws DriverException
    {
        genericOpen(portName, false);
    }

    /**
     **************************************************************************
     **
     ** Opens the device in debug mode.
     ** @param portName The serial port name, for instance "/dev/ttyUSB0".
     ** @throws DriverException for all errors.
     **
     **
     ***************************************************************************
     */
    @Command(name = "debugOpen", description = "Opens the device in debug mode")
    public void debugOpen(
            @Argument(description = "The serial port name, for example /dev/ttyUSB0.")
            String portName
    ) throws DriverException
    {
        genericOpen(portName, true);
    }
    
    private void genericOpen(String portName, boolean debug) throws DriverException {
        close();
        
        final SerialPort port = new SerialPort();
        port.openPort(portName);
        lhPort = Optional.of(port);
        
        final Hardware hw = new ASCIIModbus(port, LH_COMM_TIMEOUT, debug);
        lhHardware = Optional.of(hw);
        
        lhClient = Optional.of(new Client(hw));
        
        final String model = lhClient.get().getModel();
        log.info("Device model: " + model);
        if (!model.matches(".*(2016|3016|5016).*")) {
            throw new DriverException("Unknown device model");
        }
        final int mapver = lhClient.get().getRegisterMapVersion();
        log.info(String.format("Modbus register map v%d.%02d", mapver/100, mapver%100));
    }

    /**
     **************************************************************************
     **
     ** Closes the device.
     ** @throws DriverException for all errors.
     **
     ***************************************************************************
     */
    @Command(name = "close", description = "Closes the serial port used to communicate with the counter.")
    public void close() throws DriverException {
        if (lhPort.isPresent()) {
            lhPort.get().closePort();
        }
        
        lhPort = Optional.empty();
        lhHardware = Optional.empty();
        lhClient = Optional.empty();
    }

    /**
     **************************************************************************
     **
     ** Stops the device so that it no longer updates its internal data set.
     ** Has no effect if recording is already stopped.
     ** @throws DriverException for all errors.
     **
     ***************************************************************************
     */
    @Command(
        name = "stopRecording",
        description = "Stops the counter so that it no longer updates its internal data set.")
    public void stopRecording() throws DriverException {
        lhClient.get().stopRecording();
    }

    /**
     **************************************************************************
     **
     ** Starts the device, appending all future records to its internal data set.
     ** Has no effect if the device is already active.
     ** @throws DriverException
     **
     ***************************************************************************
     */
    @Command(
        name = "startRecording",
        description = "Starts the counter, appending all future records to its internal data set.")
    public void startRecording() throws DriverException {
        lhClient.get().startRecording();
    }

    /**
     **************************************************************************
     **
     ** Clears all records from the device's internal data set. Recording must first be stopped.
     ** @throws DriverException
     **
     ***************************************************************************
     */
    @Command(
        name = "clearRecords",
        description = "Clears all readings from the counter's internal data set.")
    public void clearRecords() throws DriverException {
        lhClient.get().clearRecords();
    }

    /**
     **************************************************************************
     **
     ** Saves the device's internal data set as ASCII text. The internal data
     ** set remains unchanged. Recording must first be stopped.
     ** @param filename The name of a file to receive the data.
     ** @throws DriverException for all checked exceptions.
     ** 
     ** <code>
     ** begin record
     **     start     yyyy-mm-ddThh24:mm:ssZ
     **     duration  xxxxx seconds
     **     location  B84_123
     **     status    laser! sampler!
     **     channel 10.0   counter      12230 particles
     **     channel  2.5   counter      14 particles
     **     channel temp   temperature  39.1 Celsius
     **     channel hum    relhumidity  1.0 percent
     ** end record
     ** </code>
     **
     ***************************************************************************
     */
    @Command(
        name = "getRecords",
        description = "Saves the counter's buffered data set as ASCII text.")
    public void getRecords(
        @Argument(description="The name of the file in which to save")
        String filename
    ) throws DriverException {
        try (PrintWriter p = new PrintWriter(filename, "US-ASCII")) {
            printRecords(p);
        }
        catch(FileNotFoundException | UnsupportedEncodingException exc) {
            throw new DriverException("Can't open " + filename, exc);
        }
    }

    /**
     **************************************************************************
     **
     ** Prints to stdout the device's internal data set using the default encoding.
     ** The internal data
     ** set remains unchanged. Recording must first be stopped.
     ** @throws DriverException for all checked exceptions.
     ** @see #getRecords(java.lang.String) 
     **
     ***************************************************************************
     */
    @Command(
        name = "getRecords",
        description = "Prints to stdout the counter's buffered data set using the default encoding.")
    public void getRecords() throws DriverException {
        printRecords(new PrintWriter(System.out));
    }
    
    private void printRecords(PrintWriter out) throws DriverException {
        lhClient.get().getRecords().forEach(
                rec -> {
                    out.printf("begin record%n");
                    out.printf("    start %s%n", rec.getStartTime().toString());
                    out.printf("    duration %d seconds%n", rec.getDuration().getSeconds());
                    out.printf("    location %s%n", rec.getLocation());
                    out.printf("    status", rec);
                    rec.getFlags().stream().forEach(
                            flag -> out.printf(" %s", flag)
                    );
                    out.println();
                    rec.getChannels().forEach(
                            datum -> {
                                final DataChannel chan = datum.getChannel();
                                out.printf("    channel");
                                out.printf(" %s", chan.getName());
                                out.printf(" %s", chan.getType());
                                if (chan.isAnalog()) {
                                    out.printf(" %5.2f", datum.getValue());
                                }
                                else {
                                    out.printf(" %d", (long)datum.getValue());
                                }
                                out.printf(" %s%n", chan.getUnits());
                            }
                    );
                    out.printf("end record%n");
                }
        );
        out.flush();
    }

}
