package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.commons.DriverException;

import java.time.Instant;
import java.time.Duration;

import java.util.List;
import java.util.Set;

import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * Encapsulates requests made of Lighthouse particle counter. Only operations and data are visible,
 * not registers or communication protocols. Implements a command set common
 * to the 2016, 3016 and 5016 family of hand-held and wall-mounted units.
 * <p>
 * Only a single thread should manipulate the instrument using this class. Only a single stream
 * of data records should be in use at any time since advancing to the next record involves updating
 * the record index register on the instrument before reading the record data registers. This lazy
 * retrieval allows you to skip needless I/O if you decide you don't want all the stored records.
 * @author tether
 */
public final class Client {
    private final Hardware hw;
    private final String model;
    private final int registerMapVersion;

    /**
     * Stores a reference to a Hardware object used to communicate with the instrument and
     * reads the device model and the register map version.
     * @param hw The communications object.
     * @throws DriverException if communication problems arise.
     */
    public Client(Hardware hw) throws DriverException {
        this.hw = hw;
        this.model = hw.readDeviceModel();
        this.registerMapVersion = hw.readDeviceMapVersion();
    }

    /**
     * Gets the model of Lighthouse counter hardware (read by constructor).
     * @return The model designation string.
     */
    public String getModel() {return model;}

    /**
     * Gets the version of the Modbus register map used by the hardware (read by constructor).
     * @return An integer with decimal digits xxyy, where xx is the major version
     * and yy the minor version.
     */
    public int getRegisterMapVersion() {return registerMapVersion;}
    
    /**
     * Stops data collection. Aborts any new record being created. Has no effect if
     * data collection has already been stopped.
     * @throws DriverException if communication problems arise.
     */
    public void stopRecording() throws DriverException  {
        hw.writeDeviceCommand(DeviceCommand.INSTRUMENT_STOP);
    }

    /**
     * Starts automatic data collection. Has no effect if data
     * collection is already started.
     * @throws DriverException if communication problems arise.
     */
    public void startRecording() throws DriverException  {
        hw.writeDeviceCommand(DeviceCommand.INSTRUMENT_START);
    }

    /**
     * Gets all the data records currently stored in the instrument's memory.
     * @return The records in ascending order by record index.
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public Stream<DataRecord> getRecords() throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't read records when the device is running.");
        }
        final int nrec = hw.readRecordCount();
        try {
            return IntStream.range(0, nrec).mapToObj(this::fetchRecord);
        }
        catch (WrapperException exc) {throw (DriverException)exc.getCause();}
    }

    // Lambda expressions can't throw checked exceptions, so we'll have to
    // wrap them in this unchecked exception.
    private static class WrapperException extends RuntimeException {
        WrapperException(DriverException exc) {super(exc);}
    }

    private DataRecord fetchRecord(int irec) {
        try {
            // Tell the instrument which record we want.
            hw.writeRecordIndex(irec);
            // Get the record data that are not from a data channel.
            final Instant startTime = hw.readRecordStartTime();
            final Duration duration = hw.readRecordDuration();
            final String location = hw.readRecordLocation();
            final Set<RecordFlag> flags = hw.readRecordFlags();
            // Get the channel data.
            final List<ChannelDatum> channels =
                Stream.of(DataChannel.values())
                .map(this::fetchChannel)
                .collect(Collectors.toList());
            return new DataRecord(startTime, duration, location, flags, channels);
        }
        catch (DriverException exc) {throw new WrapperException(exc);}
     }

    private ChannelDatum fetchChannel(DataChannel chan) {
        try {
            hw.readChannelType(chan);  // Verify the type of channel (particle, temperature, etc.)
            hw.readChannelUnits(chan); // Verify the units of measurement.
            return new ChannelDatum(
                chan,
                hw.readChannelValue(chan)
            );
        }
        catch (DriverException exc) {throw new WrapperException(exc);}
    }

    /**
     * Delete all records currently stored in the instrument.
     * @throws DriverException if communication problems arise or if data collection
     * hasn't been stopped.
     */
    public void clearRecords() throws DriverException {
        if (hw.readDeviceFlags().contains(DeviceFlag.RUNNING)) {
            throw new DriverException("Can't clear records when the device is running.");
        }
        hw.writeDeviceCommand(DeviceCommand.CLEAR_DATA_BUFFER);
    }
}
